//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		CvImage.cpp
 * @brief		OpenCV Image t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_CvImage_CPP_

//======================================================================
// include
#include "CvImage.h"
#include "iris_debug.h"

#ifdef _IRIS_SUPPORT_OPENCV

namespace iris {
namespace gx {
namespace ocv
{

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CCvImage::CCvImage(void)
{
}

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CCvImage::CCvImage(const cv::Mat& mat)
{
	Clone(mat);
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CCvImage::~CCvImage(void)
{
}

/**********************************************************************//**
 *
 * C[W̍쐬
 *
 -----------------------------------------------------------------------
 * @param [in]	size		= TCY
 * @param [in]	depth		= [x
 * @param [in]	channels	= `l
 * @return	
*//***********************************************************************/
bool CCvImage::Create(const CvSize& size, int depth, int channels)
{
	Release();
	IplImage* pImage = cvCreateImage(size, depth, channels);
	if( pImage == nullptr ) return false;
	Attach(pImage);
	return true;
}

/**********************************************************************//**
 *
 * }gbNXC[W̍쐬
 *
 -----------------------------------------------------------------------
 * @param [in]	mat		= }gbNX
 * @return	
*//***********************************************************************/
bool CCvImage::CreateFromMatrix(const cv::Mat& mat)
{
	Release();
	IplImage* pImage = cvCreateImage(mat.size(), GetIplDepth(mat.depth()), mat.channels());
	if( pImage == nullptr ) return false;
	Attach(pImage);
	return true;
}

/**********************************************************************//**
 *
 * C[W̃[h
 *
 -----------------------------------------------------------------------
 * @param [in]	lpszPath	= pX
 * @param [in]	flags		= tO
 * @return	
*//***********************************************************************/
bool CCvImage::Load(LPCSTR lpszPath, int flags)
{
	Release();
	IplImage* pImage = cvLoadImage(lpszPath, flags);
	if( pImage == nullptr )
	{
		return false;
	}
	Attach(pImage);
	return true;
}

/**********************************************************************//**
 *
 * C[W̕ۑ
 *
 -----------------------------------------------------------------------
 * @param [in]	lpszPath	= pX
 * @param [in]	params		= p[^[
 * @return	
*//***********************************************************************/
bool CCvImage::Save(LPCSTR lpszPath, const int* params) const
{
	CvMat mat =m_Mat;
	return IRIS_TO_bool(cvSaveImage(lpszPath, &mat, params));
}

/**********************************************************************//**
 *
 * N[
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
void CCvImage::Clone(const cv::Mat& src)
{
	Release();
#if 0
	IplImage image = src;
	IplImage* pImage = cvCloneImage(&image);
	Attach(pImage);
#else
	m_Mat = src.clone();
#endif
}

/**********************************************************************//**
 *
 * LȃC[Wǂ
 *
 -----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
bool CCvImage::IsValid(void) const
{
	return !m_Mat.empty();
}

/**********************************************************************//**
 *
 * C[WTCY̎擾
 *
 -----------------------------------------------------------------------
 * @return	C[WTCY
*//***********************************************************************/
CvSize CCvImage::GetSize(void) const
{
	return m_Mat.size();
}

/**********************************************************************//**
 *
 * C[W̎擾
 *
 -----------------------------------------------------------------------
 * @return	C[W
*//***********************************************************************/
s32 CCvImage::GetWidth(void) const
{
	return m_Mat.size().width;
}

/**********************************************************************//**
 *
 * C[W̎擾
 *
 -----------------------------------------------------------------------
 * @return	C[W
*//***********************************************************************/
s32 CCvImage::GetHeight(void) const
{
	return m_Mat.size().height;
}

/**********************************************************************//**
 *
 * [xtO̎擾
 *
 -----------------------------------------------------------------------
 * @return	[xtO
*//***********************************************************************/
int CCvImage::GetDepthFlag(void) const
{
	return CV_MAT_DEPTH(m_Mat.depth());
}

/**********************************************************************//**
 *
 * [x̎擾
 *
 -----------------------------------------------------------------------
 * @return	[x
*//***********************************************************************/
int CCvImage::GetDepth(void) const
{
	return GetIplDepth(m_Mat.depth());
}

/**********************************************************************//**
 *
 * `l̎擾
 *
 -----------------------------------------------------------------------
 * @return	`l
*//***********************************************************************/
int CCvImage::GetChannels(void) const
{
	return m_Mat.channels();
}

/**********************************************************************//**
 *
 * ㏑Rs[
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= Rs[
 * @param [in]	mask	= }XN
*//***********************************************************************/
void CCvImage::Copy(const cv::Mat& src, const cv::Mat& mask)
{
	CvMat smat = src;
	CvMat dmat = m_Mat;
	CvMat mmat = mask;
	cvCopy(&smat, &dmat, &mmat);
}

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @param [out]	dst0	= o̓`l0
 * @param [out]	dst1	= o̓`l1
 * @param [out]	dst2	= o̓`l2
 * @param [out]	dst3	= o̓`l3
*//***********************************************************************/
void CCvImage::Split(CvArr* dst0, CvArr* dst1, CvArr* dst2, CvArr* dst3) const
{
	CvMat mat = m_Mat;
	cvSplit(&mat, dst0, dst1, dst2, dst3);
}

/**********************************************************************//**
 *
 * 臒l
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= Rs[
 * @param [in]	threshold		= 臒l
 * @param [in]	max_value		= ől
 * @param [in]	threshold_type	= @
*//***********************************************************************/
void CCvImage::Threshold(const cv::Mat& src, double threshold, double max_value, int threshold_type)
{
	CvMat smat = src;
	CvMat dmat = m_Mat;
	cvThreshold(&smat, &dmat, threshold, max_value, threshold_type);
}

/**********************************************************************//**
 *
 * J[tH[}bg̕ϊ
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= 
 * @param [in]	code		= tO
 * @param [in]	dstChannels	= o̓`l(0 w肷ƎvZ)
*//***********************************************************************/
void CCvImage::TranslateColor(const cv::Mat& src, int code, int dstChannels)
{
	::cv::cvtColor(src, *this, code, dstChannels);
}

/**********************************************************************//**
 *
 * ϊ
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= 
*//***********************************************************************/
void CCvImage::Convert(const cv::Mat& src)
{
	ConvertScale(src, 1.0);
}

/**********************************************************************//**
 *
 * `ϊ
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= 
 * @param [in]	scale		= XP[
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::ConvertScale(const cv::Mat& src, double scale, double shift)
{
	CvMat smat = src;
	CvMat dmat = m_Mat;
	cvConvertScale(&smat, &dmat, scale, shift);
}

/**********************************************************************//**
 *
 * `ϊ(8rbg^̔zɕϊ)
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= 
 * @param [in]	scale		= XP[
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::ConvertScaleAbs(const cv::Mat& src, double scale, double shift)
{
	CvMat smat = src;
	CvMat dmat = m_Mat;
	cvConvertScaleAbs(&smat, &dmat, scale, shift);
}

/**********************************************************************//**
 *
 * XP[
 *
 -----------------------------------------------------------------------
 * @param [in]	scale		= XP[
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::Scale(double scale, double shift)
{
	ConvertScale(*this, scale, shift);
}

/**********************************************************************//**
 *
 * TCYύX
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	fx				= 
 * @param [in]	fy				= 
 * @param [in]	interpolation	= ϊ(CV_INTER_***)
*//***********************************************************************/
void CCvImage::Resize(const cv::Mat& src, double fx, double fy, int interpolation)
{
	::cv::resize(src, m_Mat, src.size(), fx, fy, interpolation);
}

/**********************************************************************//**
 *
 * TCYύX
 *
 -----------------------------------------------------------------------
 * @param [in]	width			= ϊ̕
 * @param [in]	height			= ϊ̍
 * @param [in]	fx				=
 * @param [in]	fy				=
 * @param [in]	interpolation	= ϊ(CV_INTER_***)
*//***********************************************************************/
void CCvImage::Resize(int width, int height, double fx, double fy, int interpolation)
{
	CCvImage temp = *this;
	::cv::resize(temp, m_Mat, cvSize(width, height), fx, fy, interpolation);
}

/**********************************************************************//**
 *
 * AtBϊ
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	matrix			= AtBs
 * @param [in]	flags			= tO
 * @param [in]	borderMode		= 
 * @param [in]	fillval			= 
*//***********************************************************************/
void CCvImage::WarpAffine(const cv::Mat& src, const cv::Mat& matrix, int flags, int borderMode, CvScalar fillval)
{
	if( !IsValid() ) CreateFromMatrix(src);
	cv::warpAffine(src, m_Mat, matrix, src.size(), flags, borderMode, fillval);
}

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	smoothtype		= ^Cv
 * @param [in]	size1			= 
 * @param [in]	size2			= 
 * @param [in]	sigma1			= 
 * @param [in]	sigma2			= 
*//***********************************************************************/
void CCvImage::Smooth(const cv::Mat& src, int smoothtype, int size1, int size2, double sigma1, double sigma2)
{
	if( !IsValid() ) CreateFromMatrix(src);
	CvMat smat = src;
	CvMat dmat = m_Mat;
	cvSmooth(&smat, &dmat, smoothtype, size1, size2, sigma1, sigma2);
}

/**********************************************************************//**
 *
 * (CV_BLUR_NO_SCALE)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	size1			= 
 * @param [in]	size2			= 
*//***********************************************************************/
void CCvImage::SmoothBlurNoScale(const cv::Mat& src, int size1, int size2)
{
	Smooth(src, CV_BLUR_NO_SCALE, size1, size2);
}

/**********************************************************************//**
 *
 * (CV_BLUR)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	size1			= 
 * @param [in]	size2			= 
*//***********************************************************************/
void CCvImage::SmoothBlur(const cv::Mat& src, int size1, int size2)
{
	Smooth(src, CV_BLUR, size1, size2);
}

/**********************************************************************//**
 *
 * (CV_MEDIAN)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	size			= 
*//***********************************************************************/
void CCvImage::SmoothMedian(const cv::Mat& src, int size)
{
	cv::medianBlur(src, m_Mat, size);
}

/**********************************************************************//**
 *
 * (CV_BILATERAL)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	d				= 
 * @param [in]	sigmaColor		= 
 * @param [in]	sigmaSpace		= 
 * @param [in]	borderType		= 
*//***********************************************************************/
void CCvImage::SmoothBilateral(const cv::Mat& src, int d, double sigmaColor, double sigmaSpace, int borderType)
{
	cv::bilateralFilter(src, m_Mat, d, sigmaColor, sigmaSpace, borderType);
}

/**********************************************************************//**
 *
 * (CV_BILATERAL)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	ksize			= 
 * @param [in]	sigma1			= 
 * @param [in]	sigma2			= 
 * @param [in]	borderType		= 
 *//***********************************************************************/
void CCvImage::SmoothGaussian(const cv::Mat& src, const CvSize& ksize, double sigma1/* =0 */, double sigma2/* =0 */, int borderType)
{
	cv::GaussianBlur(src, m_Mat, ksize, sigma1, sigma2, borderType);
}

void CCvImage::SmoothGaussian(const cv::Mat& src, int ksize, double sigma1/* =0 */, double sigma2/* =0 */, int borderType)
{
	SmoothGaussian(src, cvSize(ksize, ksize), sigma1, sigma2, borderType);
}

/**********************************************************************//**
 *
 * tB^[
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	kernel			= 
 * @param [in]	anchor			= 
*//***********************************************************************/
void CCvImage::Filter2D(const cv::Mat& src, const cv::Mat& kernel, const CvPoint& anchor)
{
	if( !IsValid() ) CreateFromMatrix(src);
	cv::filter2D(src, m_Mat, GetDepth(), kernel, anchor);
}

/**********************************************************************//**
 *
 * tHW[ϊ
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	operation		= (CV_MOP_***)
 * @param [in]	kernel			= 
 * @param [in]	anchor			= 
 * @param [in]	iterations		= 
 * @param [in]	borderType		= 
 * @param [in]	bordefValue		= 
*//***********************************************************************/
void CCvImage::MorphologyEx(const cv::Mat& src, int operation, const cv::Mat& kernel, const cv::Point& anchor, int iterations
							, int borderType, const cv::Scalar& bordefValue)
{
	if( !IsValid() ) CreateFromMatrix(src);
	::cv::morphologyEx(src, m_Mat, operation, kernel, anchor, iterations, borderType, bordefValue);
}

void CCvImage::MorphologyEx(const cv::Mat& src, int operation, const CCvConvKernel& element, int iterations
							, int borderType, const cv::Scalar& bordefValue)
{
	cv::Mat kernel;
	cv::Point anchor;
	element.Convert(kernel, anchor);
	MorphologyEx(src, operation, kernel, anchor, iterations, borderType, bordefValue);
}

/**********************************************************************//**
 *
 * tHW[ϊ(Dilate)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	operation		= (CV_MOP_***)
 * @param [in]	kernel			= 
 * @param [in]	anchor			= 
 * @param [in]	iterations		= 
 * @param [in]	borderType		= 
 * @param [in]	bordefValue		= 
*//***********************************************************************/
void CCvImage::MorphologyDilate(const cv::Mat& src, const cv::Mat& kernel, const cv::Point& anchor, int iterations
								, int borderType, const cv::Scalar& bordefValue)
{
	if( !IsValid() ) CreateFromMatrix(src);
	::cv::dilate(src, m_Mat, kernel, anchor, iterations, borderType, bordefValue);
}

void CCvImage::MorphologyDilate(const cv::Mat& src, const CCvConvKernel& element, int iterations
								, int borderType, const cv::Scalar& bordefValue)
{
	cv::Mat kernel;
	cv::Point anchor;
	element.Convert(kernel, anchor);
	MorphologyDilate(src, kernel, anchor, iterations, borderType, bordefValue);
}

/**********************************************************************//**
 *
 * tHW[ϊ(Erode)
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	operation		= (CV_MOP_***)
 * @param [in]	kernel			= 
 * @param [in]	anchor			= 
 * @param [in]	iterations		= 
 * @param [in]	borderType		= 
 * @param [in]	bordefValue		= 
 *//***********************************************************************/
void CCvImage::MorphologyErode(const cv::Mat& src, const cv::Mat& kernel, const cv::Point& anchor, int iterations
								, int borderType, const cv::Scalar& bordefValue)
{
	if( !IsValid() ) CreateFromMatrix(src);
	::cv::erode(src, m_Mat, kernel, anchor, iterations, borderType, bordefValue);
}

void CCvImage::MorphologyErode(const cv::Mat& src, const CCvConvKernel& element, int iterations
								, int borderType, const cv::Scalar& bordefValue)
{
	cv::Mat kernel;
	cv::Point anchor;
	element.Convert(kernel, anchor);
	MorphologyErode(src, kernel, anchor, iterations, borderType, bordefValue);
}

/**********************************************************************//**
 *
 * 𑜓xAbv
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= 
 * @param [in]	dstsize	= o͉摜TCY
*//***********************************************************************/
void CCvImage::PyrUp(const cv::Mat& src, const cv::Size& dstsize)
{
	::cv::pyrUp(src, m_Mat, dstsize);
}

/**********************************************************************//**
 *
 * 𑜓x_E
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= 
 * @param [in]	dstsize	= o͉摜TCY
*//***********************************************************************/
void CCvImage::PyrDown(const cv::Mat& src, const cv::Size& dstsize)
{
	::cv::pyrDown(src, m_Mat, dstsize);
}

/**********************************************************************//**
 *
 * ̈敪
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= 
 * @param [in]	storage		= 
 * @param [in]	comp		= 
 * @param [in]	level		= 
 * @param [in]	threshold1	= 臒l
 * @param [in]	threshold2	= 臒l
*//***********************************************************************/
void CCvImage::PyrSegmentation(const cv::Mat& src, CvMemStorage* storage, CvSeq** comp, int level, double threshold1, double threshold2)
{
	if( !IsValid() ) CreateFromMatrix(src);
	IplImage smat = src;
	IplImage dmat = m_Mat;
	cvPyrSegmentation(&smat, &dmat, storage, comp, level, threshold1, threshold2);
}

/**********************************************************************//**
 *
 * ϒlVtgɂ摜̃ZOg
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= 
 * @param [in]	sp			= 
 * @param [in]	sr			= 
 * @param [in]	max_level	= 
 * @param [in]	termcrit	= 
*//***********************************************************************/
void CCvImage::PyrMeanShiftFiltering(const cv::Mat& src, double sp, double sr, int max_level, CvTermCriteria termcrit)
{
	if( !IsValid() ) CreateFromMatrix(src);
	CvMat smat = src;
	CvMat dmat = m_Mat;
	cvPyrMeanShiftFiltering(&smat, &dmat, sp, sr, max_level, termcrit);
}

/**********************************************************************//**
 *
 *
 *
 -----------------------------------------------------------------------
 * @param [in]	src				= 
 * @param [in]	corners			= 
 * @param [out]	corner_count	= 
 * @param [in]	quality_level	= 
 * @param [in]	min_distance	= 
 * @param [in]	mask			= 
 * @param [in]	block_size		= 
 * @param [in]	use_harris		= 
 * @param [in]	k				= 
*//***********************************************************************/
void CCvImage::GoodFeaturesToTrack(const cv::Mat& src, CvPoint2D32f* corners, int* corner_count
								   , double quality_level, double min_distance
								   , const CvArr* mask, int block_size, int use_harris, double k)
{
	CvMat smat = src;
	CvMat dmat = m_Mat;
	CCvImage temp;
	temp.Create(src.size(), IPL_DEPTH_32F, 1);
	CvMat tmat = temp;
	cvGoodFeaturesToTrack(&smat, &dmat, &tmat, corners, corner_count, quality_level, min_distance
		, mask, block_size, use_harris, k);
}

/**********************************************************************//**
 *
 * }b`O
 *
 -----------------------------------------------------------------------
 * @param [in]	src		= 
 * @param [in]	templ	= C[W
 * @param [in]	method	= T@(CV_TM_***)
*//***********************************************************************/
void CCvImage::MatchTemplate(const cv::Mat& src, const cv::Mat& templ, int method)
{
	if( !IsValid() )
	{
		Create(cvSize(src.cols - templ.cols + 1, src.rows - templ.rows + 1), IPL_DEPTH_32F, 1);
	}
	cv::matchTemplate(src, templ, m_Mat, method);
}

/**********************************************************************//**
 *
 * }b`O
 *
 -----------------------------------------------------------------------
 * @param [in]	templ	= C[W
 * @param [in]	method	= T@(CV_TM_***)
 * @return }b`̈
*//***********************************************************************/
CvRect CCvImage::MatchTemplate(const cv::Mat& templ, int method)
{
	CCvImage dst;
	dst.Create(cvSize(m_Mat.cols - templ.cols + 1, m_Mat.rows - templ.rows + 1), IPL_DEPTH_32F, 1);
	dst.MatchTemplate(m_Mat, templ, method);

	CvPoint loc;
	switch(method)
	{
	case CV_TM_SQDIFF:
	case CV_TM_SQDIFF_NORMED:
		loc = dst.GetMinLoc();
		break;
	default:
		loc = dst.GetMaxLoc();
		break;
	}
	return cvRect(loc.x, loc.y, loc.x + templ.cols, loc.y + templ.rows);
}

/**********************************************************************//**
 *
 * `}b`O
 *
 -----------------------------------------------------------------------
 * @param [in]	src			= C[W
 * @param [in]	method		= T@
 * @param [in]	parameter	= p[^(󖢎gp)
 * @return vx(0 ɋ߂قǈv)
*//***********************************************************************/
double CCvImage::MatchShape(const cv::Mat& src, int method, double parameter)
{
	return ::cv::matchShapes(m_Mat, src, method, parameter);
}

/**********************************************************************//**
 *
 * ROI ̃Zbg
 *
 -----------------------------------------------------------------------
 * @param [in]	rc	= ROI ͈
*//***********************************************************************/
cv::Mat CCvImage::SetROI(const CvRect& rc)
{
#if 0
	CvMat mat = m_Mat;
	IplImage head, *image;
	image = cvGetImage(&mat, &head);
	cvSetImageROI(image, rc);
#else
	//m_Mat.adjustROI(rc.y, rc.y + rc.height, rc.x, rc.x + rc.width);
#endif
	return m_Mat(rc);
}

/**********************************************************************//**
 *
 * ROI ̎擾
 *
 -----------------------------------------------------------------------
 * @return ROI ͈
*//***********************************************************************/
CvRect CCvImage::GetROI(void) const
{
#if 0
	CvMat mat = m_Mat;
	IplImage head, *image;
	image = cvGetImage(&mat, &head);
	return cvGetImageROI(image);
#else
	cv::Size size;
	cv::Point ofs;
	m_Mat.locateROI(size, ofs);
	return cvRect(ofs.x, ofs.y, size.width, size.height);
#endif
}

/**********************************************************************//**
 *
 * ROI ̃Zbg
 *
*//***********************************************************************/
void CCvImage::ResetROI(void)
{
	CvMat mat = m_Mat;
	IplImage head, *image;
	image = cvGetImage(&mat, &head);
	cvResetImageROI(image);
}

/**********************************************************************//**
 *
 * COI ̃Zbg
 *
 -----------------------------------------------------------------------
 * @param [in]	coi	= COI
*//***********************************************************************/
void CCvImage::SetCOI(int coi)
{
	CvMat mat = m_Mat;
	IplImage head, *image;
	image = cvGetImage(&mat, &head);
	cvSetImageCOI(image, coi);
}

/**********************************************************************//**
 *
 * COI ̎擾
 *
 -----------------------------------------------------------------------
 * @return COI
*//***********************************************************************/
int CCvImage::GetCOI(void) const
{
	CvMat mat = m_Mat;
	IplImage head, *image;
	image = cvGetImage(&mat, &head);
	return cvGetImageCOI(image);
}

/**********************************************************************//**
 *
 *
 *
*//***********************************************************************/
void CCvImage::MinMaxLoc(double* min_val, double* max_val, CvPoint* min_loc, CvPoint* max_loc, const CvArr* mask) const
{
	::cv::minMaxLoc(m_Mat, min_val, max_val, (cv::Point*)min_loc, (cv::Point*)max_loc, ::cv::cvarrToMat(mask));
}

/**********************************************************************//**
 *
 *
 *
 -----------------------------------------------------------------------
 * @return 
*//***********************************************************************/
CvPoint CCvImage::GetMinLoc(const CvArr* mask) const
{
	double min_val, max_val;
	CvPoint min_loc;
	MinMaxLoc(&min_val, &max_val, &min_loc, nullptr, mask);
	return min_loc;
}

/**********************************************************************//**
 *
 *
 *
 -----------------------------------------------------------------------
 * @return 
*//***********************************************************************/
CvPoint CCvImage::GetMaxLoc(const CvArr* mask) const
{
	double min_val, max_val;
	CvPoint max_loc;
	MinMaxLoc(&min_val, &max_val, nullptr, &max_loc, mask);
	return max_loc;
}

/**********************************************************************//**
 *
 * C擾
 *
 -----------------------------------------------------------------------
 * @param [in]	pt1				= _1
 * @param [in]	pt2				= _2
 * @param [in]	connectivity	= 4 or 8
 * @param [in]	left_to_right	= ɍEɎ邩
 * @return	CCe[^
*//***********************************************************************/
CCvImage::line::iterator CCvImage::Line(const CvPoint& pt1, const CvPoint& pt2, int connectivity, int left_to_right)
{
	CvLineIterator it;
	CvMat mat = m_Mat;
	cvInitLineIterator(&mat, pt1, pt2, &it, connectivity, left_to_right);
	return &it;
}

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @param [in]	pts			= _z
 * @param [in]	npts		= _z
 * @param [in]	contours	= 
 * @param [in]	is_closed	= 
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::DrawPolyLine(CvPoint** pts, const int* npts, int contours, int is_closed, const CvScalar& color
		, int thickness, int line_type, int shift)
{
	CvMat mat = m_Mat;
	cvPolyLine(&mat, pts, npts, contours, is_closed, color, thickness, line_type, shift);
}

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @param [in]	pt1			= _1
 * @param [in]	pt2			= _2
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::DrawLine(const CvPoint& pt1, const CvPoint& pt2, const CvScalar& color
		, int thickness, int line_type, int shift)
{
	cv::line(m_Mat, pt1, pt2, color, thickness, line_type, shift);
}

/**********************************************************************//**
 *
 * `
 *
 -----------------------------------------------------------------------
 * @param [in]	pt1			= ̓_
 * @param [in]	pt2			= E̓_
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::DrawRectangle(const CvPoint& pt1, const CvPoint& pt2, const CvScalar& color
		, int thickness, int line_type, int shift)
{
	::cv::rectangle(m_Mat, pt1, pt2, color, thickness, line_type, shift);
}

/**********************************************************************//**
 *
 * `
 *
 -----------------------------------------------------------------------
 * @param [in]	rc			= ̈
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
 * @return	
*//***********************************************************************/
void CCvImage::DrawRectangle(const CvRect& rc, const CvScalar& color
		, int thickness, int line_type, int shift)
{
	::cv::rectangle(m_Mat, rc, color, thickness, line_type, shift);
}

/**********************************************************************//**
 *
 * ~`
 *
 -----------------------------------------------------------------------
 * @param [in]	center		= _
 * @param [in]	radius		= a
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::DrawCircle(const CvPoint& center, int radius, const CvScalar& color
		, int thickness, int line_type, int shift)
{
	::cv::circle(m_Mat, center, radius, color, thickness, line_type, shift);
}

/**********************************************************************//**
 *
 * ȉ~
 *
 -----------------------------------------------------------------------
 * @param [in]	center		= _
 * @param [in]	axes		= 
 * @param [in]	angle		= 
 * @param [in]	start_angle	= 
 * @param [in]	end_angle	= 
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::DrawEllipse(const CvPoint& center, const CvSize& axes
		, double angle, double start_angle, double end_angle, const CvScalar& color
		, int thickness, int line_type, int shift)
{
	::cv::ellipse(m_Mat, center, axes, angle, start_angle, end_angle, color, thickness, line_type, shift);
}

/**********************************************************************//**
 *
 * ȉ~
 *
 -----------------------------------------------------------------------
 * @param [in]	box			= 
 * @param [in]	color		= F
 * @param [in]	thickness	= 
 * @param [in]	line_type	= C^Cv
*//***********************************************************************/
void CCvImage::DrawEllipse(const CvBox2D& box, const CvScalar& color
		, int thickness, int line_type)
{
	::cv::ellipse(m_Mat, box, color, thickness, line_type);
}

/**********************************************************************//**
 *
 * ֊s`
 *
 -----------------------------------------------------------------------
 * @param [in]	contour			= 
 * @param [in]	external_color	= O̐F
 * @param [in]	hole_color		= ̐F
 * @param [in]	thickness		= 
 * @param [in]	line_type		= C^Cv
 * @param [in]	offset			= 
*//***********************************************************************/
void CCvImage::DrawContours(CvSeq* contour, const CvScalar& external_color, const CvScalar& hole_color, int max_level
		, int thickness, int line_type, const CvPoint& offset)
{
	CvMat mat = m_Mat;
	cvDrawContours(&mat, contour, external_color, hole_color, max_level, thickness, line_type, offset);
}

/**********************************************************************//**
 *
 * eLXg`
 *
 -----------------------------------------------------------------------
 * @param [in]	text	= eLXg
 * @param [in]	org		= `N_
 * @param [in]	font	= tHg
 * @param [in]	color	= F
*//***********************************************************************/
void CCvImage::DrawText(LPCSTR text, const CvPoint& org, const CvFont* font, CvScalar color/* =CV_RGB */)
{
	::cv::putText(m_Mat, text, org, font->font_face, (font->vscale + font->hscale)*0.5, color, font->thickness, font->line_type);
	//CvMat mat = m_Mat;
	//cvPutText(&mat, text, org, font, color);
}

/**********************************************************************//**
 *
 *
 *
 -----------------------------------------------------------------------
 * @param [in]	pts			= _z
 * @param [in]	npts		= _z
 * @param [in]	color		= F
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::FillConvexPoly(const ::cv::Point* pts, int npts, const CvScalar& color
		, int line_type, int shift)
{
	::cv::fillConvexPoly(m_Mat, pts, npts, color, line_type, shift);
}

/**********************************************************************//**
 *
 * p`hԂ
 *
 -----------------------------------------------------------------------
 * @param [in]	pts			= _z
 * @param [in]	npts		= _z
 * @param [in]	contours	= 
 * @param [in]	color		= F
 * @param [in]	line_type	= C^Cv
 * @param [in]	shift		= 
*//***********************************************************************/
void CCvImage::FillPoly(const ::cv::Point** pts, const int* npts, int contours, const CvScalar& color
						, int line_type, int shift, ::cv::Point offset)
{
	::cv::fillPoly(m_Mat, pts, npts, contours, color, line_type, shift, offset);
}

/**********************************************************************//**
 *
 * sNZJ[̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	x	= xW
 * @param [in]	y	= yW
 * @return	sNZRGBA
*//***********************************************************************/
bool CCvImage::GetPixelRGBA8888(s32 x, s32 y, IrisRGBA8888& rgba) const
{
	const u8* addr = m_Mat.ptr() + GetWidth() * y + x * GetChannels();
	if( GetDepth() != 8 ) return false;

	rgba.a = 0xFF;
	switch( GetChannels() )
	{
	case 1:
		rgba.b = rgba.g = rgba.r = *addr;
		break;
	case 3:
		{
			const IrisBGR888* tmp = pointer_cast<const IrisBGR888*>(addr);
			rgba.b = tmp->b;
			rgba.g = tmp->g;
			rgba.r = tmp->r;
		}
		break;
	default:
		return false;
	}
	return true;
}

/**********************************************************************//**
 *
 * sNZJ[̐ݒ
 *
 -----------------------------------------------------------------------
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @param [in]	rgba	= sNZRGBA
*//***********************************************************************/
bool CCvImage::SetPixelRGBA8888(s32 x, s32 y, const IrisRGBA8888& rgba)
{
	u8* addr = m_Mat.ptr() + GetWidth() * y + x * GetChannels();
	if( GetDepth() != 8 ) return false;
	switch( GetChannels() )
	{
	//case 1:
	//	break;
	case 3:
		{
			IrisBGR888* tmp = pointer_cast<IrisBGR888*>(addr);
			tmp->b = rgba.b;
			tmp->g = rgba.g;
			tmp->r = rgba.r;
		}
		break;
	default:
		return false;
	}
	return true;
}

/**********************************************************************//**
 *
 * IPL_DEPTH_*** ̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	cv_depth	= 
*//***********************************************************************/
int CCvImage::GetIplDepth(int cv_depth)
{
	int depth = CV_MAT_DEPTH(cv_depth);
	switch(depth)
	{
	case CV_1F:
		return IPL_DEPTH_1U;
	case CV_8S:
		return IPL_DEPTH_8S;
	case CV_8U:
		return IPL_DEPTH_8U;
	case CV_16S:
		return IPL_DEPTH_16S;
	case CV_16U:
		return IPL_DEPTH_16U;
	case CV_32S:
		return IPL_DEPTH_32S;
	case CV_32F:
		return IPL_DEPTH_32F;
	case CV_64F:
		return IPL_DEPTH_64F;
	}
	return 0;
}

/**********************************************************************//**
 *
 * A^b`
 *
 -----------------------------------------------------------------------
 * @param [in]	pImage		= o^C[W
*//***********************************************************************/
void CCvImage::Attach(IplImage* pImage)
{
	IRIS_ASSERT( pImage != nullptr );
	//::cv::Mat mat = pImage;
	m_Mat = pImage;
	//cvReleaseImage(&pImage);
}

}	// end of namespace ocv
}	// end of namespace gx
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../../../unit/UnitCore.h"
#include "../gui/CvWindow.h"
#include "../math/CvAffineMatrix.h"
#include "../math/CvShape.h"
#include "CvConvKernel.h"
#include "CvFont.h"
#include "../../../iris_using.h"
#include "../../../iris_iostream.h"
#include <stdio.h>
#include <tchar.h>

//======================================================================
// test
IRIS_UNITTEST(CCvImageUnitTest, Func)
{
	CCvImage image, gray;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path) ) return;
	window.Create("Show Image");
	gray.TranslateColor(image, CV_BGR2GRAY);
	window.ShowImage(gray);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, Resize)
{
	CCvImage image;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path) ) return;
	window.Create("Show Image");
	image.Resize(image.GetWidth()*2, image.GetHeight()*2);
	window.ShowImage(image);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, Smooth)
{
	CCvImage image;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;

	CCvImage blur, nblur, median, gaussian, bilateral;
	nblur.SmoothBlurNoScale(image);
	blur.SmoothBlur(image);
	median.SmoothMedian(image);
	gaussian.SmoothGaussian(image);

	CCvWindow window[5];
	window[0].Create("BLUR NO SCALE");
	window[0].ShowImage(nblur);
	window[1].Create("BLUR");
	window[1].ShowImage(blur);
	window[2].Create("MEDIAN");
	window[2].ShowImage(median);
	window[3].Create("GAUSSIAN");
	window[3].ShowImage(gaussian);

	//bilateral.SmoothBilateral(image);
	//window[4].Create("BILATERAL");
	//window[4].ShowImage(bilateral);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, Filter2D)
{
	CCvImage image, after;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;
	window.Create("Show Image");
	CCvMatrix kernel;
	float data[] = { 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1  };
	kernel.Create(1, 21, CV_32F, data);
	kernel.Normalize();
	after.Filter2D(image, kernel, cvPoint(0, 0));
	window.ShowImage(after);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, RectSubPix)
{
	CCvImage image, out;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path) ) return;
	window.Create("Show Image");
	out.GetRectSubPix(image, cvPoint2D32f(image.GetWidth()/3, image.GetHeight()/3));
	window.ShowImage(out);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, QuadrangleSubPix)
{
	CCvImage image, out;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path) ) return;
	window.Create("Show Image");
	CCvAffineMatrix affine(cvPoint2D32f(image.GetWidth()/2.0f, image.GetHeight()/2.0f), 45.0f);
	out.GetQuadrangleSubPix(image, affine);
	window.ShowImage(out);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, WarpAffin)
{
	CCvImage image, after;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;
	window.Create("Show Image");
	CCvAffineMatrix affine;
	CCvShape<3> src_pnt, dst_pnt;
	src_pnt[0] = cvPoint2D32f(200.0, 200.0);
	src_pnt[1] = cvPoint2D32f(250.0, 200.0);
	src_pnt[2] = cvPoint2D32f(200.0, 100.0);
	dst_pnt[0] = cvPoint2D32f(300.0, 100.0);
	dst_pnt[1] = cvPoint2D32f(300.0, 50.0);
	dst_pnt[2] = cvPoint2D32f(200.0, 100.0);
	affine.GetAffineTransform(src_pnt, dst_pnt);
	after.WarpAffine(image, affine);
	window.ShowImage(after);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, WarpAffinROI)
{
	CCvImage image, out;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;
	window.Create("Show Image");
	CvRect rc = cvRect(image.GetWidth()/4, image.GetHeight()/4, image.GetWidth()/2, image.GetHeight()/2);
	CCvImage roi = image.SetROI(rc);
	rc = image.GetROI();
	out.Clone(image);
	CCvAffineMatrix affine(cvPoint2D32f(image.GetWidth()/4.0f, image.GetHeight()/4.0f), -45.0, 1.0);
	out.WarpAffine(image, affine);
	image.ResetROI();
	out.ResetROI();
	out.DrawRectangle(rc, CV_RGB(255, 0, 0), 2);
	window.ShowImage(out);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, MorphologyEx)
{
	CCvImage image, tmp;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;
	tmp.Clone(image);

	CCvConvKernel element(9, 9, 4, 4, CV_SHAPE_RECT);
	CCvImage open, close, gradient, tophat, blackhat;
	open.MorphologyEx(image, CV_MOP_OPEN, element);
	close.MorphologyEx(image, CV_MOP_CLOSE, element);
	gradient.MorphologyEx(image, CV_MOP_GRADIENT, element);
	tophat.MorphologyEx(image, CV_MOP_TOPHAT, element);
	blackhat.MorphologyEx(image, CV_MOP_BLACKHAT, element);

	CCvWindow window[5];
	window[0].Create("CV_MOP_OPEN");
	window[0].ShowImage(open);
	window[1].Create("CV_MOP_CLOSE");
	window[1].ShowImage(close);
	window[2].Create("CV_MOP_GRADIENT");
	window[2].ShowImage(gradient);
	window[3].Create("CV_MOP_TOPHAT");
	window[3].ShowImage(tophat);
	window[4].Create("CV_MOP_BLACKHAT");
	window[4].ShowImage(blackhat);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, Pyr)
{
	CCvImage image;
	CCvWindow window1, window2;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;
	CCvImage up, down;
	up.Create(cvSize(image.GetWidth()*2, image.GetHeight()*2), image.GetDepth(), image.GetChannels());
	down.Create(cvSize(image.GetWidth()/2, image.GetHeight()/2), image.GetDepth(), image.GetChannels());
	up.PyrUp(image);
	down.PyrDown(image);
	window1.Create("PyrUp");
	window1.ShowImage(up);
	window2.Create("PyrDown");
	window2.ShowImage(down);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, PyrMeanShiftFiltering)
{
	CCvImage image;
	CCvWindow window;
	CHAR path[MAX_PATH];
	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);

	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;
	CCvImage out;
	out.PyrMeanShiftFiltering(image, 30.0, 30.0, 2);
	window.Create("Show Image");
	window.ShowImage(out);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, MatchTemplate)
{
	CCvImage image, templ;
	CCvWindow window;
	CHAR path[MAX_PATH];

	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	if( !image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;

	std::cout << "T摜t@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	if( !templ.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) ) return;

	CvRect rc = image.MatchTemplate(templ, CV_TM_CCOEFF_NORMED);
	image.DrawRectangle(rc, CV_RGB(255, 0, 0), 3);

	window.Create("Show Image");
	window.ShowImage(image);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, MatchShape)
{
	CCvImage image, gray, templ;
	CCvWindow window;
	CHAR path[MAX_PATH];

	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	if( !image.Load(path) ) return;
	gray.TranslateColor(image, CV_BGR2GRAY);

	std::cout << "rt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	if( !templ.Load(path, CV_LOAD_IMAGE_GRAYSCALE) ) return;

	double result = gray.MatchShape(templ, CV_CONTOURS_MATCH_I1);
	char text[16];
	CCvFont font(CV_FONT_HERSHEY_SIMPLEX, 1.0, 4);
    sprintf_s(text, 16, "%.5f", result);
	image.DrawText(text, cvPoint (10, image.GetHeight() - 10), font, CV_RGB(255, 0, 0));

	window.Create("Show Image");
	window.ShowImage(image);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, Threshold)
{
	CCvImage image;
	CCvWindow image_window;
	CCvWindow window[3];
	CHAR path[MAX_PATH];
	CHAR wname[3][16] = { {"red"}, {"green"}, {"blue"} };

	std::cout << "Jt@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	if( !image.Load(path) ) return;

	CCvImage ch[3];
	for( int i=0; i < 3; ++i )
		ch[i].Create(image.GetSize(), IPL_DEPTH_8U, 1);
	CvMat rmat = ch[0];
	CvMat gmat = ch[1];
	CvMat bmat = ch[2];
	image.Split(&bmat, &gmat, &rmat, nullptr);
	for( int i=0; i < 3; ++i )
	{
		//ch[i].SmoothGaussian(ch[i], 5);
		ch[i].Threshold(ch[i], 0, 255, CV_THRESH_BINARY_INV | CV_THRESH_OTSU);
		window[i].Create(wname[i]);
		window[i].ShowImage(ch[i]);
	}
	image_window.Create("Show Image");
	image_window.ShowImage(image);
	cvWaitKey();
}

IRIS_UNITTEST(CCvImageUnitTest, CopyRedMask)
{
	CCvImage image, templ;
	CCvWindow window;
	CHAR path[MAX_PATH];

	std::cout << "Rs[t@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	ASSERT_TRUE( image.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) );

	std::cout << "Rs[t@C͂ĂB" << std::endl;
	std::cin.getline(path, MAX_PATH);
	ASSERT_TRUE( templ.Load(path, CV_LOAD_IMAGE_ANYDEPTH | CV_LOAD_IMAGE_ANYCOLOR) );

	CCvImage red;
	red.Create(templ.GetSize(), IPL_DEPTH_8U, 1);
	CvMat red_mat = red;
	templ.Split(nullptr, nullptr, &red_mat, nullptr);
	red.SmoothGaussian(red, 5);
	red.Threshold(red, 128, 255, CV_THRESH_BINARY_INV | CV_THRESH_OTSU);

	templ.Copy(image, red);

	window.Create("Show Image");
	window.ShowImage(templ);
	cvWaitKey();
}

#endif	// #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#endif
