//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndYUV.h
 * @brief		YUV t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011-2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndYUV_H_
#define INCG_IRIS_FndYUV_H_

//======================================================================
// include
#include "FndColor.h"
#include "../../ml/iml_math.hpp"

namespace iris {
namespace fnd
{

//======================================================================
// struct
typedef struct tagYUV_PARAM
{
	struct {
		f32 r;
		f32 g;
		f32 b;
	} y, u, v;
	f32 rv;
	f32 gu, gv;
	f32 bu;
} YUV_PARAM;

struct ITU_R_BT601 : public YUV_PARAM
{
	ITU_R_BT601(void)
	{
		y.r = 0.29891f;
		y.g = 0.58661f;
		y.b = 0.11448f;
		u.r = -0.16784f;
		u.g = -0.33126f;
		u.b = 0.5f;
		v.r = 0.5f;
		v.g = -0.41869f;
		v.b = -0.08131f;

		rv = 1.402f;
		gu = -0.34414f;
		gv = -0.71414f;
		bu = 1.772f;
	}
};

struct ITU_R_BT709 : public YUV_PARAM
{
	ITU_R_BT709(void)
	{
		y.r = 0.2126f;
		y.g = 0.7152f;
		y.b = 0.0722f;
		u.r = -0.1146f;
		u.g = -0.3854f;
		u.b = 0.5f;
		v.r = 0.5f;
		v.g = -0.4542f;
		v.b = -0.0458f;

		rv = 1.5748f;
		gu = -0.1873f;
		gv = -0.4681f;
		bu = 1.8556f;
	}
};

//======================================================================
// declare
template<int YRANGE_, int CRANGE_>class CYPbPr;
template<int YRANGE_, int CRANGE_>class CYCbCr;

//======================================================================
// class
/**
 * @brief	YCbCr J[NX
 * @tparam	PARAM_		= ϊp[^
 * @tparam	YRANGE_		= Y ͈̔
 * @tparam	CRANGE_		= Cb,Cr ͈̔
*/
template<typename PARAM_, int YRANGE_=219, int CRANGE_=224>
class CYCbCrBase : public IIrisObject
{
public:
	static const int BITSIZE = 8;

	typedef typename int_least_type<BITSIZE>::UInt	value_type;	//!< YUV ě^
	typedef PARAM_	param_type;

private:
	static const value_type YRANGE	= YRANGE_;
	static const value_type CRANGE	= CRANGE_;
	static const value_type LOWER	= (256 - YRANGE) / 2;
	static const value_type BIAS	= 1u<<(BITSIZE-1);

private:
	value_type	m_y;
	value_type	m_u;
	value_type	m_v;

	static const param_type	s_param;

public:
	/**
	 * @brief	RXgN^
	 * @{
	*/
	CYCbCrBase(void) : m_y(0), m_u(0), m_v(0) {}
	CYCbCrBase(value_type y, value_type u, value_type v) : m_y(y), m_u(u), m_v(v) {}
	CYCbCrBase(const IrisFColor& fcol) : m_y(0), m_u(0), m_v(0) { cast_from_fcolor(fcol); }
	/**
	 * @}
	*/

public:
	/**
	 * @name	e̎擾
	 * @{
	*/
	value_type	y(void)	const	{ return m_y; }
	value_type	u(void)	const	{ return m_u; }
	value_type	v(void)	const	{ return m_v; }
	/**
	 * @}
	*/

	/**
	 * @name	e̐ݒ
	 * @{
	*/
	void		y(value_type v)	{ m_y = v; }
	void		u(value_type v)	{ m_u = v; }
	void		v(value_type v)	{ m_v = v; }
	/**
	 * @}
	*/

public:
	operator IrisFColor (void)	const	{ IrisFColor col; cast_to_fcolor(col); return col; }

protected:
	/**
	 * @ϊ
	 * @{
	*/
	void	cast_from_fcolor(const IrisFColor& col)
	{
		f32 y = col.r * s_param.y.r + col.g * s_param.y.g + col.b * s_param.y.b;
		f32 u = col.r * s_param.u.r + col.g * s_param.u.g + col.b * s_param.u.b;
		f32 v = col.r * s_param.v.r + col.g * s_param.v.g + col.b * s_param.v.b;
		m_y = static_cast<value_type>(y * YRANGE + LOWER);
		m_u = static_cast<value_type>(u * CRANGE + BIAS);
		m_v = static_cast<value_type>(v * CRANGE + BIAS);
	}
	void	cast_to_fcolor(IrisFColor& col) const
	{
		value_type y = (m_y - LOWER);
		int u = static_cast<int>(m_u) - BIAS;
		int v = static_cast<int>(m_v) - BIAS;
		col.r = (y + s_param.rv * v) / YRANGE;
		col.g = (y + s_param.gu * u + s_param.gv * v) / CRANGE;
		col.b = (y + s_param.bu * u) / CRANGE;
	}
	/**
	 * @}
	*/
};

template<typename PARAM_, int YRANGE_, int CRANGE_>
const PARAM_	CYCbCrBase<PARAM_, YRANGE_, CRANGE_>::s_param;

/**
 * @brief	YCbCr (SD)
*/
template<int YRANGE_=219, int CRANGE_=224>
class CYCbCr : public CYCbCrBase<ITU_R_BT601, YRANGE_, CRANGE_> 
{
public:
	/**
	 * @brief	RXgN^
	 * @{
	*/
	CYCbCr(void) {}
	CYCbCr(value_type y, value_type u, value_type v) : CYCbCrBase(y, u, v) {}
	CYCbCr(const IrisFColor& fcol) : CYCbCrBase(fcol) {}
	/**
	 * @}
	*/
};
	
/**
 * @brief	YCbCr (HD)
*/
template<int YRANGE_=219, int CRANGE_=224>
class CYPbPr : public CYCbCrBase<ITU_R_BT709, YRANGE_, CRANGE_>
{
public:
	/**
	 * @brief	RXgN^
	 * @{
	*/
	CYPbPr(void) {}
	CYPbPr(value_type y, value_type u, value_type v) : CYCbCrBase(y, u, v) {}
	CYPbPr(const IrisFColor& fcol) : CYCbCrBase(fcol) {}
	/**
	 * @}
	*/
};

}	// end of namespace fnd
}	// end of namespace iris

#endif
