//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndTga.cpp
 * @brief		tga t@CNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndTga_CPP_

//======================================================================
// include
#include "FndTga.h"
#include "../io/FndFile.h"
#include "../memory/FndMemory.h"
#include <string.h>
#include "../../iris_debug.h"

namespace iris {
namespace fnd
{

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CTga::CTga(void)
: m_pBuffer(nullptr)
, m_Size(0)
{
}

/**********************************************************************//**
 *
 * fXgN^
 *
*//***********************************************************************/
CTga::~CTga(void)
{
	Release();
}

/**********************************************************************//**
 *
 * t@Cǂݍ
 *
 ----------------------------------------------------------------------
 * @param [in]	lpFileName	= t@C
 * @return	
*//***********************************************************************/
template<>
bool CTga::ReadFile<CHAR>(LPCSTR lpFileName)
{
	return ReadFileA(lpFileName);
}
template<>
bool CTga::ReadFile<WCHAR>(LPCWSTR lpFileName)
{
	return ReadFileW(lpFileName);
}
/// CTga::ReadFile Q
bool CTga::ReadFileA(LPCSTR  lpFileName)
{
	if( lpFileName == nullptr ) return false;
	CFile file;
	if( !file.OpenA(lpFileName, FOPEN_READ) ) return false;
	Release();

	u32 size = file.GetSize();
	Alloc(size, 0);
	if( m_pBuffer == nullptr ) return false;
	file.Read(m_pBuffer, size);

	switch(GetImageType())
	{
	case TGA_IMAGETYPE_NONE:
	default:
		Release();
		return false;
	case TGA_IMAGETYPE_INDEX:
	case TGA_IMAGETYPE_INDEX_RLE:
	case TGA_IMAGETYPE_FULL:
	case TGA_IMAGETYPE_FULL_RLE:
	case TGA_IMAGETYPE_GRAY:
	case TGA_IMAGETYPE_GRAY_RLE:
		break;
	}
	return true;
}
/// CTga::ReadFile Q
bool CTga::ReadFileW(LPCWSTR lpFileName)
{
	if( lpFileName == nullptr ) return false;
	CFile file;
	if( !file.OpenW(lpFileName, FOPEN_READ) ) return false;
	Release();

	u32 size = file.GetSize();
	Alloc(size, 0);
	if( m_pBuffer == nullptr ) return false;
	file.Read(m_pBuffer, size);

	switch(GetImageType())
	{
	case TGA_IMAGETYPE_NONE:
	default:
		Release();
		return false;
	case TGA_IMAGETYPE_INDEX:
	case TGA_IMAGETYPE_INDEX_RLE:
	case TGA_IMAGETYPE_FULL:
	case TGA_IMAGETYPE_FULL_RLE:
	case TGA_IMAGETYPE_GRAY:
	case TGA_IMAGETYPE_GRAY_RLE:
		break;
	}
	return true;
}

/**********************************************************************//**
 *
 * t@Cɏ
 *
 ----------------------------------------------------------------------
 * @param [in]	lpFileName	= t@C
 * @return	
*//***********************************************************************/
template<>
bool CTga::WriteFile<CHAR>(LPCSTR lpFileName)
{
	return WriteFileA(lpFileName);
}
template<>
bool CTga::WriteFile<WCHAR>(LPCWSTR lpFileName)
{
	return WriteFileW(lpFileName);
}
/// CTga::WriteFile Q
bool CTga::WriteFileA(LPCSTR  lpFileName)
{
	if( lpFileName == nullptr ) return false;
	if( m_pBuffer == nullptr ) return false;
	CFile file;
	if( !file.OpenA(lpFileName, FOPEN_WRITE|FOPEN_CREATE) ) return false;

	file.Write(m_pBuffer, m_Size);
	return true;
}
/// CTga::WriteFile Q
bool CTga::WriteFileW(LPCWSTR lpFileName)
{
	if( lpFileName == nullptr ) return false;
	if( m_pBuffer == nullptr ) return false;
	CFile file;
	if( !file.OpenW(lpFileName, FOPEN_WRITE|FOPEN_CREATE) ) return false;

	file.Write(m_pBuffer, m_Size);
	return true;
}

/**********************************************************************//**
 *
 * 
 *
*//***********************************************************************/
void CTga::Release(void)
{
	Dealloc();
}

/**********************************************************************//**
 *
 * Rs[̍쐬
 *
 ----------------------------------------------------------------------
 * @param [in]	tga	=	Rs[tga
 * @return	
*//***********************************************************************/
bool CTga::Duplicate(CTga& tga)
{
	Release();

	u32 size = tga.GetBufferSize();
	if( size == 0 ) return false;
	Alloc(size, 0);
	if( m_pBuffer == nullptr ) return false;
	memcpy(m_pBuffer, tga.GetBuffer(), size);
	return true;
}

/**********************************************************************//**
 *
 * LȃC[Wǂ
 *
 ----------------------------------------------------------------------
 * @return	obt@AhX
*//***********************************************************************/
bool CTga::IsValid(void)	const
{
	return m_pBuffer != nullptr;
}

/**********************************************************************//**
 *
 * obt@̎擾
 *
 ----------------------------------------------------------------------
 * @return	obt@AhX
*//***********************************************************************/
const u8* CTga::GetBuffer(void)	const
{
	return m_pBuffer;
}

/**********************************************************************//**
 *
 * t@Cwb_̎擾
 *
 ----------------------------------------------------------------------
 * @return	t@Cwb_AhX
*//***********************************************************************/
LPCTGA_HEADER CTga::GetHeader(void)	const
{
	if( m_pBuffer == nullptr ) return nullptr;
	return pointer_cast<const TGA_HEADER*>(m_pBuffer);
}

/**********************************************************************//**
 *
 * pbgf[^̗L擾
 *
 ----------------------------------------------------------------------
 * @return	^Ul
*//***********************************************************************/
u8 CTga::IsHasPalette(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return phdr->ColorMapType;
}

/**********************************************************************//**
 *
 * pbgf[^̎擾
 *
 ----------------------------------------------------------------------
 * @return	pbgf[^AhX
*//***********************************************************************/
const u8* CTga::GetPalette(void)	const
{
	if( m_pBuffer == nullptr ) return nullptr;
	LPCTGA_HEADER phdr = GetHeader();
	u32 ofs = sizeof(TGA_HEADER) + phdr->IDLength;
	return m_pBuffer + ofs;
}

/**********************************************************************//**
 *
 * pbgf[^TCY̎擾
 *
 ----------------------------------------------------------------------
 * @return	pbgf[^TCY
*//***********************************************************************/
u32 CTga::GetPaletteSize(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return (u32)phdr->ColorMapNum * (phdr->ColorMapBit >> 3);
}

/**********************************************************************//**
 *
 * CfbNXf[^̎擾
 *
 ----------------------------------------------------------------------
 * @return	CfbNXf[^AhX
*//***********************************************************************/
const u8* CTga::GetIndex(void)	const
{
	if( m_pBuffer == nullptr ) return nullptr;
	LPCTGA_HEADER phdr = GetHeader();
	u32 ofs = sizeof(TGA_HEADER) + phdr->IDLength + GetPaletteSize();
	return m_pBuffer + ofs;
}

/**********************************************************************//**
 *
 * CfbNXf[^TCY̎擾
 *
 ----------------------------------------------------------------------
 * @return	CfbNXf[^TCY
*//***********************************************************************/
u32 CTga::GetIndexSize(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return (u32)phdr->Width * phdr->Height;
}

/**********************************************************************//**
 *
 * C[Wf[^̎擾
 *
 ----------------------------------------------------------------------
 * @return	C[Wf[^AhX
*//***********************************************************************/
const u8* CTga::GetImage(void)	const
{
	if( m_pBuffer == nullptr ) return nullptr;
	LPCTGA_HEADER phdr = GetHeader();
	u32 ofs = sizeof(TGA_HEADER) + phdr->IDLength + GetPaletteSize();
	return m_pBuffer + ofs;
}

/**********************************************************************//**
 *
 * C[Wf[^TCY̎擾
 *
 ----------------------------------------------------------------------
 * @return	C[Wf[^TCY
*//***********************************************************************/
u32 CTga::GetImageSize(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return (u32)phdr->Width * phdr->Height * GetBytePerPixel();
}

/**********************************************************************//**
 *
 * [x擾
 *
 ----------------------------------------------------------------------
 * @return	[x
*//***********************************************************************/
u8 CTga::GetPixelDepth(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return phdr->PixelDepth;
}

/**********************************************************************//**
 *
 * J[f[^bit擾
 *
 ----------------------------------------------------------------------
 * @return	J[f[^bit
*//***********************************************************************/
u8 CTga::GetColorMapBits(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return phdr->ColorMapBit;
}

/**********************************************************************//**
 *
 * ̎擾
 *
 ----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
s32 CTga::GetWidth(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return (s32)phdr->Width;
}

/**********************************************************************//**
 *
 * c̎擾
 *
 ----------------------------------------------------------------------
 * @return	c
*//***********************************************************************/
s32 CTga::GetHeight(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return (s32)phdr->Height;
}

/**********************************************************************//**
 *
 * C[W^Cv̎擾
 *
 ----------------------------------------------------------------------
 * @return	C[W^Cv
*//***********************************************************************/
u8 CTga::GetImageType(void)	const
{
	if( m_pBuffer == nullptr ) return TGA_IMAGETYPE_NONE;
	LPCTGA_HEADER phdr = GetHeader();
	return phdr->ImageType;
}

/**********************************************************************//**
 *
 * sNZf[^̊i[擾
 *
 ----------------------------------------------------------------------
 * @return	
*//***********************************************************************/
u8 CTga::GetPixelPutDirection(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	return (u8)((phdr->ImageDesc >> 4) & TGA_PPD_MASK);
}

/**********************************************************************//**
 *
 * 1pixel̃oCg擾
 *
 ----------------------------------------------------------------------
 * @return	1pixel̃oCg
*//***********************************************************************/
int CTga::GetBytePerPixel(void)	const
{
	if( m_pBuffer == nullptr ) return 0;
	LPCTGA_HEADER phdr = GetHeader();
	int ret = 0;
	switch(phdr->ImageType)
	{
	case TGA_IMAGETYPE_NONE:
	default:
		break;
	case TGA_IMAGETYPE_INDEX:
	case TGA_IMAGETYPE_INDEX_RLE:
		IRIS_ASSERT( phdr->ColorMapType != 0 );
	case TGA_IMAGETYPE_FULL:
	case TGA_IMAGETYPE_FULL_RLE:
		ret = 3;
		if( phdr->PixelDepth > 24 ) ret = 4;
		break;
	case TGA_IMAGETYPE_GRAY:
	case TGA_IMAGETYPE_GRAY_RLE:
		ret = 1;
		if( phdr->PixelDepth == 16 ) ret = 2;
		break;
	}
	return ret;
}

/**********************************************************************//**
 *
 * wWpixelf[^̎擾
 *
 ----------------------------------------------------------------------
 * @param [in]	x	= xW
 * @param [in]	y	= yW
 * @return	pixelf[^AhX
*//***********************************************************************/
void* CTga::GetPixel(int x, int y)	const
{
	if( m_pBuffer == nullptr ) return nullptr;
	LPCTGA_HEADER phdr = GetHeader();
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	if( x < 0 || x > w ) return nullptr;
	if( y < 0 || y > h ) return nullptr;
	int ppd = (phdr->ImageDesc >> 4) & TGA_PPD_MASK;
	if( ppd & TGA_PPD_RIGHT )	x = (w-x-1);
	if( !(ppd & TGA_PPD_TOP) )	y = (h-y-1);

	switch( phdr->ImageType )
	{
	case TGA_IMAGETYPE_INDEX:
		{
			const u8* idx = GetIndex();
			u32 palette_size = (u32)(phdr->PixelDepth > 24 ? 4 : 3);
			int idx_size = 1;
			if( GetPaletteSize() > palette_size*256 ) idx_size = 2;
			int ofs = x*idx_size + y*w*idx_size;
			idx += ofs;
			return (void*)idx;
		}
		// no break;
	case TGA_IMAGETYPE_FULL:
		{
			const u8* img = GetImage();
			int palette_size = phdr->PixelDepth > 24 ? 4 : 3;
			int ofs = x*palette_size + y*w*palette_size;
			img += ofs;
			return (void*)img;
		}
		// no break;
	case TGA_IMAGETYPE_GRAY:
		break;
	case TGA_IMAGETYPE_INDEX_RLE:
	case TGA_IMAGETYPE_FULL_RLE:
	case TGA_IMAGETYPE_GRAY_RLE:
		break;
	}
	return nullptr;
}

/**********************************************************************//**
 *
 * w̃sNZ̐F擾
 *
 ----------------------------------------------------------------------
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @param [out]	rgba	= F
 * @return	
*//***********************************************************************/
bool CTga::GetPixelRGBA8888(s32 x, s32 y, IrisRGBA8888& rgba) const
{
	if( m_pBuffer == nullptr ) return false;
	bool ret = false;
	LPCTGA_HEADER phdr = GetHeader();
	switch( phdr->ImageType )
	{
	case TGA_IMAGETYPE_INDEX:
		{
			void* pixel = GetPixel(x, y);
			if( pixel != nullptr )
			{
				u32 palette_size = (u32)(phdr->PixelDepth > 24 ? 4 : 3);
				const u8* palette = GetPalette();
				u16 idx = 0;
				if( GetPaletteSize() > palette_size*256 )	idx = *(u16*)idx;
				else										idx = (u16)*(u8*)idx;
				rgba.r = *(palette + idx + 0);
				rgba.g = *(palette + idx + 1);
				rgba.b = *(palette + idx + 2);
				rgba.a = (u8)(palette_size > 3 ? *(palette + idx + 3) : 0xFF);
			}
		}
		break;
	case TGA_IMAGETYPE_FULL:
		{
			u8* pixel = (u8*)GetPixel(x, y);
			if( pixel != nullptr )
			{
				int palette_size = phdr->PixelDepth > 24 ? 4 : 3;
				rgba.r = *(pixel + 0);
				rgba.g = *(pixel + 1);
				rgba.b = *(pixel + 2);
				rgba.a = (u8)(palette_size > 3 ? *(pixel + 3) : 0xFF);
			}
		}
		break;
	case TGA_IMAGETYPE_GRAY:
		break;
	case TGA_IMAGETYPE_INDEX_RLE:
	case TGA_IMAGETYPE_FULL_RLE:
	case TGA_IMAGETYPE_GRAY_RLE:
	case TGA_IMAGETYPE_NONE:
	default:
		break;
	}
	return ret;
}

/**********************************************************************//**
 *
 * w̃sNZ̐Fݒ
 *
 ----------------------------------------------------------------------
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @param [in]	rgba	= F
 * @return	
*//***********************************************************************/
bool CTga::SetPixelRGBA8888(s32 x, s32 y, const IrisRGBA8888& rgba)
{
	if( m_pBuffer == nullptr ) return false;
	bool ret = false;
	LPCTGA_HEADER phdr = GetHeader();
	switch( phdr->ImageType )
	{
	case TGA_IMAGETYPE_FULL:
		{
			u8* pixel = (u8*)GetPixel(x, y);
			if( pixel != nullptr )
			{
				int palette_size = phdr->PixelDepth > 24 ? 4 : 3;
				*(pixel + 0) = rgba.r;
				*(pixel + 1) = rgba.g;
				*(pixel + 2) = rgba.b;
				if( palette_size > 3 ) *(pixel + 3) = rgba.a;
			}
		}
		break;
	case TGA_IMAGETYPE_GRAY:
		break;
	case TGA_IMAGETYPE_INDEX:
	case TGA_IMAGETYPE_INDEX_RLE:
	case TGA_IMAGETYPE_FULL_RLE:
	case TGA_IMAGETYPE_GRAY_RLE:
	case TGA_IMAGETYPE_NONE:
	default:
		break;
	}
	return ret;
}

/**********************************************************************//**
 *
 * obt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImage(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	bool ret = false;
	switch(phdr->ImageType)
	{
	case TGA_IMAGETYPE_NONE:
	default:
		break;
	case TGA_IMAGETYPE_INDEX:
		ret = CreateImageFromIndex(pBuffer, nPixelFormat, nLineAlign, nOrder);
		break;
	case TGA_IMAGETYPE_INDEX_RLE:
		ret = CreateImageFromIndexRLE(pBuffer, nPixelFormat, nLineAlign, nOrder);
		break;
	case TGA_IMAGETYPE_FULL:
		ret = CreateImageFromFullColor(pBuffer, nPixelFormat, nLineAlign, nOrder);
		break;
	case TGA_IMAGETYPE_FULL_RLE:
		ret = CreateImageFromFullColorRLE(pBuffer, nPixelFormat, nLineAlign, nOrder);
		break;
	case TGA_IMAGETYPE_GRAY:
		ret = CreateImageFromGray(pBuffer, nPixelFormat, nLineAlign, nOrder);
		break;
	case TGA_IMAGETYPE_GRAY_RLE:
		ret = CreateImageFromGrayRLE(pBuffer, nPixelFormat, nLineAlign, nOrder);
		break;
	}
	return ret;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_INDEXobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromIndex(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	if( GetPaletteSize() > 1024 )
	{
		return CreateImageFromIndex16(pBuffer, nPixelFormat, nLineAlign, nOrder);
	}
	return CreateImageFromIndex8(pBuffer, nPixelFormat, nLineAlign, nOrder);
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_INDEXobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromIndex8(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i=0;
	u16 w = phdr->Width;
	u16 h = phdr->Height;
	int pallete_size = phdr->ColorMapBit > 24 ? 4 : 3;
	const u8* plt = GetPalette();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	IRIS_ASSERT( GetPaletteSize() <= 1024 );
	if( pallete_size == 4 )
		pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGRA8, nPixelFormat, &pf_size);
	else
		pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGR8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;

	int line_buf_size = roundup(w * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}

	const u8* idx = GetIndex();
	for( u16 y=0; y < h; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < w; ++x, ++idx )
		{
			u8 index = *idx;
			const u8* src = (plt + index * pallete_size + i);
			(*pfnFmtExchange)(dstx, src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_INDEXobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromIndex16(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i=0;
	u16 width = phdr->Width;
	u16 height = phdr->Height;
	int pallete_size = phdr->ColorMapBit > 24 ? 4 : 3;
	const u8* plt = GetPalette();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	IRIS_ASSERT( GetPaletteSize() <= 1024 );
	if( pallete_size == 4 )
		pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGRA8, nPixelFormat, &pf_size);
	else
		pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGR8, nPixelFormat, &pf_size);

	if( pfnFmtExchange == nullptr ) return false;

	int line_buf_size = roundup(width * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (width-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (height-1) * line_buf_size;
	}

	const u16* idx = (const u16*)GetIndex();
	for( u16 y=0; y < height; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < width; ++x, ++idx )
		{
			u16 index = *idx;
			index = (u16)((index >> 8) | (index << 8));
			const u8* src = plt + index * pallete_size + i;
			(*pfnFmtExchange)(dstx, src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColor(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	switch( phdr->PixelDepth )
	{
	case 16:
		return CreateImageFromFullColor16(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 24:
		return CreateImageFromFullColor24(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 32:
		return CreateImageFromFullColor32(pBuffer, nPixelFormat, nLineAlign, nOrder);
	}
	return false;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColor16(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	u16 w = phdr->Width;
	u16 h = phdr->Height;
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchangeBySrc<CBGRA5551_REV>(nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}

	const u16* src = (const u16*)GetImage();
	for( u16 y=0; y < h; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < w; ++x, ++src )
		{
			(*pfnFmtExchange)(dstx, (const u8*)src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColor24(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	u16 w = phdr->Width;
	u16 h = phdr->Height;
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGR8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}

	const u8* src = GetImage();
	for( u16 y=0; y < h; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < w; ++x, src+=3 )
		{
			(*pfnFmtExchange)(dstx, src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColor32(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	u16 w = phdr->Width;
	u16 h = phdr->Height;
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGRA8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}

	const u8* src = GetImage();
	for( u16 y=0; y < h; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < w; ++x, src+=4 )
		{
			(*pfnFmtExchange)(dstx, src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_GRAYobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromGray(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	switch( phdr->PixelDepth )
	{
	case 8:
		return CreateImageFromGray8(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 16:
		return CreateImageFromGray16(pBuffer, nPixelFormat, nLineAlign, nOrder);
	}
	return false;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_GRAYobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromGray8(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	u16 w = phdr->Width;
	u16 h = phdr->Height;
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_L8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}

	const u8* src = GetImage();
	for( u16 y=0; y < h; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < w; ++x, ++src )
		{
			(*pfnFmtExchange)(dstx, src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_GRAYobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromGray16(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	u16 w = phdr->Width;
	u16 h = phdr->Height;
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_LA8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);
	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}

	const u8* src = GetImage();
	for( u16 y=0; y < h; ++y, dsty+=nexty )
	{
		u8* dstx = dsty + offset;
		for( u16 x=0; x < w; ++x, src+=2 )
		{
			(*pfnFmtExchange)(dstx, src);
			dstx += nextx;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_INDEX_RLEobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromIndexRLE(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i=0;
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	const u8* idx = GetIndex();
	const u8* plt = GetPalette();
	u8*	dst = (u8*)pBuffer;
	int pallete_size = phdr->ColorMapBit > 24 ? 4 : 3;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	int size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	if( pallete_size == 4 )
		pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGRA8, nPixelFormat, &pf_size);
	else
		pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGR8, nPixelFormat, &pf_size);

	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);

	int dst_w = 0;
	int dst_h = 0;

	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}
	u8* dstx = dsty + offset;

	while( dst_h < h )
	{
		size = (*idx & 0x7F) + 1;
		if( *idx++ & 0x80 )
		{
			u8 index = *idx++;
			const u8* src = plt + index * pallete_size;
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
		else
		{
			for( i = 0; i < size; ++i )
			{
				u8 index = *idx++;
				const u8* src = plt + index * pallete_size;
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULL_RLEobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColorRLE(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	LPCTGA_HEADER phdr = GetHeader();
	switch( phdr->PixelDepth )
	{
	case 16:
		return CreateImageFromFullColorRLE16(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 24:
		return CreateImageFromFullColorRLE24(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 32:
		return CreateImageFromFullColorRLE32(pBuffer, nPixelFormat, nLineAlign, nOrder);
	}
	return false;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColorRLE16(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i;
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	const u8* img = GetImage();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	int size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchangeBySrc<CBGRA5551_REV>(nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);

	int dst_w = 0;
	int dst_h = 0;

	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}
	u8* dstx = dsty + offset;

	while( dst_h < h )
	{
		size = (*img & 0x7F) + 1;
		if( *img++ & 0x80 )
		{
			const u8* src = img;
			img += 2;
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
		else
		{
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, img);
				img += 2;
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColorRLE24(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i;
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	const u8* img = GetImage();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	int size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGR8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);

	int dst_w = 0;
	int dst_h = 0;

	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}
	u8* dstx = dsty + offset;

	while( dst_h < h )
	{
		size = (*img & 0x7F) + 1;
		if( *img++ & 0x80 )
		{
			const u8* src = img;
			img += 3;
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
		else
		{
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, img);
				img += 3;
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_FULLobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromFullColorRLE32(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	IRIS_ASSERT( pBuffer != nullptr );
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i;
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	const u8* img = GetImage();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	int size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_BGRA8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);

	int dst_w = 0;
	int dst_h = 0;

	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}
	u8* dstx = dsty + offset;

	while( dst_h < h )
	{
		size = (*img & 0x7F) + 1;
		if( *img++ & 0x80 )
		{
			const u8* src = img;
			img += 4;
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
		else
		{
			for( i = 0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, img);
				img += 4;
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_GRAYobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromGrayRLE(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	LPCTGA_HEADER phdr = GetHeader();
	switch( phdr->PixelDepth )
	{
	case 8:
		return CreateImageFromGrayRLE8(pBuffer, nPixelFormat, nLineAlign, nOrder);
	case 16:
		return CreateImageFromGrayRLE16(pBuffer, nPixelFormat, nLineAlign, nOrder);
	}
	return false;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_GRAYobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @param [in]	nLineAlign		= 1sPʂ̃ACgoCg
 * @param [in]	nOrder			= sNZi[
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromGrayRLE8(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i=0;
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	const u8* img = GetImage();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	int size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_L8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);

	int dst_w = 0;
	int dst_h = 0;

	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}
	u8* dstx = dsty + offset;

	while( dst_h < h )
	{
		size = (*img & 0x7F) + 1;
		if( *img++ & 0x80 )
		{
			const u8* src = img++;
			for( i=0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
		else
		{
			for( i=0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, img++);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * TGA_IMAGETYPE_GRAYobt@ɓWJ
 *
 ----------------------------------------------------------------------
 * @param [out]	pBuffer			= o̓obt@
 * @param [in]	nPixelFormat	= pixeltH[}bg
 * @return	
*//***********************************************************************/
bool CTga::CreateImageFromGrayRLE16(void* pBuffer, PIXEL_FORMAT nPixelFormat, int nLineAlign, int nOrder) const
{
	if( m_pBuffer == nullptr ) return false;
	LPCTGA_HEADER phdr = GetHeader();
	IRIS_ASSERT( nLineAlign > 0 );

	int i=0;
	int w = (int)phdr->Width;
	int h = (int)phdr->Height;
	const u8* img = GetImage();
	u8*	dst = (u8*)pBuffer;
	CColorExchange::PFN_COLOR_FORMAT_EXCHANGE pfnFmtExchange = func_nullptr;
	int pf_size = 0;
	int size = 0;
	const int ppd = ((phdr->ImageDesc >> 4) & TGA_PPD_MASK) ^ nOrder;

	pfnFmtExchange = CColorExchange::GetColorFormatExchange(PF_LA8, nPixelFormat, &pf_size);
	if( pfnFmtExchange == nullptr ) return false;
	int line_buf_size = roundup(w * pf_size, nLineAlign);

	int dst_w = 0;
	int dst_h = 0;

	int nextx = pf_size;
	int nexty = line_buf_size;
	int offset = 0;
	u8* dsty = dst;
	if( ppd & PPD_RIGHT )
	{
		nextx = -pf_size;
		offset = (w-1) * pf_size;
	}
	if( ppd & PPD_TOP )
	{
		nexty = -line_buf_size;
		dsty = dst + (h-1) * line_buf_size;
	}
	u8* dstx = dsty + offset;

	while( dst_h < h )
	{
		size = (*img & 0x7F) + 1;
		if( *img++ & 0x80 )
		{
			const u8* src = img;
			img += 2;
			for( i=0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, src);
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
		else
		{
			for( i=0; i < size; ++i )
			{
				(*pfnFmtExchange)(dstx, img);
				img += 2;
				dstx += nextx;
				if( ++dst_w >= w )
				{
					dst_w = 0;
					++dst_h;
					dsty += nexty;
					dstx = dsty + offset;
				}
			}
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * obt@̃m
 *
 ----------------------------------------------------------------------
 * @param [in]	size	= TCY
 * @param [in]	arg		= 
*//***********************************************************************/
void CTga::Alloc(u32 size, s32 arg)
{
	Dealloc();
	m_pBuffer = static_cast<u8*>(irisAlloc(size, arg));
	m_Size = size;
}

/**********************************************************************//**
 *
 * obt@̃
 *
*//***********************************************************************/
void CTga::Dealloc(void)
{
	if( m_pBuffer != nullptr )
	{
		irisFree(m_pBuffer);
		m_pBuffer = nullptr;
		m_Size = 0;
	}
}

}	// end of namespace fnd
}	// end of namespace iris


#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../../unit/UnitCore.h"
#include <stdio.h>
#include <string.h>
#include <tchar.h>
#include "iris_using.h"
#include "iris_iostream.h"

//======================================================================
// test
IRIS_UNITTEST(CFndTgaUnitTest, Func)
{
	CTga tga;
	TCHAR path[MAX_PATH] = TEXT("../../../data/image/sample.tga");
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "Jtgat@C͂ĂB" << std::endl;
	std::tcin >> path;
#endif

	if( !tga.ReadFile(path) )
	{
		std::cout << "t@CI[vɎs܂B" << std::endl;
		return;
	}
	std::cout << "file open" << std::endl;
}

#endif	// #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
