//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndLuminance.h
 * @brief		J[ template t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndLuminance_H_
#define INCG_IRIS_FndLuminance_H_

//======================================================================
// include
#include "FndColor.h"

namespace iris {
namespace fnd
{

//======================================================================
// class
/**
 * @brief	PxNX
*/
class CLuminance : public IIrisObject
{
protected:
	u8		m_Luminance;	//!< Pxl

public:
	static const int MAX_VALUE = 0xFF;
	static const int BYTESIZE = sizeof(u8);
	static const int BITCOUNT = 8;

public:
	/**
	 * @name RXgN^
	 * @{
	*/
	CLuminance(void) : m_Luminance(0) {}
	CLuminance(u8 luminance) : m_Luminance(luminance) {}
	CLuminance(u8 _r, u8 _g, u8 _b) { calc(_r, _g, _b); }
	CLuminance(const IrisFColor& fcol) { copy(fcol); }
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	CLuminance(const CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		copy(rObj);
	}
	/**
	 * @}
	*/

public:
	/// J[^ɕϊ
	void			cast(IrisFColor& rfcol)	const
	{
		f32 l = static_cast<f32>(m_Luminance);
		rfcol.r = l / MAX_VALUE;
		rfcol.g = l / MAX_VALUE;
		rfcol.b = l / MAX_VALUE;
		rfcol.a = 1.f;
	}
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	void			cast(CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rcol)	const
	{
		IrisFColor col;
		cast(col);
		rcol.copy(col);
	}

	/// J[^Rs[
	void			copy(const IrisFColor& rfcol)
	{
		CRGBA8888 col(rfcol);
		copy(col);
	}

	/// J[^vZ
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	void			copy(const CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		CRGBA8888 col = rObj;
		copy(col);
	}
	template<typename _OTN, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	void			copy(const CColor<_OTN, 8, 8, 8, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		calc(rObj.r(), rObj.g(), rObj.b());
	}

	void			calc(u8 r, u8 g, u8 b)
	{
		m_Luminance = Calc<8>(r, g, b);
	}

public:
	operator u8			(void)		const	{ return m_Luminance; }
	operator IrisFColor	(void)		const	{ IrisFColor fcol; cast(fcol); return fcol; }
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	operator CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT> (void) const
	{
		IrisFColor col;
		cast(col);
		return CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>(col);
	}

public:

	template<int B>
	static u8 Calc(u8 r, u8 g, u8 b)
	{
		return impl<B, void>::Calc(r, g, b);
	}

private:
	template<int B, typename TMP>
	struct impl
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			f64 dbColor = b * 0.11448;	// B
			dbColor += g * 0.58661;		// G
			dbColor += r * 0.29891;		// R

			if( dbColor > 0xFF ) dbColor = 0xFF;
			u8 ret = (u8)dbColor;
			if((ret + 1 - dbColor) <= (dbColor - ret)) ++ret;
			return ret;
		}
	};

	template<typename TMP>
	struct impl<2, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( r + (g<<1) + (b>>1) ) >> 2) & 0xFF);
		}
	};
	template<typename TMP>
	struct impl<3, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( (r<<1) + ((g<<2)+(g>>1)) + b ) >> 3) & 0xFF);
		}
	};
	template<typename TMP>
	struct impl<4, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( ((r<<2)+r) + ((g<<3)+g) + (b<<1) ) >> 4) & 0xFF);
		}
	};
	template<typename TMP>
	struct impl<5, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( ((r<<3)+r+(r>>1)) + (g*19) + ((b<<1)+b+(b>>1)) ) >> 5) & 0xFF);
		}
	};
	template<typename TMP>
	struct impl<6, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( (r*19) + (g*37) + (b*7) ) >> 6) & 0xFF);
		}
	};
	template<typename TMP>
	struct impl<7, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( (r*38) + (g*75) + (b*15) ) >> 7) & 0xFF);
		}
	};
	template<typename TMP>
	struct impl<8, TMP>
	{
		static u8 Calc(u16 r, u16 g, u16 b)
		{
			return (u8)( ( ( (r*77) + (g*150) + (b*29) ) >> 8) & 0xFF);
		}
	};
};

/**
 * @brief	Px + At@NX
*/
class CLuminanceAlpha : public CLuminance
{
protected:
	u8		m_Alpha;	//!< At@l

public:
	static const int BYTESIZE = sizeof(u16);
	static const int BITCOUNT = 16;
public:
	/**
	 * @name RXgN^
	 * @{
	*/
	CLuminanceAlpha(void) : m_Alpha(0xFF) {}
	CLuminanceAlpha(u8 luminance, u8 alpha) : CLuminance(luminance), m_Alpha(alpha) {}
	CLuminanceAlpha(u8 _r, u8 _g, u8 _b, u8 _a) { calc(_r, _g, _b, _a); }
	CLuminanceAlpha(const CLuminance& l) : CLuminance(l), m_Alpha(0xFF) {}
	CLuminanceAlpha(const IrisFColor& fcol) { copy(fcol); }
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	CLuminanceAlpha(const CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		copy(rObj);
	}
	/**
	 * @}
	*/

public:
	/// J[^ɕϊ
	void			cast(IrisFColor& rfcol)	const
	{
		f32 l = static_cast<f32>(m_Luminance);
		rfcol.r = l / MAX_VALUE;
		rfcol.g = l / MAX_VALUE;
		rfcol.b = l / MAX_VALUE;
		rfcol.a = static_cast<f32>(m_Alpha) / MAX_VALUE;
	}

	/// J[^Rs[
	void			copy(const IrisFColor& rfcol)
	{
		CRGBA8888 col(rfcol);
		copy(col);
	}

	/// J[^vZ
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	void			copy(const CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		CRGBA8888 col = rObj;
		copy(col);
	}
	template<typename _OTN
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	void			copy(const CColor<_OTN, 8, 8, 8, 8, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		calc(rObj.r(), rObj.g(), rObj.b(), rObj.a());
	}

	void			calc(u8 r, u8 g, u8 b, u8 a)
	{
		m_Luminance = Calc<8>(r, g, b);
		m_Alpha = a;
	}
public:
	operator u16			(void)			{ return static_cast<u16>((m_Alpha << 8) | m_Luminance); }
};

}	// end of namespace fnd
}	// end of namespace iris

#endif
