//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndEdgeDetector.cpp
 * @brief		GbWot@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndEdgeDetector_CPP_

//======================================================================
// include
#include "FndEdgeDetector.h"

namespace iris {
namespace fnd
{

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
CEdgeDetector::CEdgeDetector(void)
{
	for( int i=0; i < MAX_KERNEL; ++i )
	{
		m_pKernel[i] = nullptr;
	}
}

/**********************************************************************//**
 *
 * o
 *
 -----------------------------------------------------------------------
 * @param [out]	lpDst	= obitmap
 * @param [in]	lpSrc	= bitmap
 * @param [in]	rate	= 臒li%j(0 ~ 1)
 * @return	
*//***********************************************************************/
bool CEdgeDetector::Detect(IImage* lpDst, const IImage* lpSrc, xf32 rate)
{
	if( lpDst == nullptr ) return false;
	if( lpSrc == nullptr ) return false;
	if( m_pKernel[0] == nullptr ) return false;

	s32 w = lpSrc->GetWidth();
	s32 h = lpSrc->GetHeight();
	s32 max = 0;
	IrisRGBA8888 col;

	// ől߂
	for( int i=0; i < MAX_KERNEL; ++i )
	{
		if( m_pKernel[i] != nullptr )
		{
			int ofs = m_pKernel[i]->GetAperture()>>1;
			for( s32 y=ofs; y < h-ofs; ++y )
			{
				for( s32 x=ofs; x < w-ofs; ++x )
				{
					s32 tmp = Convolution(x, y, lpSrc, m_pKernel[i]);
					if( max < tmp ) max = tmp;
				}
			}
		}
	}

	// 臒l
	s32 threshold = XF_XF32_TO_S32(max * rate);
	for( s32 y=0; y < h; ++y )
	{
		for( s32 x=0; x < w; ++x )
		{
			s32 density = 0;
			for( int i=0; i < MAX_KERNEL; ++i )
			{
				if( m_pKernel[i] != nullptr )
				{
					int ofs = m_pKernel[i]->GetAperture()>>1;
					if( (x >= ofs && x < w-ofs)
						&& (y >= ofs && y < h-ofs) )
					{
						s32 tmp = Convolution(x, y, lpSrc, m_pKernel[i]);
						if( density < tmp ) density = tmp;
					}
				}
			}
			if( density > threshold )	density = 0;
			else						density = 0xFF;
			col.col = (u32)(IRIS_ARGB(0xFF, density, density, density));
			lpDst->SetPixelRGBA8888(x, y, col);
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * o
 *
 * @note	ꎞobt@gčɏ
 *
 -----------------------------------------------------------------------
 * @param [out]	lpDst		= obitmap
 * @param [in]	lpSrc		= bitmap
 * @param [in]	rate		= 臒li%j(0 ~ 1)
 * @param [in]	lpWork		= ꎞobt@
 * @param [in]	WorkSize	= ꎞobt@TCY
 * @return	
*//***********************************************************************/
bool CEdgeDetector::DetectFast(IImage* lpDst, const IImage* lpSrc, xf32 rate, void* lpWork, u32 WorkSize)
{
	if( lpDst == nullptr ) return false;
	if( lpSrc == nullptr ) return false;
	if( m_pKernel[0] == nullptr ) return false;
	if( lpWork == nullptr )
	{
		IRIS_WARNING("lpWork == nullptr.");
		return Detect(lpDst, lpSrc, rate);
	}

	s32 w = lpSrc->GetWidth();
	s32 h = lpSrc->GetHeight();
	IRIS_VERIFYRETURN( WorkSize >= w*h*sizeof(s32), false );
	s32* work = (s32*)lpWork;
	for( int i=0; i < w*h; ++i ) work[i] = 0;
	s32 max = 0;
	IrisRGBA8888 col;
	int ofs = m_pKernel[0]->GetAperture()>>1;
	for( int i=1; i < MAX_KERNEL; ++i )
	{
		IRIS_ASSERT( m_pKernel[i] == nullptr || ofs == m_pKernel[i]->GetAperture()>>1 );
	}

	// Zx߂
	work += ofs*w;
	for( s32 y=ofs; y < h-ofs; ++y )
	{
		work += ofs;
		for( s32 x=ofs; x < w-ofs; ++x, ++work )
		{
			s32 density = 0;
			for( int i=0; i < MAX_KERNEL; ++i )
			{
				if( m_pKernel[i] != nullptr )
				{
					s32 tmp = Convolution(x, y, lpSrc, m_pKernel[i]);
					if( density < tmp ) density = tmp;
				}
			}
			if( max < density ) max = density;
			*work = density;
		}
		work += ofs;
	}

	// 臒l
	s32 threshold = XF_XF32_TO_S32(max * rate);
	work = (s32*)lpWork;
	for( s32 y=0; y < h; ++y )
	{
		for( s32 x=0; x < w; ++x, ++work )
		{
			s32 density = *work;
			if( density > threshold )	density = 0;
			else						density = 0xFF;
			col.col = (u32)(IRIS_ARGB(0xFF, density, density, density));
			lpDst->SetPixelRGBA8888(x, y, col);
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * ZxvZ
 *
 -----------------------------------------------------------------------
 * @param [in]	x		= xW
 * @param [in]	y		= yW
 * @param [in]	lpSrc	= bitmap
 * @param [in]	lpKernel= RA
 * @return	
*//***********************************************************************/
s32 CEdgeDetector::Convolution(s32 x, s32 y, const IImage* lpSrc, LPKERNEL lpKernel)
{
	IRIS_ASSERT( lpKernel != nullptr );
	IRIS_ASSERT( lpSrc != nullptr );
	s32 r=0, g=0, b=0;
	IrisRGBA8888 tmp;
	s32* pv = lpKernel->GetCore();
	int aperture = lpKernel->GetAperture();
	int ofs = aperture>>1;
	if( x < ofs || y < ofs ) return 0;

	for( int yy=0; yy < aperture; ++yy )
	{
		for( int xx=0; xx < aperture; ++xx, ++pv )
		{
			s32 value = *pv;
			lpSrc->GetPixelRGBA8888(x+xx-ofs, y+yy-ofs, tmp);
			r += tmp.r * value;
			g += tmp.g * value;
			b += tmp.b * value;
		}
	}
	if( r < 0 ) r = -r;
	if( g < 0 ) g = -g;
	if( b < 0 ) b = -b;
	return (r + g + b)/3;
}

}	// end of namespace fnd
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../../unit/UnitCore.h"
#include "FndBitmap.h"
#include "iris_using.h"
#include "../../iris_iostream.h"
#include <stdio.h>
#include <string.h>
#include <tchar.h>

//======================================================================
// define
#define TEST_FAST	1

//======================================================================
// test
IRIS_UNITTEST(CEdgeDetectorUnitTest, Func)
{
	::iris::fnd::CBitmap bitmap, out;
	TCHAR path[MAX_PATH] = TEXT("../../../data/image/sample.bmp");
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "Jbitmapt@C͂ĂB" << std::endl;
	std::tcin >> path;
#endif

	if( !bitmap.ReadFile(path) )
	{
		std::cout << "t@CI[vɎs܂B" << std::endl;
		return;
	}
#if TEST_FAST
	IrisU32 work_size = (bitmap.GetWidth() * bitmap.GetHeight()) * sizeof(IrisS32);
	u8* work = new u8 [work_size];
#endif
	std::cout << "file open" << std::endl;
	out.Duplicate(bitmap);
	std::cout << "duplicate"  << std::endl;

	xf32 rate = XF32_CONST(1.0f/16.0f);
//	xf32 rate = XF32_CONST(1.0f/8.0f);
//	xf32 rate = XF32_CONST(1.0f/4.0f);
	CEdgeDetector edge;
	TCHAR fname[MAX_PATH];
	// Prewitt
	{
		CEdgeDetector::PREWITT_KERNEL_H<> prewitt_h;
		CEdgeDetector::PREWITT_KERNEL_V<> prewitt_v;
		edge.SetKernel(0, &prewitt_h);
		edge.SetKernel(1, &prewitt_v);
#if TEST_FAST
		if( edge.DetectFast(&out, &bitmap, rate, work, work_size) )
#else
		if( edge.Detect(&out, &bitmap, rate) )
#endif
		{
			std::cout << "prewitt." << std::endl;
			_tcscpy_s(fname, MAX_PATH, path);
			_tcscat_s(fname, MAX_PATH, TEXT(".prewitt.bmp"));
			out.WriteFile(fname);
		}
	}

	// Sobel
	{
		CEdgeDetector::SOBEL_KERNEL_H<> sobel_h;
		CEdgeDetector::SOBEL_KERNEL_V<> sobel_v;
		edge.SetKernel(0, &sobel_h);
		edge.SetKernel(1, &sobel_v);
#if TEST_FAST
		if( edge.DetectFast(&out, &bitmap, rate, work, work_size) )
#else
		if( edge.Detect(&out, &bitmap, rate) )
#endif
		{
			std::cout << "sobel." << std::endl;
			_tcscpy_s(fname, MAX_PATH, path);
			_tcscat_s(fname, MAX_PATH, TEXT(".sobel.bmp"));
			out.WriteFile(fname);
		}
	}

	// Laplacian 4
	{
		CEdgeDetector::LAPLACIAN4_KERNEL<> laplacian;
		edge.SetKernel(0, &laplacian);
		edge.SetKernel(1, nullptr);
#if TEST_FAST
		if( edge.DetectFast(&out, &bitmap, rate, work, work_size) )
#else
		if( edge.Detect(&out, &bitmap, rate) )
#endif
		{
			std::cout << "laplacian4." << std::endl;
			_tcscpy_s(fname, MAX_PATH, path);
			_tcscat_s(fname, MAX_PATH, TEXT(".laplacian4.bmp"));
			out.WriteFile(fname);
		}
	}

	// Laplacian 8
	{
		CEdgeDetector::LAPLACIAN8_KERNEL<> laplacian;
		edge.SetKernel(0, &laplacian);
		edge.SetKernel(1, nullptr);
#if TEST_FAST
		if( edge.DetectFast(&out, &bitmap, rate, work, work_size) )
#else
		if( edge.Detect(&out, &bitmap, rate) )
#endif
		{
			std::cout << "laplacian8." << std::endl;
			_tcscpy_s(fname, MAX_PATH, path);
			_tcscat_s(fname, MAX_PATH, TEXT(".laplacian8.bmp"));
			out.WriteFile(fname);
		}
	}

#if TEST_FAST
	delete [] work;
#endif
}

#endif	// #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
