//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndColor.h
 * @brief		J[ template t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndColor_H_
#define INCG_IRIS_FndColor_H_

//======================================================================
// include
#include "../../iris_object.h"
#include "../../iris_debug.h"
#include "../../c++0x/cpp0x_enable_if.hpp"

namespace iris {
namespace fnd
{

//======================================================================
// declare
template<typename TN_, int RBITS_, int GBITS_, int BBITS_, int ABITS_
			, int RSHIFT_, int GSHIFT_, int BSHIFT_, int ASHIFT_>
class CColor;

//======================================================================
// typedef
// low byte 疽
typedef CColor<u32, 8, 8, 8, 8,  0,  8, 16, 24>	CRGBA8888;
typedef CColor<u32, 8, 8, 8, 8, 24, 16,  8,  0>	CRGBA8888_REV;
typedef CColor<u32, 8, 8, 8, 8, 16,  8,  0, 24>	CBGRA8888;
typedef CColor<u32, 8, 8, 8, 8,  8, 16, 24,  0>	CBGRA8888_REV;
typedef CColor<u32, 8, 8, 8, 8,  0, 10, 20, 30>	CRGBAXXX2;
typedef CColor<u32, 8, 8, 8, 8, 22, 12,  2,  0>	CRGBAXXX2_REV;
typedef CColor<u32, 8, 8, 8, 8, 20, 10,  0, 30>	CBGRAXXX2;
typedef CColor<u32, 8, 8, 8, 8,  2, 12, 22,  0>	CBGRAXXX2_REV;

typedef CColor<u24, 8, 8, 8, 0,  0,  8, 16, 24>	CRGB888;
typedef CColor<u24, 8, 8, 8, 0, 16,  8,  0, 24>	CBGR888;
typedef CColor<u32, 8, 8, 8, 0,  0,  8, 16, 24>	CCOLORREF;

typedef CColor<u16, 4, 4, 4, 4, 12, 8,  4,  0>	CRGBA4444;
typedef CColor<u16, 4, 4, 4, 4,  0, 4,  8, 12>	CRGBA4444_REV;
typedef CColor<u16, 5, 5, 5, 1, 11, 6,  1,  0>	CRGBA5551;
typedef CColor<u16, 5, 5, 5, 1,  0, 5, 10, 15>	CRGBA5551_REV;
typedef CColor<u16, 5, 5, 5, 1,  1, 6, 11,  0>	CBGRA5551;
typedef CColor<u16, 5, 5, 5, 1, 10, 5,  0, 15>	CBGRA5551_REV;
typedef CColor<u16, 5, 6, 5, 0, 11, 5,  0,  0>	CRGB565;
typedef CColor<u16, 5, 6, 5, 0,  0, 5, 11,  0>	CRGB565_REV;

typedef CColor< u8, 3, 3, 2, 0,  5, 2,  0,  0>	CRGB332;
typedef CColor< u8, 3, 3, 2, 0,  0, 3,  6,  0>	CRGB332_REV;

//======================================================================
// class
/**
 * @brief	J[ev[g
 * @details	(@ref section_autoexp_CColor "autoexp")
 * @tparam	TN_		= RGBA̐lێ^
 * @tparam	RBITS_	= R lbit
 * @tparam	GBITS_	= G lbit
 * @tparam	BBITS_	= B lbit
 * @tparam	ABITS_	= A lbit
 * @tparam	RSHIFT_	= R lւ̉EVtg
 * @tparam	GSHIFT_	= G lւ̉EVtg
 * @tparam	BSHIFT_	= B lւ̉EVtg
 * @tparam	ASHIFT_	= A lւ̉EVtg
*/
template<typename TN_, int RBITS_, int GBITS_, int BBITS_, int ABITS_
			, int RSHIFT_, int GSHIFT_, int BSHIFT_, int ASHIFT_>
class CColor : public IIrisObject
{
	typedef CColor<TN_, RBITS_, GBITS_, BBITS_, ABITS_, RSHIFT_, GSHIFT_, BSHIFT_, ASHIFT_>	_Myt;
public:
	typedef enum
	{
		BSZ_R = RBITS_,			//!< R lbit
		BSZ_G = GBITS_,			//!< G lbit
		BSZ_B = BBITS_,			//!< B lbit
		BSZ_A = ABITS_,			//!< A lbit
		BSZ_ALL = BSZ_R + BSZ_G + BSZ_B + BSZ_A	//!< bit
	} BITSIZE;
	typedef enum
	{
		SFT_R	= RSHIFT_,		//!< R lւ̉EVtg
		SFT_G	= GSHIFT_,		//!< G lւ̉EVtg
		SFT_B	= BSHIFT_,		//!< B lւ̉EVtg
		SFT_A	= ASHIFT_		//!< A lւ̉EVtg
	} SHIFT;

	typedef TN_		value_type;	//!< RGBA̐lێ^
	typedef typename int_least_type<BSZ_R>::UInt	value_type_r;
	typedef typename int_least_type<BSZ_G>::UInt	value_type_g;
	typedef typename int_least_type<BSZ_B>::UInt	value_type_b;
	typedef typename int_least_type<BSZ_A>::UInt	value_type_a;
	static const int	BYTESIZE = sizeof(value_type);
	static const int	BITCOUNT = BSZ_ALL;

private:
	value_type		m_col;	//!< J[l

//	IRIS_STATIC_ASSERT( sizeof(value_type)*8 >= BSZ_ALL );

public:
	static bool	HasAplha(void)	{ return BSZ_A != 0; }

public:
	/**
	 * @name RXgN^
	 * @{
	*/
	CColor(void) : m_col(0) {}
	CColor(value_type col) : m_col(col) {}
	CColor(value_type_r _r, value_type_g _g, value_type_b _b, value_type_a _a=0) : m_col(0) { r(_r); g(_g); b(_b); a(_a); }
	CColor(const IrisFColor& fcol) : m_col(0) { copy(fcol); }

	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	CColor(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	: m_col(0)
	{
		r(rObj);
		g(rObj);
		b(rObj);
		a(rObj);
	}
	CColor(const _Myt& rObj)
	{
		m_col = rObj.m_col;
	}

	/**
	 * @}
	*/

	/**
	 * @name	operation
	 * @{
	*/
	_Myt&	operator *= (f32 rate)
	{
		r( static_cast<value_type_r>( r() * rate ) );
		g( static_cast<value_type_g>( g() * rate ) );
		b( static_cast<value_type_b>( b() * rate ) );
		a( static_cast<value_type_a>( a() * rate ) );
		return *this;
	}


	bool	operator == (value_type rhs) const
	{
		return m_col == rhs;
	}
	bool	operator == (const _Myt& rhs) const
	{
		return m_col == rhs.m_col;
	}
	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	bool	operator == (const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rhs) const
	{
		// lhs ̐xɍ킹
		_Myt tmp(rhs);
		return *this == tmp;
	}

	bool	operator != (value_type rhs) const
	{
		return m_col != rhs;
	}
	bool	operator != (const _Myt& rhs) const
	{
		return m_col == rhs.m_col;
	}
	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	bool	operator != (const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rhs) const
	{
		// lhs ̐xɍ킹
		_Myt tmp(rhs);
		return *this != tmp;
	}
	/**
	 * @}
	*/

public:
	/**
	 * @name	e̍ől
	 * @{
	*/
	IRIS_CONSTEXPR value_type_r	max_r(void)	const	{ return (((value_type_r)1<<BSZ_R)-1); }
	IRIS_CONSTEXPR value_type_g	max_g(void)	const	{ return (((value_type_g)1<<BSZ_G)-1); }
	IRIS_CONSTEXPR value_type_b	max_b(void)	const	{ return (((value_type_b)1<<BSZ_B)-1); }
	IRIS_CONSTEXPR value_type_a	max_a(void)	const	{ return (((value_type_a)1<<BSZ_A)-1); }
	/**
	 * @}
	*/

public:
	/**
	 * @name	e̎擾
	 * @{
	*/
	value_type_r	r(void)		const	{ return (value_type_r)((m_col >> SFT_R) & max_r()); }
	value_type_g	g(void)		const	{ return (value_type_g)((m_col >> SFT_G) & max_g()); }
	value_type_b	b(void)		const	{ return (value_type_b)((m_col >> SFT_B) & max_b()); }
	value_type_a	a(void)		const	{ return (value_type_a)((m_col >> SFT_A) & max_a()); }
	/**
	 * @}
	*/

	/**
	 * @name	e̕ϊ
	*/
	template<typename U, int BITS_>
	U				r(void)		const	{ return convert<BITS_, BSZ_R, U>::shift(r()); }
	template<typename U, int BITS_>
	U				g(void)		const	{ return convert<BITS_, BSZ_G, U>::shift(g()); }
	template<typename U, int BITS_>
	U				b(void)		const	{ return convert<BITS_, BSZ_B, U>::shift(b()); }
	template<typename U, int BITS_>
	U				a(void)		const	{ return convert_alpha<BITS_, BSZ_A, U>::shift(a()); }

	template<typename U>
	U				r(void)		const	{ return convert<sizeof(U)*8, BSZ_R, U>::shift(r()); }
	template<typename U>
	U				g(void)		const	{ return convert<sizeof(U)*8, BSZ_G, U>::shift(g()); }
	template<typename U>
	U				b(void)		const	{ return convert<sizeof(U)*8, BSZ_B, U>::shift(b()); }
	template<typename U>
	U				a(void)		const	{ return convert_alpha<sizeof(U)*8, BSZ_A, U>::shift(a()); }
	/**
	 * @}
	*/

	/**
	 * @name	e̐ݒ
	 * @{
	*/
	void			r(value_type_r v)		{ value_type max = (value_type)max_r(); m_col &= ~(max << SFT_R); m_col |= ((v > max ? max : (value_type)v)) << SFT_R; }
	void			g(value_type_g v)		{ value_type max = (value_type)max_g(); m_col &= ~(max << SFT_G); m_col |= ((v > max ? max : (value_type)v)) << SFT_G; }
	void			b(value_type_b v)		{ value_type max = (value_type)max_b(); m_col &= ~(max << SFT_B); m_col |= ((v > max ? max : (value_type)v)) << SFT_B; }
	void			a(value_type_a v)		{ a_<BSZ_A>(v); }

	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			r(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		r( convert<BSZ_R, ORBITS_, value_type_r>::shift(rObj.r()) );
	}
	template<typename OTN_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			r(const CColor<OTN_, BSZ_R, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		r(rObj.r());
	}

	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			g(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		g( convert<BSZ_G, OGBITS_, value_type_g>::shift(rObj.g()) );
	}
	template<typename OTN_, int ORBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			g(const CColor<OTN_, ORBITS_, BSZ_G, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		g(rObj.g());
	}

	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			b(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		b( convert<BSZ_B, OBBITS_, value_type_b>::shift(rObj.b()) );
	}
	template<typename OTN_, int ORBITS_, int OGBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			b(const CColor<OTN_, ORBITS_, OGBITS_, BSZ_B, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		b(rObj.b());
	}

	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			a(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj, typename enable_if<(OABITS_!=0)>::type*& = cpp0x::enabler::value)
	{
		a( convert<BSZ_A, OABITS_, value_type_a>::shift(rObj.a()) );
	}
	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_, int OABITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
				void			a(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, OABITS_, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj, typename enable_if<(OABITS_==0)>::type*& = cpp0x::enabler::value)
	{
		// Eӂ a  0 ̏ꍇB
		IRIS_UNUSED_VAR(rObj);
		a(max_a());
	}
	template<typename OTN_, int ORBITS_, int OGBITS_, int OBBITS_
				, int ORSHIFT_, int OGSHIFT_, int OBSHIFT_, int OASHIFT_>
	void			a(const CColor<OTN_, ORBITS_, OGBITS_, OBBITS_, BSZ_A, ORSHIFT_, OGSHIFT_, OBSHIFT_, OASHIFT_>& rObj)
	{
		a(rObj.a());
	}
	/**
	 * @}
	*/

public:
	/// J[^ɕϊ
	void			cast(IrisFColor& rfcol)	const
	{
		// BSZ_* ͒萔Ȃ̂ōœKɊ
		rfcol.r = BSZ_R ? static_cast<f32>(r()) / max_r() : 0.f;
		rfcol.g = BSZ_G ? static_cast<f32>(g()) / max_g() : 0.f;
		rfcol.b = BSZ_B ? static_cast<f32>(b()) / max_b() : 0.f;
		rfcol.a = BSZ_A ? static_cast<f32>(a()) / max_a() : 1.f;
	}

	/// J[^Rs[
	void			copy(const IrisFColor& rfcol)
	{
		r( static_cast<value_type_r>((rfcol.r > 1.0f) ? max_r() : (rfcol.r < 0.0f) ? 0 : rfcol.r * max_r()) );
		g( static_cast<value_type_g>((rfcol.g > 1.0f) ? max_g() : (rfcol.g < 0.0f) ? 0 : rfcol.g * max_g()) );
		b( static_cast<value_type_b>((rfcol.b > 1.0f) ? max_b() : (rfcol.b < 0.0f) ? 0 : rfcol.b * max_b()) );
		a( static_cast<value_type_a>((rfcol.a > 1.0f) ? max_a() : (rfcol.a < 0.0f) ? 0 : rfcol.a * max_a()) );
	}

public:
	IrisFColor	Normalize(void) const	{ IrisFColor fcol; cast(fcol); return fcol; }
public:
	operator value_type	(void)	const	{ return m_col; }
	operator IrisFColor	(void)	const	{ return Normalize(); }

private:
	template<int BTO, int BFROM, typename TO>
	class convert
	{
		template<int BTO_, int BFROM_, typename FROM>
		struct impl
		{
			template<bool bSubtractive, typename DMY>
			struct impl2
			{
				static inline TO	shift(FROM value)
				{
					// F
					return static_cast<TO>( value >> (BFROM - BTO) );
				}
			};
			template<typename DMY>
			struct impl2<false, DMY>
			{
				static inline TO	shift(FROM value)
				{
					// F
					TO tmp = static_cast<TO>(value) << (BTO - BFROM);
					return static_cast<TO>(tmp | (tmp >> BFROM));
				}
			};

			static inline TO shift(FROM value)
			{
				return impl2<(BTO_ < BFROM), void>::shift(value);
			}
		};
		// rbgꏏ̏ꍇ́AȂ
		template<int B, typename FROM>
		struct impl<B, B, FROM>
		{
			static inline TO shift(FROM value) { return static_cast<TO>(value); }
		};
		// ǂ炩̃rbg 0 ̏ꍇ́A0 m
		template<int B, typename FROM>
		struct impl<B, 0, FROM>
		{
			static inline TO shift(FROM /*value*/) { return 0; }
		};
		template<int B, typename FROM>
		struct impl<0, B, FROM>
		{
			static inline TO shift(FROM /*value*/) { return 0; }
		};
		template<typename FROM>
		struct impl<0, 0, FROM>
		{
			static inline TO shift(FROM /*value*/) { return 0; }
		};
	public:
		template<typename FROM>
		static inline TO shift(FROM value)
		{
			return impl<BTO, BFROM, FROM>::shift(value);
		}
	};

	template<int BTO, int BFROM, typename TO>
	class convert_alpha
	{
		template<int BTO_, int BFROM_, typename FROM>
		struct impl
		{
			static inline TO shift(FROM value)
			{
				return convert<BTO, BFROM, TO>::shift(value);
			}
		};
		// src  0 bit ̏ꍇ́Aől
		template<int B, typename FROM>
		struct impl<B, 0, FROM>
		{
			static inline TO shift(FROM /*value*/) { return (1<<B)-1; }
		};
	public:
		template<typename FROM>
		static inline TO shift(FROM value)
		{
			return impl<BTO, BFROM, FROM>::shift(value);
		}
	};


	template<int SIZE>
	void	a_(value_type_a v, typename enable_if<(SIZE!=0)>::type*& = cpp0x::enabler::value)
	{ value_type max = (value_type)max_a(); m_col &= ~(max << SFT_A); m_col |= ((v > max ? max : (value_type)v)) << SFT_A; }
	template<int SIZE>
	void	a_(value_type_a /*v*/, typename enable_if<(SIZE==0)>::type*& = cpp0x::enabler::value)	{}
};

}	// end of namespace fnd
}	// end of namespace iris

#endif

/**
 * @addtogroup	autoexp
 * @{
 * @addtogroup	Visualizer
 * @section		section_autoexp_CColor	; iris::fnd::CColor
 * @code
;------------------------------------------------------------------------------
; iris::fnd::CColor
;------------------------------------------------------------------------------
iris::fnd::CColor<*,*,*,*,*,*,*,*,*> {
	children
	(
		#(
			R : (($e.m_col >> ($e.SFT_R)) & ((1 << ($e.BSZ_R))-1)),
			G : (($e.m_col >> ($e.SFT_G)) & ((1 << ($e.BSZ_G))-1)),
			B : (($e.m_col >> ($e.SFT_B)) & ((1 << ($e.BSZ_B))-1)),
			A : (($e.m_col >> ($e.SFT_A)) & ((1 << ($e.BSZ_A))-1)),
			m_col : $e.m_col
		)
	)
	preview
	(
		#(
			$e.m_col
		)
	)
}
 * @endcode
 * @}
*/
