//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndBlockCipherMode.h
 * @brief		ubNÍ[hNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndBlockCipherMode_H_
#define INCG_IRIS_FndBlockCipherMode_H_

//======================================================================
// include
#include "FndCipherBase.h"

namespace iris {
namespace fnd
{

//======================================================================
// decalre
class CCipherECB;
template<u32 _BS>class CCipherCBC;
template<u32 _BS>class CCipherCTS;
template<u32 _BS>class CCipherCTR;
template<u32 _BS>class CCipherOFB;
template<u32 _BS>class CCipherCFB;

//======================================================================
// function
// ECBÍ
bool	EncryptECB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher);
// ECB
bool	DecryptECB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher);
// CBCÍ
bool	EncryptCBC(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv);
// CBC
bool	DecryptCBC(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv);
// CTSÍ
bool	EncryptCTS(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv);
// CTS
bool	DecryptCTS(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv);
// CTRÍ
bool	EncryptCTR(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, u32 counter, u8* work);
// CTR
bool	DecryptCTR(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, u32 counter, u8* work);
// OFBÍ
bool	EncryptOFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work);
// OFB
bool	DecryptOFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work);
// CFBÍ
bool	EncryptCFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work);
// CFB
bool	DecryptCFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work);


//======================================================================
// class
/// ubNÍ[hNXC^[tFCX
class ICipherMode : public IIrisObject
{
public:
	/// RXgN^
	ICipherMode(void) {}
public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	= 0;
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	= 0;
public:
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* iv, u32 nSize)	= 0;
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)	= 0;
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)	= 0;
};

/**
 * @brief	ECB (Electronic Code Book Mode)
 *			ubNɏÍubNPɘAĈÍ
 *			̓oCǵAubNoCg̐{łKv
 * @note	Ci = Ek(Mi), Mi = Dk(Ci)
 *			- + 񏈗ŕAÍł
 *			- - pl͔B
 *			- - ꕽ͓̈Íɕϊ邽߁Aǂ₷B
*/
class CCipherECB : public ICipherMode
{
	typedef enum
	{
		STREAM		= 0		//!< Xg[Í
	} TYPES;
public:
	/// RXgN^
	CCipherECB(void)	{}
public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	{ return 1; }
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	{ return STREAM; }
public:
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* /*iv*/, u32 /*nSize*/) {}
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)	{ return EncryptECB(dst, src, nSize, pCipher); }
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)	{ return DecryptECB(dst, src, nSize, pCipher); }
};

/**
 * @brief	CBC (Cipher Block Chaining Mode)
 *			ÍubNA[h
 *			̓oCǵAubNoCg̐{łKv
 * @note	Ci = Ek(Mi ^ Ci-1), Mi = Dk(Ci) ^ Ci-1
 *			- + ̂Ƃ̂ݕ񏈗\B
 *			- - rbgPʂ̃G[ƁASubNɔgyBiG[Ɏアj
 *			- - IVŒ肾ƐƎB
 *				(IVcounterłƎB)
*/
template<u32 _BS>
class CCipherCBC : public ICipherMode
{
	typedef enum
	{
		BLOCKSIZE	= _BS,	//!< ubNTCY
		STREAM		= 0		//!< Xg[Í
	} TYPES;
	u8		m_InitVector[BLOCKSIZE];	//!< xNg
public:
	/// RXgN^
	CCipherCBC(void) { for( int i=0; i < BLOCKSIZE; ++i ) m_InitVector[i] = i; }
public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	{ return 1; }
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	{ return STREAM; }
public:
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* iv, u32 nSize)
	{
		for( u32 i=0, j=0; i < BLOCKSIZE; ++i, ++j, j%=nSize ) m_InitVector[i] = iv[j];
	}
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		return EncryptCBC(dst, src, nSize, pCipher, m_InitVector);
	}
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		return DecryptCBC(dst, src, nSize, pCipher, m_InitVector);
	}
};

/**
 * @brief	CTS (Cipher Text Stealing Mode)
 *			CBCgASY
 *			ÍubNA[h
 *			̓oCǵAubNoCgȏłKv
 * @note
 *			- + Xg[Í
 *			- - rbgPʂ̃G[ƁASubNɔgyBiG[Ɏアj
 *			- - IVŒ肾ƐƎB
 *				(IVcounterłƎB)
*/
template<u32 _BS>
class CCipherCTS : public ICipherMode
{
	typedef enum
	{
		BLOCKSIZE	= _BS,	//!< ubNTCY
		STREAM		= 1		//!< Xg[Í
	} TYPES;
	u8		m_InitVector[BLOCKSIZE];	//!< xNg
public:
	/// RXgN^
	CCipherCTS(void) { for( u8 i=0; i < BLOCKSIZE; ++i ) m_InitVector[i] = i; }
public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	{ return 1; }
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	{ return STREAM; }
public:
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* iv, u32 nSize)
	{
		for( u32 i=0, j=0; i < BLOCKSIZE; ++i, ++j, j%=nSize ) m_InitVector[i] = iv[j];
	}
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		return EncryptCTS(dst, src, nSize, pCipher, m_InitVector);
	}
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		return DecryptCTS(dst, src, nSize, pCipher, m_InitVector);
	}
};

/**
 * @brief	CTR (Counter Mode)
 *			JE^[h
 * @note	Ci = Mi ^ Ek(ctr + i), Mi = Ci ^ Ek(ctr + i)
 *			- + 񏈗ŕAÍłB
 *			- + Xg[ÍB
 *			- + ÍȄꏏB
 *			- - counterŒ肾ƐƎB
 *			- - counterKXVƐƎB
 *			- - counter͗ǂB
 *			- - _uubNÍƂėpƏ񗝘_IɈSłȂB
*/
template<u32 _BS>
class CCipherCTR : public ICipherMode
{
	typedef enum
	{
		BLOCKSIZE	= _BS,	//!< ubNTCY
		STREAM		= 1		//!< Xg[Í
	} TYPES;
private:
	u32		m_Ctr;			//!< JE^
public:
	/// RXgN^
	CCipherCTR(void) : m_Ctr(0)	{}

public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	{ return 1; }
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	{ return STREAM; }
public:
	/// JE^[̐ݒ
	void	SetCounter(u32 ctr)			{ m_Ctr = ctr; }
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* iv, u32 nSize)
	{
		u8* ctr = (u8*)&m_Ctr;
		for( u32 i=0, j=0; i < 4; ++i, ++j, j%=nSize ) ctr[i] = iv[j];
	}
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		u8 work[BLOCKSIZE];
		return EncryptCTR(dst, src, nSize, pCipher, m_Ctr, work);
	}
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		u8 work[BLOCKSIZE];
		return DecryptCTR(dst, src, nSize, pCipher, m_Ctr, work);
	}

};

/**
 * @brief	OFB (Output FeedBack Mode)
 *			o̓tB[hobN[h
 *			̓oCǵAubNoCg̐{łKv
 * @note	Oi = Ek(Oi-1), Ci = Mi ^ Ek(Oi), Mi = Ci ^ Ek(Oi)
 *			- + rbgPʂ̃G[gyȂB
 *			- + Xg[ÍB
 *			- + ÍȄꏏB
 *			- - IVŒ肾ƐƎB
 *				(IVcouterł͐Ǝł͂ȂB)
*/
template<u32 _BS>
class CCipherOFB : public ICipherMode
{
	typedef enum
	{
		BLOCKSIZE	= _BS,	//!< ubNTCY
		STREAM		= 1		//!< Xg[Í
	} TYPES;
	u8		m_InitVector[BLOCKSIZE];	//!< xNg
public:
	/// RXgN^
	CCipherOFB(void) { for( int i=0; i < BLOCKSIZE; ++i ) m_InitVector[i] = i; }
public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	{ return 0; }
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	{ return STREAM; }
public:
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* iv, u32 nSize)
	{
		for( u32 i=0, j=0; i < BLOCKSIZE; ++i, ++j, j%=nSize ) m_InitVector[i] = iv[j];
	}
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		u8 work[BLOCKSIZE];
		return EncryptOFB(dst, src, nSize, pCipher, m_InitVector, work);
	}
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		u8 work[BLOCKSIZE];
		return DecryptOFB(dst, src, nSize, pCipher, m_InitVector, work);
	}
};

/**
 * @brief	CFB (Cipher FeedBack Mode)
 *			ÍtB[hobN[h
 *			̓oCǵAubNoCg̐{łKv
 * @note	Ci = Mi ^ Ek(Ci-1), Mi = Ci ^ Ek(Ci-1)
 *			- + ̂Ƃ̂ݕ񏈗\B
 *			- - ĐUɎアB
 *			- - rbgPʂ̃G[̃ubNɔgyB
 *			- - IVŒ肾ƐƎB
 *				(IVcouterłƎB)
 * @note	CTR[h𗘗pقǂB
*/
template<u32 _BS>
class CCipherCFB : public ICipherMode
{
	typedef enum
	{
		BLOCKSIZE	= _BS,	//!< ubNTCY
		STREAM		= 1		//!< Xg[Í
	} TYPES;
	u8		m_InitVector[BLOCKSIZE];	//!< xNg
public:
	/// RXgN^
	CCipherCFB(void) { for( int i=0; i < BLOCKSIZE; ++i ) m_InitVector[i] = i; }
public:
	/// Í[h̏ɕKvȍŏȃubN
	virtual u32		GetMinimumBlockNum(void)	const	{ return 0; }
	/// Xg[Íǂ
	virtual bool	IsStream(void)				const	{ return STREAM; }
public:
	/// xNg̐ݒ
	virtual void	SetInitVector(const u8* iv, u32 nSize)
	{
		for( u32 i=0, j=0; i < BLOCKSIZE; ++i, ++j, j%=nSize ) m_InitVector[i] = iv[j];
	}
public:
	/// Í
	virtual bool	Encrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		u8 work[BLOCKSIZE];
		return EncryptCFB(dst, src, nSize, pCipher, m_InitVector, work);
	}
	/// 
	virtual bool	Decrypt(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
	{
		u8 work[BLOCKSIZE];
		return DecryptCFB(dst, src, nSize, pCipher, m_InitVector, work);
	}
};

}	// end of namespace fnd
}	// end of namespace iris

#endif
