//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndBaseEncoding.cpp
 * @brief		Base GR[fBONXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011-2012 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndBaseEncoding_CPP_

//======================================================================
// include
#include "FndBaseEncoding.h"
#include "../../iris_xchar.hpp"
#include "../../iris_tchar.h"
#include "../../iris_debug.h"

namespace iris {
namespace fnd
{

namespace
{
	/**
	 * @internal
	 * @brief	GR[_[
	*/
	template<typename CHARTYPE_, eBASE_ENCODING_TYPE TYPE>
	class CEncoderImpl
	{
		template<eBASE_ENCODING_TYPE T, typename DMY_>
		struct EncodeImpl {};
		template<typename DMY_>
		struct EncodeImpl<eBASE16, DMY_>
		{
			static CHARTYPE_ Convert(u8 c)
			{
				if( c <= 0x9 ) return (CHARTYPE_)(c + IRIS_XTEXT(CHARTYPE_, '0'));
				else if( 0xa <= c && c <= 0xf ) return (CHARTYPE_)(c + IRIS_XTEXT(CHARTYPE_, 'A') - 0xa);
				return IRIS_XTEXT(CHARTYPE_, '=');
			}
		};
		template<typename DMY_>
		struct EncodeImpl<eBASE32, DMY_>
		{
			static CHARTYPE_ Convert(u8 c)
			{
				if( c <= 0x19 ) return (CHARTYPE_)(c + IRIS_XTEXT(CHARTYPE_, 'A'));
				else if( 0x1a <= c && c <= 0x1f ) return (CHARTYPE_)(c - 0x1a + IRIS_XTEXT(CHARTYPE_, '2'));
				return IRIS_XTEXT(CHARTYPE_, '=');
			}
		};
		template<typename DMY_>
		struct EncodeImpl<eBASE32_HEX, DMY_>
		{
			static CHARTYPE_ Convert(u8 c)
			{
				if( c <= 10 ) return (CHARTYPE_)(c + IRIS_XTEXT(CHARTYPE_, '0'));
				else if( 10 < c && c <= 0x1f ) return (CHARTYPE_)(c - 10 + IRIS_XTEXT(CHARTYPE_, 'A'));
				return IRIS_XTEXT(CHARTYPE_, '=');
			}
		};

		template<typename DMY_>
		struct EncodeImpl<eBASE64, DMY_>
		{
			static CHARTYPE_ Convert(u8 c)
			{
				if( c <= 0x19 ) return (CHARTYPE_)(c + IRIS_XTEXT(CHARTYPE_, 'A'));
				else if( 0x1a <= c && c <= 0x33 ) return (CHARTYPE_)(c - 0x1a + IRIS_XTEXT(CHARTYPE_, 'a'));
				else if( 0x34 <= c && c <= 0x3d ) return (CHARTYPE_)(c - 0x34 + IRIS_XTEXT(CHARTYPE_, '0'));
				else if(c == 0x3e) return IRIS_XTEXT(CHARTYPE_, '+');
				else if(c == 0x3f) return IRIS_XTEXT(CHARTYPE_, '/');
				return IRIS_XTEXT(CHARTYPE_, '=');
			}
		};

		template<eBASE_ENCODING_TYPE T, typename DMY_>
		struct DecodeImpl {};
		template<typename DMY_>
		struct DecodeImpl<eBASE16, DMY_>
		{
			static u8 Convert(CHARTYPE_ c)
			{
				const CHARTYPE_ ns = IRIS_XTEXT(CHARTYPE_, '0');
				const CHARTYPE_ ne = IRIS_XTEXT(CHARTYPE_, '9');

				const CHARTYPE_ as = IRIS_XTEXT(CHARTYPE_, 'A');
				const CHARTYPE_ ae = IRIS_XTEXT(CHARTYPE_, 'F');
				if(ns <= c && c <= ne)
					return (u8)(c - ns);
				else if(as <= c && c <= ae)
					return (u8)(c - as + 10);
				return 0;
			}
		};

		template<typename DMY_>
		struct DecodeImpl<eBASE32, DMY_>
		{
			static u8 Convert(CHARTYPE_ c)
			{
				const CHARTYPE_ ns = IRIS_XTEXT(CHARTYPE_, '2');
				const CHARTYPE_ ne = IRIS_XTEXT(CHARTYPE_, '7');

				const CHARTYPE_ as = IRIS_XTEXT(CHARTYPE_, 'A');
				const CHARTYPE_ ae = IRIS_XTEXT(CHARTYPE_, 'Z');
				if(ns <= c && c <= ne)
					return (u8)(c - ns + 0x1a);
				else if(as <= c && c <= ae)
					return (u8)(c - as);
				return 0;
			}
		};

		template<typename DMY_>
		struct DecodeImpl<eBASE32_HEX, DMY_>
		{
			static u8 Convert(CHARTYPE_ c)
			{
				const CHARTYPE_ ns = IRIS_XTEXT(CHARTYPE_, '0');
				const CHARTYPE_ ne = IRIS_XTEXT(CHARTYPE_, '9');

				const CHARTYPE_ as = IRIS_XTEXT(CHARTYPE_, 'A');
				const CHARTYPE_ ae = IRIS_XTEXT(CHARTYPE_, 'V');
				if(ns <= c && c <= ne)
					return (u8)(c - ns);
				else if(as <= c && c <= ae)
					return (u8)(c - as + 10);
				return 0;
			}
		};

		template<typename DMY_>
		struct DecodeImpl<eBASE64, DMY_>
		{
			static u8 Convert(CHARTYPE_ c)
			{
				const CHARTYPE_ ns = IRIS_XTEXT(CHARTYPE_, '0');
				const CHARTYPE_ ne = IRIS_XTEXT(CHARTYPE_, '9');

				const CHARTYPE_ as1 = IRIS_XTEXT(CHARTYPE_, 'A');
				const CHARTYPE_ ae1 = IRIS_XTEXT(CHARTYPE_, 'Z');

				const CHARTYPE_ as2 = IRIS_XTEXT(CHARTYPE_, 'a');
				const CHARTYPE_ ae2 = IRIS_XTEXT(CHARTYPE_, 'z');

				const CHARTYPE_ ex1 = IRIS_XTEXT(CHARTYPE_, '+');
				const CHARTYPE_ ex2 = IRIS_XTEXT(CHARTYPE_, '/');

				if(as1 <= c && c <= ae1)
					return (u8)(c - as1);
				else if(as2 <= c && c <= ae2)
					return (u8)(c - as2 + 0x1a);
				else if(ns <= c && c <= ne)
					return (u8)(c - ns + 0x34);
				else if(c == ex1 )
					return 0x3e;
				else if(c == ex2 )
					return 0x3f;
				return 0;
			}
		};

	public:
		static CHARTYPE_	Encode(u8 c)		{ return EncodeImpl<TYPE, void>::Convert(c); }
		static u8			Decode(CHARTYPE_ c)	{ return DecodeImpl<TYPE, void>::Convert(c); }
	};
}

/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
template<typename CHARTYPE_, eBASE_ENCODING_TYPE TYPE>
CTBaseEncoding<CHARTYPE_, TYPE>::CTBaseEncoding(void)
{
}

/**********************************************************************//**
 *
 * GR[h
 *
 ----------------------------------------------------------------------
 * @param [out]	dst			= o̓obt@
 * @param [in]	length		= o͕
 * @param [in]	src			= ̓obt@
 * @param [in]	size		= ̓obt@TCY
 * @return	񂾃TCYiKvȃTCYj
*//***********************************************************************/
template<typename CHARTYPE_, eBASE_ENCODING_TYPE TYPE>
u32 CTBaseEncoding<CHARTYPE_, TYPE>::Encode(_Mylpstr dst, size_t length, const iris::u8 *src, size_t size)
{
	IRIS_ASSERT( src != nullptr );
	IRIS_ASSERT( (void*)src != (void*)dst );
	IRIS_ASSERT( size != 0 );

	const _Mychar padding = IRIS_XTEXT(_Mychar, '=');

	u32 offset = 0;
	const u8* cur = src;
	const u8* end = src + size;
	u32 written = 0;

	_Mylpstr out = dst;
	if( length == 0 ) out = nullptr;
	u16 data = (*cur << 8);
	if( cur+1 < end ) data |= static_cast<u16>(*(cur+1));

	for( ; cur < end; )
	{
		u32 shift = 16-offset-BITS;
		u8 plane = static_cast<u8>((data >> shift) & MASK);

		_Mychar code = ToBase(plane);
		if( out != nullptr )
		{
			if( written+1 >= length ) break;
			*out++ = code;
		}
		++written;

		offset += BITS;
		if( offset & 0x8 )
		{
			data <<= 8;
			if( (++cur) + 1 < end ) data |= static_cast<u16>(*(cur+1));
			offset &= 0x7;
		}
	}

	u32 align_dif = written % NBASE;
	if( align_dif )
	{
		if( written+align_dif+1 < length )
		{
			// ߂
			if( out != nullptr )
			{
				for( u32 i=0; i < align_dif; ++i, ++out, ++written )
				{
					*out = padding;
				}
				*out = 0;
			}
			else
			{
				written += align_dif;
			}
		}
		else
		{
			// 
			align_dif = NBASE - align_dif;
			if( out != nullptr )
			{
				for( u32 i=0; i < align_dif; ++i, --out, --written )
				{
					*out = 0;
				}
			}
			else
			{
				written -= align_dif;
			}
		}
	}
	else if( written+1 < length )
	{
		*out = 0;
	}
	++written;
	return written;
}

/**********************************************************************//**
 *
 * fR[h
 *
 ----------------------------------------------------------------------
 * @param [out]	dst			= o̓obt@
 * @param [in]	length		= o͕
 * @param [in]	src			= ̓obt@
 * @param [in]	size		= ̓obt@TCY
 * @return	񂾃TCYiKvȃTCYj
*//***********************************************************************/
template<typename CHARTYPE_, eBASE_ENCODING_TYPE TYPE>
u32 CTBaseEncoding<CHARTYPE_, TYPE>::Decode(iris::u8 *dst, size_t size, _Mylpcstr src, size_t length)
{
	IRIS_ASSERT( src != nullptr );
	IRIS_ASSERT( (void*)src != (void*)dst );
	IRIS_ASSERT( size != 0 );

	u32 offset = 0;
	_Mylpcstr cur = src;
	_Mylpcstr end = src + length;
	u32 written = 0;

	u8* out = dst;
	if( size == 0 ) out = nullptr;
	u16 data = 0;

	for( ; cur < end; ++cur )
	{
		u16 plane = ToPlane(*cur) & MASK;
		u32 shift = 16-BITS-offset;
		data &= static_cast<u16>(~(MASK << shift));
		data |= plane << shift;
		offset += BITS;

		if( offset & 0x8 )
		{
			++written;
			if( out != nullptr )
			{
				if( written+1 >= size ) break;
				*out++ = static_cast<u8>((data >> 8) & 0xFF);
			}
			data <<= 8;
			offset &= 0x7;
		}
	}

	return 0;
}

/**********************************************************************//**
 *
 * ToBase
 *
 ----------------------------------------------------------------------
 * @param [in]	c	= 
 * @return	
*//***********************************************************************/
template<typename CHARTYPE_, eBASE_ENCODING_TYPE TYPE>
CHARTYPE_ CTBaseEncoding<CHARTYPE_, TYPE>::ToBase(u8 c)
{
	return CEncoderImpl<CHARTYPE_, TYPE>::Encode(c);
}

/**********************************************************************//**
 *
 * ToPlane
 *
 ----------------------------------------------------------------------
 * @param [in]	c	= 
 * @return	
*//***********************************************************************/
template<typename CHARTYPE_, eBASE_ENCODING_TYPE TYPE>
u8 CTBaseEncoding<CHARTYPE_, TYPE>::ToPlane(_Mychar c)
{
	return CEncoderImpl<CHARTYPE_, TYPE>::Decode(c);
}

// `
template class CTBaseEncoding< CHAR, eBASE16>;
template class CTBaseEncoding<WCHAR, eBASE16>;
template class CTBaseEncoding< CHAR, eBASE32>;
template class CTBaseEncoding<WCHAR, eBASE32>;
template class CTBaseEncoding< CHAR, eBASE32_HEX>;
template class CTBaseEncoding<WCHAR, eBASE32_HEX>;
template class CTBaseEncoding< CHAR, eBASE64>;
template class CTBaseEncoding<WCHAR, eBASE64>;

}	// end of namespace fnd
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../../unit/UnitCore.h"
#include "iris_using.h"
#include "../../iris_iostream.h"
#include <stdio.h>
#include <tchar.h>

static const int BUF_LEN		= 256;
static const int BUF_BASELEN	= (BUF_LEN)*2+1;

//======================================================================
// test
IRIS_UNITTEST(CFndBaseEncodingUnitTest, Base16)
{
	TCHAR buf[BUF_LEN] = TEXT("test");
	TCHAR base[BUF_BASELEN];
	TCHAR out[BUF_LEN] = TEXT("test");

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "ϊ镶͂ĂB" << std::endl;
	std::tcin >> buf;
#else
#endif
	{
		CBaseEncoding<eBASE16>::Encode(base, BUF_BASELEN, (u8*)buf, (size_t)(_tcslen(buf)+1)*sizeof(TCHAR));

		_tprintf(base);
		_tprintf(IRIS_TEXT("\n"));

		CBaseEncoding<eBASE16>::Decode((u8*)out, BUF_LEN*sizeof(TCHAR), base, BUF_BASELEN);

		_tprintf(out);
		_tprintf(IRIS_TEXT("\n"));

		IRIS_ASSERT( _tcsequ(buf, out) );
	}
}

IRIS_UNITTEST(CFndBaseEncodingUnitTest, Base32)
{
	TCHAR buf[BUF_LEN] = TEXT("test");
	TCHAR base[BUF_BASELEN];
	TCHAR out[BUF_LEN] = TEXT("test");

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "ϊ镶͂ĂB" << std::endl;
	std::tcin >> buf;
#else
#endif
	{
		CBaseEncoding<eBASE32>::Encode(base, BUF_BASELEN, (u8*)buf, (size_t)(_tcslen(buf)+1)*sizeof(TCHAR));

		_tprintf(base);
		_tprintf(IRIS_TEXT("\n"));

		CBaseEncoding<eBASE32>::Decode((u8*)out, BUF_LEN*sizeof(TCHAR), base, BUF_BASELEN);

		_tprintf(out);
		_tprintf(IRIS_TEXT("\n"));

		IRIS_ASSERT( _tcsequ(buf, out) );
	}
}

IRIS_UNITTEST(CFndBaseEncodingUnitTest, Base32Hex)
{
	TCHAR buf[BUF_LEN] = TEXT("test");
	TCHAR base[BUF_BASELEN];
	TCHAR out[BUF_LEN] = TEXT("test");

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "ϊ镶͂ĂB" << std::endl;
	std::tcin >> buf;
#else
#endif
	{
		CBaseEncoding<eBASE32_HEX>::Encode(base, BUF_BASELEN, (u8*)buf, (size_t)(_tcslen(buf)+1)*sizeof(TCHAR));

		_tprintf(base);
		_tprintf(IRIS_TEXT("\n"));

		CBaseEncoding<eBASE32_HEX>::Decode((u8*)out, BUF_LEN*sizeof(TCHAR), base, BUF_BASELEN);

		_tprintf(out);
		_tprintf(IRIS_TEXT("\n"));

		IRIS_ASSERT( _tcsequ(buf, out) );
	}
}

IRIS_UNITTEST(CFndBaseEncodingUnitTest, Base64)
{
	TCHAR buf[BUF_LEN] = TEXT("test");
	TCHAR base[BUF_BASELEN];
	TCHAR out[BUF_LEN] = TEXT("test");

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "ϊ镶͂ĂB" << std::endl;
	std::tcin >> buf;
#else
#endif
	{
		CBaseEncoding<eBASE64>::Encode(base, BUF_BASELEN, (u8*)buf, (size_t)(_tcslen(buf)+1)*sizeof(TCHAR));

		_tprintf(base);
		_tprintf(IRIS_TEXT("\n"));

		CBaseEncoding<eBASE64>::Decode((u8*)out, BUF_LEN*sizeof(TCHAR), base, BUF_BASELEN);

		_tprintf(out);
		_tprintf(IRIS_TEXT("\n"));

		IRIS_ASSERT( _tcsequ(buf, out) );
	}
}

#endif	// #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
