//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		stm32_bootloader_unittest.c
 * @brief		stm32 bootloader P̃eXg t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_stm32_bootloader_unittest_C_

//======================================================================
// include
#include "platform/embedded/stm/stm32f10x/host/STMBootloader.h"

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "unit/UnitCore.h"
#include "platform/windows/win/io/WXOverlapped.h"
#include "platform/windows/win/io/WXComPort.h"
#include "platform/windows/win/base/WXError.h"
#include "iris_using.h"
#include "iris_debug.h"
#include "iris_iostream.h"
#include <stdio.h>

//======================================================================
// declare
int go_test(void);
int rd_test(void);
int wd_test(void);
int bn_test(void);
int er_test(void);
int bk_test(void);

//======================================================================
// class
class CStm32bootloaderUnitTest : public iris::unit::CUnitTest<CStm32bootloaderUnitTest>
{
public:
	CComPort m_port;
public:
	virtual void Abort(void)
	{
		m_port.Close();
	}
};

//======================================================================
// test
IRIS_UNITTEST_FIX(CStm32bootloaderUnitTest, Stm32bootloaderUnitTest)
{
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;
	// COM1 J
	if( !port.Open(1) ) 
	{
		CLastError es;
		IRIS_WARNING(es.ToStringA());
		return;
	}
	DCB dcb = {0};
	dcb.DCBlength = sizeof(DCB);
	dcb.BaudRate = CBR_115200;
	dcb.Parity = EVENPARITY;
	dcb.ByteSize = 8;
	dcb.StopBits = ONESTOPBIT;
	if( !port.SetState(&dcb) ) return;

	if( !stm32_bootloader_connect(port) ) return;

	puts("connect bootloader.");

	if( !stm32_bootloader_get(port) ) return;

	int n;
	if( !stm32_bootloader_get_id_size(port, &n) ) return;

	char* pid = new char [n];
	if( !stm32_bootloader_get_id(port, pid, n) ) return;
	printf("ID = ");
	for( int i=0; i < n; ++i )
		printf("%.2x ", pid[i]);
	puts("");
	delete [] pid;

	int ver;
	if( !stm32_bootloader_get_ver(port, &ver, nullptr) ) return;
	printf("bootloader version = 0x%.2x\n", ver);

	while(1)
	{
#define TEST_CMD(_CMD)	\
	do { if( strcmp(cmd, #_CMD) == 0 )	{ ret = _CMD##_test(); continue; } } while(0)

		char cmd[256];

		std::cout << "" << std::endl;
		std::cout << "COMMAND" << std::endl;
		std::cout << "go xxxx(h)                   : test go command" << std::endl;
		std::cout << "rd xxxx(h) yyyy(h)           : test read_memory command" << std::endl;
		std::cout << "wd xxxx(h) yyyy(h) dddd(h)   : test write_memory command" << std::endl;
		std::cout << "er xxxx(h) (yyyy(h))         : test erase_memory command " << std::endl;
		std::cout << "bn path xxxx(h)              : write binary file " << std::endl;
		std::cout << "bk                    : break" << std::endl;
		std::cout << "q                     : exit" << std::endl;
		std::cout << "" << std::endl;

		std::safe_cin >> cmd;
		int ret = 0;
		TEST_CMD(go);
		TEST_CMD(rd);
		TEST_CMD(wd);
		TEST_CMD(er);
		TEST_CMD(bn);
		TEST_CMD(bk);
		if( strcmp(cmd, "q") == 0 ) break;
		if( ret > 0 )
		{
			std::cout << "OK!!" << std::endl;
		}
	}

#if 0
	int addr;
	if( !stm32_bootloader_read_memory(port, &addr, 0x8003000, 4) ) return;
	printf("0x08003000 : %.8x\n", addr);

#if 0
	{
		if( !stm32_bootloader_read_memory(port, &addr, 0x8003000, 4) ) return;
		printf("0x08003000 : %.8x\n", addr);

		// 0x8003000 y[W폜
		if( !stm32_bootloader_erase_memory_page(port, 12) ) return;

		if( !stm32_bootloader_read_memory(port, &addr, 0x8003000, 4) ) return 1;
		printf("0x08003000 : %.8x\n", addr);

		int src = 0xCDCDABAB;
		if( !stm32_bootloader_write_memory(port, &src, 0x8003000, 4) ) return 1;

		if( !stm32_bootloader_read_memory(port, &addr, 0x8003000, 4) ) return 1;
		printf("0x08003000 : %.8x\n", addr);
	}

	{
		if( !stm32_bootloader_read_memory(port, &addr, 0x8000000, 4) ) return 1;
		printf("0x08000000 : %.8x\n", addr);
		// Sy[W폜
		if( !stm32_bootloader_erase_memory_all(port) ) return 1;

		if( !stm32_bootloader_read_memory(port, &addr, 0x8000000, 4) ) return 1;
		printf("0x08000000 : %.8x\n", addr);

		if( !stm32_bootloader_go(port, 0x08003000) ) return;
	}
#endif
#endif
}


// seXg
int go_test(void)
{
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;

	u32 address = 0;
	std::cout << "sAhX͂ĂB(h)" << std::endl;
	std::safe_cin >> std::hex >> address;

	return stm32_bootloader_go(port, address);
}

// read eXg
int rd_test(void)
{
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;

	u32 address = 0;
	u32 size = 0;
	void* buf = nullptr;
	std::cout << "ǂݎJnAhX͂ĂB(h)" << std::endl;
	std::safe_cin >> std::hex >> address;
	std::cout << "ǂݎTCY͂ĂB(h)" << std::endl;
	std::safe_cin >> std::hex >> size;

	buf = new char [size];

	if( !stm32_bootloader_read_memory(port, buf, address, size) )
	{
		std::cout << "read memory failed." << std::endl;
		return 0;
	}

	u32* p = (u32*)buf;
	for( u32 i=0; i < ((size+15)/16); ++i )
	{
		std::cout << "0x" << std::hex << address+i*4*4 << " : ";
		for( u32 j=0; j < 4 && i*16+j*4 < size; ++j, ++p )
		{
			std::cout << "0x" << std::setw(8) << std::setfill('0') << *p << " ";
		}
		std::cout << std::endl;
	}
	return 1;
}

// write eXg
int wd_test(void)
{
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;
	u32 address = 0;
	u32 size = 0;
	u32 data = 0;
	u32 read = 0;
	std::cout << "݃AhX͂ĂB(h)" << std::endl;
	std::safe_cin >> std::hex >> address;
	do
	{
		std::cout << "݃TCY͂ĂB[1, 2, 4]" << std::endl;
		std::safe_cin >> std::dec >> size;
	} while( size != 1 && size != 2 && size != 4);
	std::cout << "ޒl͂ĂB" << std::endl;
	std::safe_cin >> std::hex >> data;

	if( !stm32_bootloader_write_memory(port, &data, address, size) )
	{
		std::cout << "write memory failed." << std::endl;
		return 0;
	}

	if( !stm32_bootloader_read_memory(port, &read, address, size) )
	{
		std::cout << "read memory failed." << std::endl;
		return 0;
	}
	if( data != read )
	{
		std::cout << "verify error. read=" << std::hex << std::setw(size*2) << std::setfill('0') << read << std::endl;
		return 0;
	}
	return 1;
}

// write binary file eXg
int bn_test(void)
{
	int ret = 0;
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;
	u32 address = 0;
	char path[MAX_PATH];

	std::cout << "ރt@CpX͂ĂB(h)" << std::endl;
	std::safe_cin >> path;

	std::cout << "݃AhX͂ĂB(h)" << std::endl;
	std::safe_cin >> std::hex >> address;

	CHFile file;
	if( !file.OpenA(path, "rb") )
	{
		std::cerr << "file open failed. " << path << std::endl;
		return 0;
	}

	DWORD size = file.GetSize();
	std::clog << "file size = 0x" << std::hex << size << std::endl;
	u8* data = new u8 [size];
	DWORD read;
	if( !file.Read(data, size, &read) )
		goto exit;

	int page = (address - 0x08000000) / 0x0400;
	for( int i=0, pn=(size+0x3FF)/0x0400; i < pn; ++i )
	{
		if( !stm32_bootloader_erase_memory_page(port, page+i) )
		{
			std::cerr << "erase command failed. page = " << std::dec << page+i << std::endl;
			goto exit;
		}
	}

	if( !stm32_bootloader_write_memory(port, data, address, size) )
	{
		std::cerr << "write command failed." << std::endl;
		goto exit;
	}

	if( !stm32_bootloader_verify(port, data, address, size) )
	{
		std::cerr << "verify error." << std::endl;
		goto exit;
	}

	ret = 1;
exit:
	delete [] data;
	return ret;
}

// erase eXg
int er_test(void)
{
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;
	u32 address = 0;
	u32 size = 0;
	std::cout << "폜TCY͂ĂB(h) 0 = all erase" << std::endl;
	std::safe_cin >> std::hex >> size;

	if( size == 0 )
	{
		if( !stm32_bootloader_erase_memory_all(port) )
			return 0;
	}
	else
	{
		std::cout << "폜JnAhX͂ĂB(h)" << std::endl;
		std::safe_cin >> std::hex >> address;

		int page = (address-0x08000000)/0x0400;
		for( int i=0, pn=(size+0x3FF)/0x0400; i < pn; ++i )
		{
			if( !stm32_bootloader_erase_memory_page(port, page+i) )
				return 0;
		}
	}

	return 1;
}

// break eXg
int bk_test(void)
{
	CComPort& port = CStm32bootloaderUnitTest::GetCurrent()->m_port;
	return stm32_bootloader_break(port);
}

#endif
