//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathXFVector2.inl
 * @brief		2DxNgt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathXFVector2_inl_
#define INCG_IRIS_MathXFVector2_inl_

namespace iris {
namespace xf
{

//======================================================================
// function
/**
 * @brief	vfݒ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	x	= xl
 * @param [in]	y	= yl
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Set(IrisXFVec2* pv0, xf32 x, xf32 y)
{
	XFPU_NULLASSERT( pv0 );
	pv0->x = x; pv0->y = y;
	return pv0;
}

/**
 * @brief	vfRs[
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Copy(IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = pv1->x; pv0->y = pv1->y;
	return pv0;
}

/**
 * @brief	+0.0fɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2PositiveZero(IrisXFVec2* pv0)
{
	XFPU_NULLASSERT( pv0 );
	pv0->x = pv0->y = XF32_ZERO;
	return pv0;
}

/**
 * @brief	-0.0fɏ
 * @param [out]	pv0	= o̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2NegativeZero(IrisXFVec2* pv0)
{
	XFPU_NULLASSERT( pv0 );
	IrisVec2* v = (IrisVec2*)pv0;
	v->iv.x = v->iv.y = 0x80000000;
	return pv0;
}

/**
 * @brief	؂グ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisIVec2*		XFpuVec2Ceil(IrisIVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = XF_XF32_TO_S32( XF32_Ceil(pv1->x) );
	pv0->y = XF_XF32_TO_S32( XF32_Ceil(pv1->y) );
	return pv0;
}

/**
 * @brief	0ۂ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisIVec2*		XFpuVec2Trunc(IrisIVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = XF_XF32_TO_S32( XF32_Trunc(pv1->x) );
	pv0->y = XF_XF32_TO_S32( XF32_Trunc(pv1->y) );
	return pv0;
}

/**
 * @brief	ߖTۂ(ľܓ)
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisIVec2*		XFpuVec2Round(IrisIVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = XF_XF32_TO_S32( XF32_Round(pv1->x) );
	pv0->y = XF_XF32_TO_S32( XF32_Round(pv1->y) );
	return pv0;
}

/**
 * @brief	؂̂
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisIVec2*		XFpuVec2Floor(IrisIVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = XF_XF32_TO_S32( XF32_Floor(pv1->x) );
	pv0->y = XF_XF32_TO_S32( XF32_Floor(pv1->y) );
	return pv0;
}

/**
 * @brief	xNgϊ
 * @param [out]	pv0	= xNg
 * @param [in]	pv1	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2FromIVec2(IrisXFVec2* pv0, const IrisIVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = XF_S32_TO_XF32( pv1->x );
	pv0->y = XF_S32_TO_XF32( pv1->y );
	return pv0;
}

/**
 * @brief	xNg̉Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Add(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + pv2->x;
	pv0->y = pv1->y + pv2->y;
	return pv0;
}

/**
 * @brief	xNǧZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 팸xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Sub(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	pv0->x = pv1->x - pv2->x;
	pv0->y = pv1->y - pv2->y;
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= 搔xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Mul(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	pv0->x = XF32_Mul(pv1->x, pv2->x);
	pv0->y = XF32_Mul(pv1->y, pv2->y);
	return pv0;
}

/**
 * @brief	xNg̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 폜xNg
 * @param [in]	pv2	= xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Div(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	pv0->x = XF32_Div(pv1->x, pv2->x);
	pv0->y = XF32_Div(pv1->y, pv2->y);
	return pv0;
}

/**
 * @brief	xNg̊evf̕]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Neg(IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
#if 0
	pv0->x = -pv1->x;
	pv0->y = -pv1->y;
#else
	IrisVec2* v0 = (IrisVec2*)pv0;
	const IrisVec2* v1 = (const IrisVec2*)pv1;
	v0->iv.x = (s32)(v1->iv.x ^ 0x80000000);
	v0->iv.y = (s32)(v1->iv.y ^ 0x80000000);
#endif
	return pv0;
}

/**
 * @brief	xNg̊evf̐Βl擾
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Abs(IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
#if 0
	pv0->x = XF32_ABS(pv1->x);
	pv0->y = XF32_ABS(pv1->y);
#else
	IrisVec2* v0 = (IrisVec2*)pv0;
	const IrisVec2* v1 = (const IrisVec2*)pv1;
	v0->iv.x = v1->iv.x & 0x7FFFFFFF;
	v0->iv.y = v1->iv.y & 0x7FFFFFFF;
#endif
	return pv0;
}

/**
 * @brief	xNg̊Ԃ̓}
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Lerp(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2, xf32 t)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	pv0->x = pv1->x + XF32_Mul(t, (pv2->x - pv1->x));
	pv0->y = pv1->y + XF32_Mul(t, (pv2->y - pv1->y));
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Scale(IrisXFVec2* pv0, const IrisXFVec2* pv1, xf32 s)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = XF32_Mul(pv1->x, s);
	pv0->y = XF32_Mul(pv1->y, s);
	return pv0;
}

/**
 * @brief	xNg̃XJ[l̏Z
 *				pv0 = pv1 * s + vp2
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= 搔xNg
 * @param [in]	pv2	= ZxNg
 * @param [in]	s	= XJ[l
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2ScaleAdd(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2, xf32 s)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	pv0->x = XF32_Mul(pv1->x, s) + pv2->x;
	pv0->y = XF32_Mul(pv1->y, s) + pv2->y;
	return pv0;
}

/**
 * @brief	xNg̃G~[gXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pt1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pt2	= ̓xNg
 * @param [in]	t	= XJ[l
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Hermite(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pt1
									  , const IrisXFVec2* pv2, const IrisXFVec2* pt2, xf32 t)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pt1 );
	XFPU_NULLASSERT( pv2 );
	XFPU_NULLASSERT( pt2 );
	xf32 t2 = XF32_Mul(t, t);
	xf32 t3 = XF32_Mul(t, t2);
	IrisXFVec2 v[4];
	XFpuVec2Scale(&v[0], pv1, (( 2 * t3) - (3 * t2) + XF32_ONE));
	XFpuVec2Scale(&v[1], pt1, ((     t3) - (2 * t2) + t       ));
	XFpuVec2Scale(&v[3], pv2, ((-2 * t3) + (3 * t2)           ));
	XFpuVec2Scale(&v[2], pt2, ((     t3) - (    t2)           ));
	XFpuVec2Add(&v[0], &v[0], &v[1]);		// v[0] = v[0] + v[1]
	XFpuVec2Add(&v[0], &v[0], &v[2]);		// v[0] = v[0] + v[2]
	XFpuVec2Add(pv0,   &v[0], &v[3]);		// *pv0 = v[0] + v[3]
	return pv0;
}

/**
 * @brief	xNg̃xWGXvC
 * @param [out]	pv0	= o̓xNg
 * @param [in]	n	= xNgz
 * @param [in]	pva	= xNgz
 * @param [in]	t	= 
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Bezier(IrisXFVec2* pv0, s32 n, const IrisXFVec2* pva, xf32 t)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pva );
	IRIS_ASSERT( n >= 2 );
	if( t <= XF32_ZERO ) { XFpuVec2Copy(pv0, &pva[0]); return pv0; }
	if( t >= XF32_ONE  ) { XFpuVec2Copy(pv0, &pva[n-1]); return pv0; }
	XFpuVec2PositiveZero(pv0);
	xf32 tt = XF32_ONE;			// t^ip
	xf32 it = XF32_ONE - t;		// (1-t)^ip
	s32 nn = n-1;
	const IrisXFVec2* pv = pva;
	for( s32 i=0; i < n; ++i, ++pv )
	{
		xf32 J = ::iris::math::Combination(nn, i) * XF32_Mul(tt, XF_Pow(it, nn-i));
		pv0->x += XF32_Mul(pv->x, J);
		pv0->y += XF32_Mul(pv->y, J);
		tt = XF32_Mul(tt, t);
	}
	return pv0;
}

/**
 * @brief	xNg̃Nv
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	min	= ŏl
 * @param [in]	max	= ől
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Clamp(IrisXFVec2* pv0, const IrisXFVec2* pv1, xf32 min, xf32 max)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < min) ? min : ((pv1->x > max) ? max : pv1->x);
	pv0->y = (pv1->y < min) ? min : ((pv1->y > max) ? max : pv1->y);
	return pv0;
}

/**
 * @brief	xNg̊evf̑傫Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Max(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x > pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y > pv2->y) ? pv1->y : pv2->y;
	return pv0;
}

/**
 * @brief	xNg̊evf̏Ԃ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= rxNg
 * @param [in]	pv2	= rxNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Min(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < pv2->x) ? pv1->x : pv2->x;
	pv0->y = (pv1->y < pv2->y) ? pv1->y : pv2->y;
	return pv0;
}

/**
 * @brief	
 * @param [in]	pv0	= 
 * @param [in]	pv1	= 
 * @return	ϒl
*/
IRIS_XFPU_INLINE xf32			XFpuVec2InnerProduct(const IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	return XF32_Mul(pv0->x, pv1->x) + XF32_Mul(pv0->y, pv1->y);
}

/**
 * @brief	O
 * @param [in]	pv0	= 
 * @param [in]	pv1	= 
 * @return	ϒl
*/
IRIS_XFPU_INLINE xf32			XFpuVec2OuterProduct(const IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	return XF32_Mul(pv0->x, pv1->y) - XF32_Mul(pv0->y, pv1->x);
}

/**
 * @brief	xNg̊evf̑a擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̑a
*/
IRIS_XFPU_INLINE xf32			XFpuVec2Funnel(const IrisXFVec2* pv0)
{
	XFPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y);
}

/**
 * @brief	xNg̊evf̕ς擾
 * @param [in]	pv0	= ̓xNg
 * @return	evf̕
*/
IRIS_XFPU_INLINE xf32			XFpuVec2Average(const IrisXFVec2* pv0)
{
	XFPU_NULLASSERT( pv0 );
	return (pv0->x + pv0->y) / 2;
}

/**
 * @brief	xNgǂԂ
 * @param [in]	pv0	= rxNg
 * @param [in]	pv1	= rxNg
 * @return	^Ul
*/
IRIS_XFPU_INLINE IrisBool		XFpuVec2IsEqual(const IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	if( (pv0->x != pv1->x) || (pv0->y != pv1->y) ) return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	[xNgǂԂ
 * @param [in]	pv0	= ̓xNg
 * @return	^Ul
*/
IRIS_XFPU_INLINE IrisBool		XFpuVec2IsZero(const IrisXFVec2* pv0)
{
	XFPU_NULLASSERT( pv0 );
	const IrisVec2* v = (const IrisVec2*)pv0;
	if( ((v->iv.x | v->iv.y) & 0x7FFFFFFFU) == 0 )
		return IRIS_TRUE;
	return IRIS_FALSE;
}

/**
 * @brief	,̕IrisXFVec2ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2SignFloat(IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < XF32_ZERO) ? -XF32_ONE : (pv1->x > XF32_ZERO) ? XF32_ONE : XF32_ZERO;
	pv0->y = (pv1->y < XF32_ZERO) ? -XF32_ONE : (pv1->y > XF32_ZERO) ? XF32_ONE : XF32_ZERO;
	return (pv0);
}

/**
 * @brief	,̕IrisIVec2ŕԂ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisIVec2*		XFpuVec2SignInt(IrisIVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	pv0->x = (pv1->x < XF32_ZERO) ? -1 : (pv1->x > XF32_ZERO) ? 1 : 0;
	pv0->y = (pv1->y < XF32_ZERO) ? -1 : (pv1->y > XF32_ZERO) ? 1 : 0;
	return (pv0);
}

/**
 * @brief	xNg̐K
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Normalize(IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	xf32 x = pv1->x, y = pv1->y;
	xf32 q = XF_Sqrt( XF32_Mul(x, x) + XF32_Mul(y, y) );
#if IRIS_XFPU_ASSERT_ZERODIV
	IRIS_ASSERT( q != XF32_ZERO );
	q = XF32_Div(XF32_ONE, q);
#else
	if( q != XF32_ZERO ) q = XF32_Div(XF32_ONE, q);
#endif
	pv0->x = XF32_Mul(x, q);
	pv0->y = XF32_Mul(y, q);
	return pv0;
}

/**
 * @brief	xNg̒Ԃ
 * @param [in]	pv0	= ̓xNg
 * @return	xNg̒
*/
IRIS_XFPU_INLINE xf32			XFpuVec2Length(const IrisXFVec2* pv0)
{
	XFPU_NULLASSERT( pv0 );
	return (XF_Sqrt( XF32_Mul(pv0->x, pv0->x) + XF32_Mul(pv0->y, pv0->y) ));
}

/**
 * @brief	xNg̊Ԃ̋
 * @param [in]	pv0	= ̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	xNg̒
*/
IRIS_XFPU_INLINE xf32			XFpuVec2Distance(const IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	xf32 x = pv0->x - pv1->x;
	xf32 y = pv0->y - pv1->y;
	return (XF_Sqrt( XF32_Mul(x, x) + XF32_Mul(y, y) ));
}

/**
 * @brief	IuWFNg\ʂJɌ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	pv3	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2FaceForward(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2, const IrisXFVec2* pv3)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	XFPU_NULLASSERT( pv3 );
	xf32 d = XF32_Mul(pv2->x, pv3->x) + XF32_Mul(pv2->y, pv3->y);
	if(d < XF32_ZERO)
	{
		pv0->x = pv1->x;
		pv0->y = pv1->y;
	} 
	else
	{
		pv0->x = -pv1->x;
		pv0->y = -pv1->y;
	}
	return pv0;
}

/**
 * @brief	˃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Reflect(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	xf32 d = XF32_Mul(pv1->x, pv2->x) + XF32_Mul(pv1->y, pv2->y);
	pv0->x = pv1->x - XF32_Mul(2 * d, pv2->x);
	pv0->y = pv1->y - XF32_Mul(2 * d, pv2->y);
	return pv0;
}

/**
 * @brief	܃xNg𐶐
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @param [in]	pv2	= ̓xNg
 * @param [in]	eta	= ܗ
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2Refract(IrisXFVec2* pv0, const IrisXFVec2* pv1, const IrisXFVec2* pv2, xf32 eta)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	XFPU_NULLASSERT( pv2 );
	xf32 d, e, f;
	d = XF32_Mul(pv1->x, pv2->x) + XF32_Mul(pv1->y, pv2->y);
	e = XF32_ONE - XF32_Mul(XF32_Mul(eta, eta), (XF32_ONE - XF32_Mul(d, d)));
	if(e < XF32_ZERO)
	{
		pv0->x = XF32_ZERO;
		pv0->y = XF32_ZERO;
	} 
	else
	{
		f = XF32_Mul(eta, d) - XF_Sqrt(e);
		pv0->x = XF32_Mul(eta, pv1->x) - XF32_Mul(f, pv2->x);
		pv0->y = XF32_Mul(eta, pv1->y) - XF32_Mul(f, pv2->y);
	}
	return pv0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec2*	XFpuVec2TruncatePrecision24(IrisXFVec2* pv0, const IrisXFVec2* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	IrisVec2* v0 = (IrisVec2*)pv0;
	const IrisVec2* v1 = (const IrisVec2*)pv1;
	v0->iv.x = (s32)(v1->iv.x & 0xFFFFFF00);
	v0->iv.y = (s32)(v1->iv.y & 0xFFFFFF00);
	return pv0;
}


}	// end of namespace xf
}	// end of namespace iris

#endif
