//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathXFQuaternion.inl
 * @brief		NH[^jI֐t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathXFQuaternion_inl_
#define INCG_IRIS_MathXFQuaternion_inl_

namespace iris {
namespace xf
{

//======================================================================
// function
/**
 * @brief	PʃNH[^jI𐶐
 * @param [out]	pq0	= o̓NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionUnit(IrisXFQuaternion* pq0)
{
	XFPU_NULLASSERT( pq0 );
	pq0->x = pq0->y = pq0->z = XF32_ZERO;
	pq0->w = XF32_ONE;
	return pq0;
}

/**
 * @brief	NH[^jIRs[
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= Rs[NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionCopy(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	memcpy(pq0, pq1, sizeof(IrisXFQuaternion));
	return pq0;
}

/**
 * @brief	NH[^jI̘avZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionAdd(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1, const IrisXFQuaternion* pq2)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	XFPU_NULLASSERT( pq2 );
	pq0->x = pq1->x + pq2->x;
	pq0->y = pq1->y + pq2->y;
	pq0->z = pq1->z + pq2->x;
	pq0->w = pq1->w + pq2->w;
	return pq0;
}

/**
 * @brief	NH[^jI̍vZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= 팸NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @return o̓xNg
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionSub(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1, const IrisXFQuaternion* pq2)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	XFPU_NULLASSERT( pq2 );
	pq0->x = pq1->x - pq2->x;
	pq0->y = pq1->y - pq2->y;
	pq0->z = pq1->z - pq2->x;
	pq0->w = pq1->w - pq2->w;
	return pq0;
}

/**
 * @brief	NH[^jI̐ςvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= 搔NH[^jI
 * @param [in]	pq2	= 搔NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionMul(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1, const IrisXFQuaternion* pq2)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	XFPU_NULLASSERT( pq2 );
	xf32 x1 = pq1->x, y1 = pq1->y, z1 = pq1->z, w1 = pq1->w;
	xf32 x2 = pq2->x, y2 = pq2->y, z2 = pq2->z, w2 = pq2->w;
	pq0->x = XF32_Mul(w1, x2) + XF32_Mul(w2, x1) + XF32_Mul(y1, z2) - XF32_Mul(z1, y2);
	pq0->y = XF32_Mul(w1, y2) + XF32_Mul(w2, y1) + XF32_Mul(z1, x2) - XF32_Mul(x1, z2);
	pq0->z = XF32_Mul(w1, z2) + XF32_Mul(w2, z1) + XF32_Mul(x1, y2) - XF32_Mul(y1, x2);
	pq0->w = XF32_Mul(w1, w2) - XF32_Mul(x1, x2) - XF32_Mul(y1, y2) - XF32_Mul(z1, z2);
	return pq0;
}

/**
 * @brief	NH[^jI̓ςvZ
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return 
*/
IRIS_XFPU_INLINE	xf32				XFpuQuaternionInnerProduct(const IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	return ( XF32_Mul(pq0->x, pq1->x) + XF32_Mul(pq0->y, pq1->y) + XF32_Mul(pq0->z, pq0->z) + XF32_Mul(pq0->w, pq0->w) );
}

/**
 * @brief	NH[^jI̋vZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionConj(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	pq0->x = -pq1->x;
	pq0->y = -pq1->y;
	pq0->z = -pq1->z;
	pq0->w =  pq1->w;
	return pq0;
}

/**
 * @brief	NH[^jIgăxNg]
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= NH[^jI
 * @param [in]	pv1	= xNg
 * @return o̓xNg
*/
IRIS_XFPU_INLINE	IrisXFVec4*			XFpuQuaternionTransform(IrisXFVec4* pv0, const IrisXFQuaternion* pq0, const IrisXFVec4* pv1)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pv1 );
	IrisXFQuaternion q1, q2;
	XFpuQuaternionConj(&q2, pq0);
	XFpuQuaternionMul(&q1, pq0, (const IrisXFQuaternion*)pv1);
	XFpuQuaternionMul((IrisXFQuaternion*)pv0, &q1, &q2);
	return pv0;
}

/**
 * @brief	NH[^jI̋ʐ`⊮
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @param [in]	t	= 
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionSlerp(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1, const IrisXFQuaternion* pq2, xf32 t)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	XFPU_NULLASSERT( pq2 );
	xf32 c = XF32_Mul(pq1->x, pq2->x) + XF32_Mul(pq1->y, pq2->y) + XF32_Mul(pq1->z, pq2->z) + XF32_Mul(pq1->w, pq2->w);
	xf32 angle;
	xf32 s;
	xf32 r1, r2;

	int reverse = 0;
	if( c < 0 ) { c = - c; reverse = 1; }

	angle = XF_Acos( c < -XF32_ONE ? -XF32_ONE : ( c > XF32_ONE ? XF32_ONE : c ) );
	s = XF_Sin( angle );
	if( s < XF32_CONST(0.00005f) ) 
	{
		memcpy(pq0, pq2, sizeof(IrisXFQuaternion));
		return (pq0);
	}
	r1 = XF32_Div( XF_Sin( XF32_Mul(( XF32_ONE - t ), angle) ), s);
	r2 = XF32_Div( XF_Sin( XF32_Mul(t, angle) ), s);
	if( reverse ) r2 = - r2;

	//@@
	//  : ٓ_ߕӂł̐xmۂɂ quat_normalizeg
	pq0->x = XF32_Mul(pq1->x, r1) + XF32_Mul(pq2->x, r2);
	pq0->y = XF32_Mul(pq1->y, r1) + XF32_Mul(pq2->y, r2);
	pq0->z = XF32_Mul(pq1->z, r1) + XF32_Mul(pq2->z, r2);
	pq0->w = XF32_Mul(pq1->w, r1) + XF32_Mul(pq2->w, r2);
	return pq0;
}

/**
 * @brief	NH[^jI̋3⊮
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @param [in]	pq2	= NH[^jI
 * @param [in]	pq3	= NH[^jI
 * @param [in]	pq4	= NH[^jI
 * @param [in]	t	= 
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionSquad(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1, const IrisXFQuaternion* pq2
													   , const IrisXFQuaternion* pq3, const IrisXFQuaternion* pq4, xf32 t)
{
	IrisXFQuaternion qa, qb;
	XFpuQuaternionSlerp(&qa, pq1, pq2, t);
	XFpuQuaternionSlerp(&qb, pq3, pq4, t);
	XFpuQuaternionSlerp(pq0, &qa, &qb, XF32_Mul(2*t, (XF32_ONE-t)));
	return pq0;
}

/**
 * @brief	NH[^jI̐K
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionNormalize(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	xf32 x = pq1->x, y = pq1->y, z = pq1->z, w = pq1->w;
	xf32 q = XF32_Div(XF32_ONE, XF_Sqrt( XF32_Mul(x, x) + XF32_Mul(y, y) + XF32_Mul(z, z) + XF32_Mul(w, w) ));
	pq0->x = XF32_Mul(x, q);
	pq0->y = XF32_Mul(y, q);
	pq0->z = XF32_Mul(z, q);
	pq0->w = XF32_Mul(w, q);
	return pq0;
}

/**
 * @brief	tNH[^jǏvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionInverse(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	xf32 n = XF32_Mul(pq1->x, pq1->x) + XF32_Mul(pq1->y, pq1->y) + XF32_Mul(pq1->z, pq1->z) + XF32_Mul(pq1->w, pq1->w);
	pq0->x = XF32_Div(-pq1->x, n);
	pq0->y = XF32_Div(-pq1->y, n);
	pq0->z = XF32_Div(-pq1->z, n);
	pq0->w = XF32_Div( pq1->w, n);
	return pq0;
}

/**
 * @brief	NH[^jI̎RΐvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionLn(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	xf32 n, theta, m;
	n = XF_Sqrt( XF32_Mul(pq1->x, pq1->x) + XF32_Mul(pq1->y, pq1->y) + XF32_Mul(pq1->z, pq1->z) );
	if(n > XF32_ZERO)
	{
		theta = XF_Acos( XF32_Div(pq1->w, n) );
		m     = XF32_Div(theta, n);
		pq0->x = XF32_Mul(pq1->x, m);
		pq0->y = XF32_Mul(pq1->y, m);
		pq0->z = XF32_Mul(pq1->z, m);
	} 
	else
	{
		pq0->x = pq1->x;
		pq0->y = pq1->y;
		pq0->z = pq1->z;
	}
	pq0->w = XF32_ZERO;
	return pq0;
}

/**
 * @brief	NH[^jI̎wvZ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pq1	= NH[^jI
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionExp(IrisXFQuaternion* pq0, const IrisXFQuaternion* pq1)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pq1 );
	xf32 theta = XF_Sqrt( XF32_Mul(pq1->x, pq1->x) + XF32_Mul(pq1->y, pq1->y) + XF32_Mul(pq1->z, pq1->z) );
	xf32 s = XF32_Div(XF_Sin(theta), theta);
	pq0->x = XF32_Mul(pq1->x, s);
	pq0->y = XF32_Mul(pq1->y, s);
	pq0->z = XF32_Mul(pq1->z, s);
	pq0->w = XF_Cos(theta);
	return pq0;
}

/**
 * @brief	NH[^jI}gbNXɕϊ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pq0	= NH[^jI
 * @return o̓}gbNX
*/
IRIS_XFPU_INLINE	IrisXFMtx44*		XFpuQuaternionToMatrix(IrisXFMtx44* pm0, const IrisXFQuaternion* pq0)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pq0 );
	xf32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;
	pm0->x.x = XF32_ONE -	XF32_Mul(XF32_Mul(XF32_2, y), y) - XF32_Mul(XF32_Mul(XF32_2, z), z);
	pm0->x.y =				XF32_Mul(XF32_Mul(XF32_2, x), y) + XF32_Mul(XF32_Mul(XF32_2, w), z);
	pm0->x.z =				XF32_Mul(XF32_Mul(XF32_2, z), x) - XF32_Mul(XF32_Mul(XF32_2, w), y);
	pm0->x.w = XF32_ZERO;
	pm0->y.x =				XF32_Mul(XF32_Mul(XF32_2, x), y) - XF32_Mul(XF32_Mul(XF32_2, w), z);
	pm0->y.y = XF32_ONE -	XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, z), z);
	pm0->y.z =				XF32_Mul(XF32_Mul(XF32_2, y), z) + XF32_Mul(XF32_Mul(XF32_2, w), x);
	pm0->y.w = XF32_ZERO;
	pm0->z.x =				XF32_Mul(XF32_Mul(XF32_2, z), x) + XF32_Mul(XF32_Mul(XF32_2, w), y);
	pm0->z.y =				XF32_Mul(XF32_Mul(XF32_2, y), z) - XF32_Mul(XF32_Mul(XF32_2, w), x);
	pm0->z.z = XF32_ONE -	XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, y), y);
	pm0->z.w = XF32_ZERO;
	pm0->w.x = XF32_ZERO;
	pm0->w.y = XF32_ZERO;
	pm0->w.z = XF32_ZERO;
	pm0->w.w = XF32_ONE;
	return pm0;
}

/**
 * @brief	}gbNXNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pm0	= }gbNX
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionFromMatrix(IrisXFQuaternion* pq0, const IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pm0 );
	xf32 x, y, z, w;
	xf32 r, t;
	const IrisXFVec4 *pv = pm0->v;

	t = pv[0].x + pv[1].y + pv[2].z + XF32_ONE;

	if( t > XF32_CONST(0.01f) ) 
	{
		w = XF32_Mul(XF_Sqrt( t ), XF32_HALF);
		r = XF32_Div(XF32_QUATER, w);
		x = XF32_Mul(( pv[1].z - pv[2].y ), r);
		y = XF32_Mul(( pv[2].x - pv[0].z ), r);
		z = XF32_Mul(( pv[0].y - pv[1].x ), r);
	} 
	else if( pv[0].x > pv[1].y )
	{
		if( pv[0].x > pv[2].z ) 
		{
			x = XF32_Mul(XF_Sqrt( XF32_ONE + pv[0].x - pv[1].y - pv[2].z ), XF32_HALF);
			r = XF32_Div(XF32_QUATER, x);
			y = XF32_Mul(( pv[1].x + pv[0].y ), r);
			z = XF32_Mul(( pv[2].x + pv[0].z ), r);
			w = XF32_Mul(( pv[1].z - pv[2].y ), r);
		}
		else
		{
			z = XF32_Mul(XF_Sqrt( XF32_ONE + pv[2].z - pv[0].x - pv[1].y ), XF32_HALF);
			r = XF32_Div(XF32_QUATER, z);
			x = XF32_Mul(( pv[2].x + pv[0].z ), r);
			y = XF32_Mul(( pv[2].y + pv[1].z ), r);
			w = XF32_Mul(( pv[0].y - pv[1].x ), r);
		}
	} 
	else
	{
		if( pv[1].y > pv[2].z )
		{
			y = XF32_Mul(XF_Sqrt( XF32_ONE + pv[1].y - pv[2].z - pv[0].x ), XF32_HALF);
			r = XF32_Div(XF32_QUATER, y);
			x = XF32_Mul(( pv[1].x + pv[0].y ), r);
			z = XF32_Mul(( pv[2].y + pv[1].z ), r);
			w = XF32_Mul(( pv[2].x - pv[0].z ), r);
		} 
		else 
		{
			z = XF32_Mul(XF_Sqrt( XF32_ONE + pv[2].z - pv[0].x - pv[1].y ), XF32_HALF);
			r = XF32_Div(XF32_QUATER, z);
			x = XF32_Mul(( pv[2].x + pv[0].z ), r);
			y = XF32_Mul(( pv[2].y + pv[1].z ), r);
			w = XF32_Mul(( pv[0].y - pv[1].x ), r);
		}
	}
	pq0->x = x;
	pq0->y = y;
	pq0->z = z;
	pq0->w = w;
	return (pq0);
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionFromRotZYX(IrisXFQuaternion* pq0, const IrisXFVec4* pv0)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pv0 );
	xf32 x = XF32_Mul(pv0->x, XF32_HALF);
	xf32 y = XF32_Mul(pv0->y, XF32_HALF);
	xf32 z = XF32_Mul(pv0->z, XF32_HALF);
	xf32 cx = XF_Cos( x );	xf32 sx = XF_Sin( x );
	xf32 cy = XF_Cos( y );	xf32 sy = XF_Sin( y );
	xf32 cz = XF_Cos( z );	xf32 sz = XF_Sin( z );
	pq0->x = XF32_Mul(XF32_Mul(cz, cy), sx) - XF32_Mul(XF32_Mul(cx, sz), sy);
	pq0->y = XF32_Mul(XF32_Mul(cx, cz), sy) + XF32_Mul(XF32_Mul(cy, sz), sx);
	pq0->z = XF32_Mul(XF32_Mul(cx, cy), sz) - XF32_Mul(XF32_Mul(cz, sy), sx);
	pq0->w = XF32_Mul(XF32_Mul(cz, cy), cx) + XF32_Mul(XF32_Mul(sz, sy), sx);
	return pq0;
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionFromRotXYZ(IrisXFQuaternion* pq0, const IrisXFVec4* pv0)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pv0 );
	xf32 x = XF32_Mul(pv0->x, XF32_HALF);
	xf32 y = XF32_Mul(pv0->y, XF32_HALF);
	xf32 z = XF32_Mul(pv0->z, XF32_HALF);
	xf32 cx = XF_Cos( x );	xf32 sx = XF_Sin( x );
	xf32 cy = XF_Cos( y );	xf32 sy = XF_Sin( y );
	xf32 cz = XF_Cos( z );	xf32 sz = XF_Sin( z );
	pq0->x = XF32_Mul(XF32_Mul(sx, cy), cz) + XF32_Mul(XF32_Mul(cx, sy), sz);
	pq0->y = XF32_Mul(XF32_Mul(cx, sy), cz) - XF32_Mul(XF32_Mul(sx, cy), sz);
	pq0->z = XF32_Mul(XF32_Mul(cx, cy), sz) + XF32_Mul(XF32_Mul(sx, sy), cz);
	pq0->w = XF32_Mul(XF32_Mul(cx, cy), cz) - XF32_Mul(XF32_Mul(sx, sy), sz);
	return pq0;
}

/**
 * @brief	]xNgNH[^jIɕϊ
 * @param [out]	pq0	= o̓NH[^jI
 * @param [in]	pv0	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionFromRotYXZ(IrisXFQuaternion* pq0, const IrisXFVec4* pv0)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pv0 );
	xf32 x = XF32_Mul(pv0->x, XF32_HALF);
	xf32 y = XF32_Mul(pv0->y, XF32_HALF);
	xf32 z = XF32_Mul(pv0->z, XF32_HALF);
	xf32 cx = XF_Cos( x );	xf32 sx = XF_Sin( x );
	xf32 cy = XF_Cos( y );	xf32 sy = XF_Sin( y );
	xf32 cz = XF_Cos( z );	xf32 sz = XF_Sin( z );
	pq0->x = XF32_Mul(XF32_Mul(cy, cz), sx) + XF32_Mul(XF32_Mul(sy, cx), sz);
	pq0->y = XF32_Mul(XF32_Mul(sy, cx), cz) - XF32_Mul(XF32_Mul(cy, sx), sz);
	pq0->z = XF32_Mul(XF32_Mul(cy, cx), sz) - XF32_Mul(XF32_Mul(sy, cz), sx);
	pq0->w = XF32_Mul(XF32_Mul(cy, cx), cz) + XF32_Mul(XF32_Mul(sy, sx), sz);
	return pq0;
}

/**
 * @brief	]xNgƉ]ʂANH[^jI𐶐
 * @param [out]	pq0		= o̓NH[^jI
 * @param [in]	angle	= ]
 * @param [in]	pvAxis	= ]xNg
 * @return o̓NH[^jI
*/
IRIS_XFPU_INLINE	IrisXFQuaternion*	XFpuQuaternionFromRotate(IrisXFQuaternion* pq0, xf32 angle, const IrisXFVec4* pvAxis)
{
	XFPU_NULLASSERT( pq0 );
	XFPU_NULLASSERT( pvAxis );
	XFpuVec4NormalizeXYZ((IrisXFVec4*)pq0, pvAxis);

	angle = XF32_Mul(angle, XF32_HALF);
	XFpuVec4Scale((IrisXFVec4*)pq0, (IrisXFVec4*)pq0, XF_Sin(angle));
	pq0->w = XF_Cos( angle );
	return pq0;
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
IRIS_XFPU_INLINE	IrisXFVec4*			XFpuQuaternionToRotZYX(IrisXFVec4* pv0, const IrisXFQuaternion* pq0)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pq0 );
	xf32 cx, sx, sy, cz, sz;
	xf32 rx, ry, rz;
	xf32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sy = -( XF32_Mul(XF32_Mul(XF32_2, z), x) - XF32_Mul(XF32_Mul(XF32_2, w), y) );
	if( sy <= XF32_CONST(0.99995f) && sy >= XF32_CONST(-0.99995f)) 
	{
		sx =			XF32_Mul(XF32_Mul(XF32_2, y), z) + XF32_Mul(XF32_Mul(XF32_2, w), x);
		cx = XF32_ONE - XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, y), y);
		sz =			XF32_Mul(XF32_Mul(XF32_2, x), y) + XF32_Mul(XF32_Mul(XF32_2, w), z);
		cz = XF32_ONE - XF32_Mul(XF32_Mul(XF32_2, y), y) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Atan2( sx, cx );
		ry = XF_Asin( sy );
		rz = XF_Atan2( sz, cz );
	}
	else
	{
#if 1
		//J |cosY|0ɋ߂ƂAZ]0Ɖ肵YX]߂
		sx =			-(XF32_Mul(XF32_Mul(XF32_2, y), z) - XF32_Mul(XF32_Mul(XF32_2, w), x));
		cx = XF32_ONE	- XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Atan2( sx, cx );
		ry = XF_Asin( sy );
		rz = XF32_ZERO;
#else
		//J |cosY|0ɋ߂ƂAX]0Ɖ肵ZY]߂
		sz =			-(XF32_Mul(XF32_Mul(XF32_2, x), y) - XF32_Mul(XF32_Mul(XF32_2, w), z));
		cz = XF32_ONE	- XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF32_ZERO;
		ry = XF_Asin( sy );
		rz = XF_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = XF32_ZERO;
	return (pv0);
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
IRIS_XFPU_INLINE	IrisXFVec4*			XFpuQuaternionToRotXYZ(IrisXFVec4* pv0, const IrisXFQuaternion* pq0)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pq0 );
	xf32 cx, sx, sy, cz, sz;
	xf32 rx, ry, rz;
	xf32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sy = XF32_Mul(XF32_Mul(XF32_2, z), x) + XF32_Mul(XF32_Mul(XF32_2, w), y);
	if( sy <= XF32_CONST(0.99995f) && sy >= XF32_CONST(-0.99995f)) 
	{
		sx =			-(XF32_Mul(XF32_Mul(XF32_2, y), z) - XF32_Mul(XF32_Mul(XF32_2, w), x));
		cx = XF32_ONE	- XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, y), y);
		sz =			-(XF32_Mul(XF32_Mul(XF32_2, x), y) - XF32_Mul(XF32_Mul(XF32_2, w), z));
		cz = XF32_ONE	- XF32_Mul(XF32_Mul(XF32_2, y), y) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Atan2( sx, cx );
		ry = XF_Asin( sy );
		rz = XF_Atan2( sz, cz );
	} 
	else 
	{
#if 1
		//J |cosY|0ɋ߂ƂAZ]0Ɖ肵XY]߂
		sx =			XF32_Mul(XF32_Mul(XF32_2, y), z) + XF32_Mul(XF32_Mul(XF32_2, w), x);
		cx = XF32_ONE - XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Atan2( sx, cx );
		ry = XF_Asin( sy );
		rz = XF32_ZERO;
#else
		//J |cosY|0ɋ߂ƂAX]0Ɖ肵YZ]߂
		sz =			XF32_Mul(XF32_Mul(XF32_2, x), y) + XF32_Mul(XF32_Mul(XF32_2, w), z);
		cz = XF32_ONE - XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF32_ZERO;
		ry = XF_Asin( sy );
		rz = XF_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = XF32_ZERO;
	return (pv0);
}

/**
 * @brief	NH[^jI]xNgɕϊ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pq0	= ]NH[^jI
 * @return o̓xNg
*/
IRIS_XFPU_INLINE	IrisXFVec4*			XFpuQuaternionToRotYXZ(IrisXFVec4* pv0, const IrisXFQuaternion* pq0)
{
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pq0 );
	xf32 sx, sy, cy, sz, cz;
	xf32 rx, ry, rz;
	xf32 x = pq0->x, y = pq0->y, z = pq0->z, w = pq0->w;

	sx = -(XF32_Mul(XF32_Mul(XF32_2, y), z) - XF32_Mul(XF32_Mul(XF32_2, w), x));
	if( sx <= XF32_CONST(0.99995f) && sx >= XF32_CONST(-0.99995f))
	{
		sy =			XF32_Mul(XF32_Mul(XF32_2, z), x) + XF32_Mul(XF32_Mul(XF32_2, w), y);
		cy = XF32_ONE - XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, y), y);
		sz =			XF32_Mul(XF32_Mul(XF32_2, x), y) + XF32_Mul(XF32_Mul(XF32_2, w), z);
		cz = XF32_ONE - XF32_Mul(XF32_Mul(XF32_2, x), x) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Asin( sx );
		ry = XF_Atan2( sy, cy );
		rz = XF_Atan2( sz, cz );
	} 
	else 
	{
#if 1
		//J |cosX|0ɋ߂ƂAZ]0Ɖ肵YX]߂
		sy =			-(XF32_Mul(XF32_Mul(XF32_2, z), x) - XF32_Mul(XF32_Mul(XF32_2, w), y));
		cy = XF32_ONE	- XF32_Mul(XF32_Mul(XF32_2, y), y) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Asin( sx );
		ry = XF_Atan2( sy, cy );
		rz = XF32_ZERO;
#else
		//J |cosX|0ɋ߂ƂAY]0Ɖ肵XZ]߂
		sz =			-(XF32_Mul(XF32_Mul(XF32_2, x), y) - XF32_Mul(XF32_Mul(XF32_2, w), z));
		cz = XF32_ONE	- XF32_Mul(XF32_Mul(XF32_2, y), y) - XF32_Mul(XF32_Mul(XF32_2, z), z);
		rx = XF_Asin( sx );
		ry = XF32_ZERO;
		rz = XF_Atan2( sz, cz );
#endif
	}
	pv0->x = rx;
	pv0->y = ry;
	pv0->z = rz;
	pv0->w = XF32_ZERO;
	return (pv0);
}

}	// end of namespace xf
}	// end of namespace iris

#endif
