//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathXFMatrix44.inl
 * @brief		4x4}gbNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathXFMatrix44_inl_
#define INCG_IRIS_MathXFMatrix44_inl_

namespace iris {
namespace xf
{

//======================================================================
// function
/**
 * @brief	Pʃ}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Unit(IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pm0 );
	pm0->x.x = pm0->y.y = pm0->z.z = pm0->w.w = XF32_ONE;
	pm0->x.y = pm0->x.z = pm0->x.w = 
	pm0->y.x = pm0->y.z = pm0->y.w =
	pm0->z.x = pm0->z.y = pm0->z.w = 
	pm0->w.x = pm0->w.y = pm0->w.z = XF32_ZERO;
	return pm0;
}

/**
 * @brief	[s̐
 * @param [out]	pm0	= o̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Zero(IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pm0 );
	XFpuVec4Zero(&pm0->x);
	XFpuVec4Zero(&pm0->y);
	XFpuVec4Zero(&pm0->z);
	XFpuVec4Zero(&pm0->w);
	return pm0;
}

/**
 * @brief	}gbNX̃Rs[
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Copy(IrisXFMtx44* pm0, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
#if 1
	pm0->_00 = pm1->_00;
	pm0->_01 = pm1->_01;
	pm0->_02 = pm1->_02;
	pm0->_03 = pm1->_03;
	pm0->_10 = pm1->_10;
	pm0->_11 = pm1->_11;
	pm0->_12 = pm1->_12;
	pm0->_13 = pm1->_13;
	pm0->_20 = pm1->_20;
	pm0->_21 = pm1->_21;
	pm0->_22 = pm1->_22;
	pm0->_23 = pm1->_23;
	pm0->_30 = pm1->_30;
	pm0->_31 = pm1->_31;
	pm0->_32 = pm1->_32;
	pm0->_33 = pm1->_33;
#else
	XFpuVec4Copy(&pm0->x, &pm1->x);
	XFpuVec4Copy(&pm0->y, &pm1->y);
	XFpuVec4Copy(&pm0->z, &pm1->z);
	XFpuVec4Copy(&pm0->w, &pm1->w);
#endif
	return pm0;
}

/**
 * @brief	xNg̕sړʕݒ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pv0	= ̓xNg
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44SetTransfer(IrisXFMtx44* pm0, const IrisXFVec4* pv0)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	pm0->w.x = pv0->x;
	pm0->w.y = pv0->y;
	pm0->w.z = pv0->z;
	pm0->w.w = pv0->w;
	return pm0;
}

/**
 * @brief	xNg̕sړʕ擾
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pv0	= ̓xNg
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFVec4*	XFpuMtx44GetTransfer(IrisXFVec4* pv0, const IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	pv0->x = pm0->w.x;
	pv0->y = pm0->w.y;
	pv0->z = pm0->w.z;
	pv0->w = pm0->w.w;
	return pv0;
}

/**
 * @brief	}gbNX̕sړ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ړxNg
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Transfer(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, const IrisXFVec4* pv0)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	XFPU_NULLASSERT( pv0 );
	pm0->x.x = pm1->x.x;
	pm0->x.y = pm1->x.y;
	pm0->x.z = pm1->x.z;
	pm0->x.w = pm1->x.w;
	pm0->y.x = pm1->y.x;
	pm0->y.y = pm1->y.y;
	pm0->y.z = pm1->y.z;
	pm0->y.w = pm1->y.w;
	pm0->z.x = pm1->z.x;
	pm0->z.y = pm1->z.y;
	pm0->z.z = pm1->z.z;
	pm0->z.w = pm1->z.w;
	pm0->w.x = pm1->w.x + pv0->x;
	pm0->w.y = pm1->w.y + pv0->y;
	pm0->w.z = pm1->w.z + pv0->z;
	pm0->w.w = pv0->w;
	return pm0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec4*	XFpuMtx44Transform(IrisXFVec4* pv0, const IrisXFMtx44* pm0, const IrisXFVec4* pv1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	xf32 x = pv1->x;
	xf32 y = pv1->y;
	xf32 z = pv1->z;
	xf32 w = pv1->w;
	pv0->x = XF32_Mul(pm0->x.x, x) + XF32_Mul(pm0->y.x, y) + XF32_Mul(pm0->z.x, z) + XF32_Mul(pm0->w.x, w);
	pv0->y = XF32_Mul(pm0->x.y, x) + XF32_Mul(pm0->y.y, y) + XF32_Mul(pm0->z.y, z) + XF32_Mul(pm0->w.y, w);
	pv0->z = XF32_Mul(pm0->x.z, x) + XF32_Mul(pm0->y.z, y) + XF32_Mul(pm0->z.z, z) + XF32_Mul(pm0->w.z, w);
	pv0->w = XF32_Mul(pm0->x.w, x) + XF32_Mul(pm0->y.w, y) + XF32_Mul(pm0->z.w, z) + XF32_Mul(pm0->w.w, w);
	return pv0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec4*	XFpuMtx44TransformXYZ(IrisXFVec4* pv0, const IrisXFMtx44* pm0, const IrisXFVec4* pv1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	xf32 x = pv1->x;
	xf32 y = pv1->y;
	xf32 z = pv1->z;
	pv0->x = XF32_Mul(pm0->x.x, x) + XF32_Mul(pm0->y.x, y) + XF32_Mul(pm0->z.x, z);
	pv0->y = XF32_Mul(pm0->x.y, x) + XF32_Mul(pm0->y.y, y) + XF32_Mul(pm0->z.y, z);
	pv0->z = XF32_Mul(pm0->x.z, x) + XF32_Mul(pm0->y.z, y) + XF32_Mul(pm0->z.z, z);
	pv0->w = XF32_Mul(pm0->x.w, x) + XF32_Mul(pm0->y.w, y) + XF32_Mul(pm0->z.w, z);
	return pv0;
}

/**
 * @brief	xNgɃ}gbNXZ
 * @param [out]	pv0	= o̓xNg
 * @param [in]	pm0	= ̓}gbNX
 * @param [in]	pv1	= ̓xNg
 * @return	o̓xNg
*/
IRIS_XFPU_INLINE IrisXFVec4*	XFpuMtx44HomogeneousTransform(IrisXFVec4* pv0, const IrisXFMtx44* pm0, const IrisXFVec4* pv1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pv1 );
	IRIS_ASSERT( pv0 != pv1 );	// TODO : AhXɖΉ
	pv0->x = XF32_Mul(pm0->x.x, pv1->x) + XF32_Mul(pm0->y.x, pv1->y) + XF32_Mul(pm0->z.x, pv1->z) + pv1->w;
	pv0->y = XF32_Mul(pm0->x.y, pv1->x) + XF32_Mul(pm0->y.y, pv1->y) + XF32_Mul(pm0->z.y, pv1->z) + pv1->w;
	pv0->z = XF32_Mul(pm0->x.z, pv1->x) + XF32_Mul(pm0->y.z, pv1->y) + XF32_Mul(pm0->z.z, pv1->z) + pv1->w;
	pv0->w = XF32_Mul(pm0->x.w, pv1->x) + XF32_Mul(pm0->y.w, pv1->y) + XF32_Mul(pm0->z.w, pv1->z) + pv1->w;
	return pv0;
}

/**
 * @brief	Q̃}gbNX̐
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pm2	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Mul(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, const IrisXFMtx44* pm2)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	XFPU_NULLASSERT( pm2 );
	IRIS_ASSERT( pm0 != pm1 );
	IRIS_ASSERT( pm0 != pm2 );
	pm0->x.x = XF32_Mul(pm1->x.x, pm2->x.x) + XF32_Mul(pm1->y.x, pm2->x.y) + XF32_Mul(pm1->z.x, pm2->x.z) + XF32_Mul(pm1->w.x, pm2->x.w);
	pm0->x.y = XF32_Mul(pm1->x.y, pm2->x.x) + XF32_Mul(pm1->y.y, pm2->x.y) + XF32_Mul(pm1->z.y, pm2->x.z) + XF32_Mul(pm1->w.y, pm2->x.w);
	pm0->x.z = XF32_Mul(pm1->x.z, pm2->x.x) + XF32_Mul(pm1->y.z, pm2->x.y) + XF32_Mul(pm1->z.z, pm2->x.z) + XF32_Mul(pm1->w.z, pm2->x.w);
	pm0->x.w = XF32_Mul(pm1->x.w, pm2->x.x) + XF32_Mul(pm1->y.w, pm2->x.y) + XF32_Mul(pm1->z.w, pm2->x.z) + XF32_Mul(pm1->w.w, pm2->x.w);
	pm0->y.x = XF32_Mul(pm1->x.x, pm2->y.x) + XF32_Mul(pm1->y.x, pm2->y.y) + XF32_Mul(pm1->z.x, pm2->y.z) + XF32_Mul(pm1->w.x, pm2->y.w);
	pm0->y.y = XF32_Mul(pm1->x.y, pm2->y.x) + XF32_Mul(pm1->y.y, pm2->y.y) + XF32_Mul(pm1->z.y, pm2->y.z) + XF32_Mul(pm1->w.y, pm2->y.w);
	pm0->y.z = XF32_Mul(pm1->x.z, pm2->y.x) + XF32_Mul(pm1->y.z, pm2->y.y) + XF32_Mul(pm1->z.z, pm2->y.z) + XF32_Mul(pm1->w.z, pm2->y.w);
	pm0->y.w = XF32_Mul(pm1->x.w, pm2->y.x) + XF32_Mul(pm1->y.w, pm2->y.y) + XF32_Mul(pm1->z.w, pm2->y.z) + XF32_Mul(pm1->w.w, pm2->y.w);
	pm0->z.x = XF32_Mul(pm1->x.x, pm2->z.x) + XF32_Mul(pm1->y.x, pm2->z.y) + XF32_Mul(pm1->z.x, pm2->z.z) + XF32_Mul(pm1->w.x, pm2->z.w);
	pm0->z.y = XF32_Mul(pm1->x.y, pm2->z.x) + XF32_Mul(pm1->y.y, pm2->z.y) + XF32_Mul(pm1->z.y, pm2->z.z) + XF32_Mul(pm1->w.y, pm2->z.w);
	pm0->z.z = XF32_Mul(pm1->x.z, pm2->z.x) + XF32_Mul(pm1->y.z, pm2->z.y) + XF32_Mul(pm1->z.z, pm2->z.z) + XF32_Mul(pm1->w.z, pm2->z.w);
	pm0->z.w = XF32_Mul(pm1->x.w, pm2->z.x) + XF32_Mul(pm1->y.w, pm2->z.y) + XF32_Mul(pm1->z.w, pm2->z.z) + XF32_Mul(pm1->w.w, pm2->z.w);
	return pm0;
}

/**
 * @brief	}gbNX̃XP[O
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	s	= XJ[l
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Scale(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, xf32 s)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	XFpuVec4Scale(&pm0->x, &pm1->x, s);
	XFpuVec4Scale(&pm0->y, &pm1->y, s);
	XFpuVec4Scale(&pm0->z, &pm1->z, s);
	XFpuVec4Scale(&pm0->w, &pm1->w, s);
	return pm0;
}

/**
 * @brief	}gbNX̓]us߂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Transpose(IrisXFMtx44* pm0, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	pm0->x.x = pm1->x.x;
	xf32 f01 = pm1->x.y;
	xf32 f02 = pm1->x.z;
	xf32 f03 = pm1->x.w;
	pm0->x.y = pm1->y.x;
	xf32 f11 = pm1->y.y;
	xf32 f12 = pm1->y.z;
	xf32 f13 = pm1->y.w;
	pm0->x.z = pm1->z.x;
	xf32 f21 = pm1->z.y;
	xf32 f22 = pm1->z.z;
	xf32 f23 = pm1->z.w;
	pm0->x.w = pm1->w.x;
	pm0->y.x = f01;
	pm0->y.y = f11;
	pm0->y.z = f21;
	pm0->y.w = pm1->w.y;
	pm0->z.x = f02;
	pm0->z.y = f12;
	pm0->z.z = f22;
	pm0->z.w = pm1->w.z;
	pm0->w.x = f03;
	pm0->w.y = f13;
	pm0->w.z = f23;
	pm0->w.w = pm1->w.w;
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotZ(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, xf32 rz)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_Cos(rz);
	xf32 s = XF32_Sin(rz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = XF32_ZERO;
		pm0->x.w = XF32_ZERO;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = XF32_ZERO;
		pm0->y.w = XF32_ZERO;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = XF32_ONE;
		pm0->z.w = XF32_ZERO;
		pm0->w.x = XF32_ZERO;
		pm0->w.y = XF32_ZERO;
		pm0->w.z = XF32_ZERO;
		pm0->w.w = XF32_ONE;
	}
	else
	{
		IrisXFMtx44 m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = XF32_ZERO;
		m.x.w = XF32_ZERO;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = XF32_ZERO;
		m.y.w = XF32_ZERO;
		m.z.x = XF32_ZERO;
		m.z.y = XF32_ZERO;
		m.z.z = XF32_ONE;
		m.z.w = XF32_ZERO;
		m.w.x = XF32_ZERO;
		m.w.y = XF32_ZERO;
		m.w.z = XF32_ZERO;
		m.w.w = XF32_ONE;
		XFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXZ]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rz	= Z]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotIdxZ(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, u16 idz)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_CosIdx(idz);
	xf32 s = XF32_SinIdx(idz);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = +s;
		pm0->x.z = XF32_ZERO;
		pm0->x.w = XF32_ZERO;
		pm0->y.x = -s;
		pm0->y.y = +c;
		pm0->y.z = XF32_ZERO;
		pm0->y.w = XF32_ZERO;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = XF32_ONE;
		pm0->z.w = XF32_ZERO;
		pm0->w.x = XF32_ZERO;
		pm0->w.y = XF32_ZERO;
		pm0->w.z = XF32_ZERO;
		pm0->w.w = XF32_ONE;
	}
	else
	{
		IrisXFMtx44 m;
		m.x.x = +c;
		m.x.y = +s;
		m.x.z = XF32_ZERO;
		m.x.w = XF32_ZERO;
		m.y.x = -s;
		m.y.y = +c;
		m.y.z = XF32_ZERO;
		m.y.w = XF32_ZERO;
		m.z.x = XF32_ZERO;
		m.z.y = XF32_ZERO;
		m.z.z = XF32_ONE;
		m.z.w = XF32_ZERO;
		m.w.x = XF32_ZERO;
		m.w.y = XF32_ZERO;
		m.w.z = XF32_ZERO;
		m.w.w = XF32_ONE;
		XFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	ry	= Y]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotY(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, xf32 ry)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_Cos(ry);
	xf32 s = XF32_Sin(ry);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = +s;
		pm0->x.w = XF32_ZERO;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = XF32_ONE;
		pm0->y.z = XF32_ZERO;
		pm0->y.w = XF32_ZERO;
		pm0->z.x = -s;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = +c;
		pm0->z.w = XF32_ZERO;
		pm0->w.x = XF32_ZERO;
		pm0->w.y = XF32_ZERO;
		pm0->w.z = XF32_ZERO;
		pm0->w.w = XF32_ONE;
	}
	else
	{
		IrisXFMtx44 m;
		m.x.x = +c;
		m.x.y = XF32_ZERO;
		m.x.z = +s;
		m.x.w = XF32_ZERO;
		m.y.x = XF32_ZERO;
		m.y.y = XF32_ONE;
		m.y.z = XF32_ZERO;
		m.y.w = XF32_ZERO;
		m.z.x = -s;
		m.z.y = XF32_ZERO;
		m.z.z = +c;
		m.z.w = XF32_ZERO;
		m.w.x = XF32_ZERO;
		m.w.y = XF32_ZERO;
		m.w.z = XF32_ZERO;
		m.w.w = XF32_ONE;
		XFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXY]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idy	= Y]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotIdxY(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, u16 idy)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_CosIdx(idy);
	xf32 s = XF32_SinIdx(idy);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = +c;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = +s;
		pm0->x.w = XF32_ZERO;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = XF32_ONE;
		pm0->y.z = XF32_ZERO;
		pm0->y.w = XF32_ZERO;
		pm0->z.x = -s;
		pm0->z.y = XF32_ZERO;
		pm0->z.z = +c;
		pm0->z.w = XF32_ZERO;
		pm0->w.x = XF32_ZERO;
		pm0->w.y = XF32_ZERO;
		pm0->w.z = XF32_ZERO;
		pm0->w.w = XF32_ONE;
	}
	else
	{
		IrisXFMtx44 m;
		m.x.x = +c;
		m.x.y = XF32_ZERO;
		m.x.z = +s;
		m.x.w = XF32_ZERO;
		m.y.x = XF32_ZERO;
		m.y.y = XF32_ONE;
		m.y.z = XF32_ZERO;
		m.y.w = XF32_ZERO;
		m.z.x = -s;
		m.z.y = XF32_ZERO;
		m.z.z = +c;
		m.z.w = XF32_ZERO;
		m.w.x = XF32_ZERO;
		m.w.y = XF32_ZERO;
		m.w.z = XF32_ZERO;
		m.w.w = XF32_ONE;
		XFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	rx	= X]ʁiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotX(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, xf32 rx)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_Cos(rx);
	xf32 s = XF32_Sin(rx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = XF32_ONE;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = XF32_ZERO;
		pm0->x.w = XF32_ZERO;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->y.w = XF32_ZERO;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = -s;
		pm0->z.z = +c;
		pm0->z.w = XF32_ZERO;
		pm0->w.x = XF32_ZERO;
		pm0->w.y = XF32_ZERO;
		pm0->w.z = XF32_ZERO;
		pm0->w.w = XF32_ONE;
	}
	else
	{
		IrisXFMtx44 m;
		m.x.x = XF32_ONE;
		m.x.y = XF32_ZERO;
		m.x.z = XF32_ZERO;
		m.x.w = XF32_ZERO;
		m.y.x = XF32_ZERO;
		m.y.y = +c;
		m.y.z = +s;
		m.y.w = XF32_ZERO;
		m.z.x = XF32_ZERO;
		m.z.y = -s;
		m.z.z = +c;
		m.z.w = XF32_ZERO;
		m.w.x = XF32_ZERO;
		m.w.y = XF32_ZERO;
		m.w.z = XF32_ZERO;
		m.w.w = XF32_ONE;
		XFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNXX]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	idx	= X]ʁiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotIdxX(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, u16 idx)
{
	XFPU_NULLASSERT( pm0 );
	xf32 c = XF32_CosIdx(idx);
	xf32 s = XF32_SinIdx(idx);
	if( pm1 == nullptr ) 
	{
		pm0->x.x = XF32_ONE;
		pm0->x.y = XF32_ZERO;
		pm0->x.z = XF32_ZERO;
		pm0->x.w = XF32_ZERO;
		pm0->y.x = XF32_ZERO;
		pm0->y.y = +c;
		pm0->y.z = +s;
		pm0->y.w = XF32_ZERO;
		pm0->z.x = XF32_ZERO;
		pm0->z.y = -s;
		pm0->z.z = +c;
		pm0->z.w = XF32_ZERO;
		pm0->w.x = XF32_ZERO;
		pm0->w.y = XF32_ZERO;
		pm0->w.z = XF32_ZERO;
		pm0->w.w = XF32_ONE;
	}
	else
	{
		IrisXFMtx44 m;
		m.x.x = XF32_ONE;
		m.x.y = XF32_ZERO;
		m.x.z = XF32_ZERO;
		m.x.w = XF32_ZERO;
		m.y.x = XF32_ZERO;
		m.y.y = +c;
		m.y.z = +s;
		m.y.w = XF32_ZERO;
		m.z.x = XF32_ZERO;
		m.z.y = -s;
		m.z.z = +c;
		m.z.w = XF32_ZERO;
		m.w.x = XF32_ZERO;
		m.w.y = XF32_ZERO;
		m.w.z = XF32_ZERO;
		m.w.w = XF32_ONE;
		XFpuMtx44Mul(pm0, &m, pm1);
	}
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiWAj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Rot(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, const IrisXFVec4* pv0)
{
	XFPU_NULLASSERT( pv0 );
	XFpuMtx44RotZ(pm0, pm1, pv0->z);
	XFpuMtx44RotY(pm0, pm0, pv0->y);
	XFpuMtx44RotX(pm0, pm0, pv0->x);
	return pm0;
}

/**
 * @brief	}gbNX̉]
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @param [in]	pv0	= ]xNgiCfbNXj
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44RotIdx(IrisXFMtx44* pm0, const IrisXFMtx44* pm1, const IrisSVec4* pv0)
{
	XFPU_NULLASSERT( pv0 );
	XFpuMtx44RotIdxZ(pm0, pm1, (u16)pv0->z);
	XFpuMtx44RotIdxY(pm0, pm0, (u16)pv0->y);
	XFpuMtx44RotIdxX(pm0, pm0, (u16)pv0->x);
	return pm0;
}

/**
 * @brief	PʍsɂȂĂ邩ǂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	^Ul
*/
IRIS_XFPU_INLINE IrisBool		XFpuMtx44IsUnit(const IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pm0 );
	const IrisMtx44* m = (const IrisMtx44*)(pm0);
	if( m->im.x.x != 0x3F800000 || m->im.y.y != 0x3F800000 || m->im.z.z != 0x3F800000 || m->im.w.w != 0x3F800000 )
		return IRIS_FALSE;
	if( (             m->im.x.y | m->im.x.z | m->im.x.w
		| m->im.y.x             | m->im.y.z | m->im.y.w
		| m->im.z.x | m->im.z.y             | m->im.z.w
		| m->im.w.x | m->im.w.y | m->im.w.z             ) != 0 )
		return IRIS_FALSE;
	return IRIS_TRUE;
}

/**
 * @brief	}gbNX̃g[XԂ
 * @param [in]	pm0	= ̓}gbNX
 * @return	}gbNX̃g[X
*/
IRIS_XFPU_INLINE xf32			XFpuMtx44Trace(const IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pm0 );
	return (pm0->x.x + pm0->y.y + pm0->z.z + pm0->w.w);
}

/**
 * @brief	s񎮂Ԃ
 * @param [in]	pm0	= ̓}gbNX
 * @return	s
*/
IRIS_XFPU_INLINE xf32			XFpuMtx44Determinant(const IrisXFMtx44* pm0)
{
	XFPU_NULLASSERT( pm0 );
	xf32 a21 = pm0->x.y;
	xf32 a31 = pm0->x.z;
	xf32 a41 = pm0->x.w;
	xf32 a22 = pm0->y.y;
	xf32 a32 = pm0->y.z;
	xf32 a42 = pm0->y.w;
	xf32 a23 = pm0->z.y;
	xf32 a33 = pm0->z.z;
	xf32 a43 = pm0->z.w;
	xf32 a24 = pm0->w.y;
	xf32 a34 = pm0->w.z;
	xf32 a44 = pm0->w.w;

	xf32 d1 = XF32_Mul(pm0->x.x
		, (XF32_Mul(XF32_Mul(a22, a33), a44)
		+  XF32_Mul(XF32_Mul(a23, a34), a42)
		+  XF32_Mul(XF32_Mul(a24, a32), a43)
		-  XF32_Mul(XF32_Mul(a24, a33), a42)
		-  XF32_Mul(XF32_Mul(a23, a32), a44)
		-  XF32_Mul(XF32_Mul(a22, a34), a43)) );
	xf32 d2 = XF32_Mul(pm0->y.x
		, (XF32_Mul(XF32_Mul(a21, a33), a44)
		+  XF32_Mul(XF32_Mul(a23, a34), a41)
		+  XF32_Mul(XF32_Mul(a24, a31), a43)
		-  XF32_Mul(XF32_Mul(a24, a33), a41)
		-  XF32_Mul(XF32_Mul(a23, a31), a44)
		-  XF32_Mul(XF32_Mul(a21, a34), a43)) );
	xf32 d3 = XF32_Mul(pm0->z.x
		, (XF32_Mul(XF32_Mul(a21, a32), a44)
		+  XF32_Mul(XF32_Mul(a22, a34), a41)
		+  XF32_Mul(XF32_Mul(a24, a31), a42)
		-  XF32_Mul(XF32_Mul(a24, a32), a41)
		-  XF32_Mul(XF32_Mul(a22, a31), a44)
		-  XF32_Mul(XF32_Mul(a21, a34), a42)) );
	xf32 d4 = XF32_Mul(pm0->w.x
		, (XF32_Mul(XF32_Mul(a21, a32), a43)
		+  XF32_Mul(XF32_Mul(a22, a33), a41)
		+  XF32_Mul(XF32_Mul(a23, a31), a42)
		-  XF32_Mul(XF32_Mul(a23, a32), a41)
		-  XF32_Mul(XF32_Mul(a22, a31), a43)
		-  XF32_Mul(XF32_Mul(a21, a33), a42)) );
	return (d1 - d2 + d3 - d4);
}

/**
 * @brief	]qsԂ
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Adjoint(IrisXFMtx44* pm0, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	int i, j, k;
	IrisXFVec4 v[3];
	IrisXFMtx33 minor;
	IrisXFMtx44  adjoint;
	for(i=0; i < 4; ++i)
	{
		for(j=0; j < 4; ++j)
		{
			if(i==0) 
			{
				v[0] = pm1->y;
				v[1] = pm1->z;
				v[2] = pm1->w;
			}
			else if(i==1)
			{
				v[0] = pm1->x;
				v[1] = pm1->z;
				v[2] = pm1->w;
			} 
			else if(i==2)
			{
				v[0] = pm1->x;
				v[1] = pm1->y;
				v[2] = pm1->w;
			} 
			else
			{
				v[0] = pm1->x;
				v[1] = pm1->y;
				v[2] = pm1->z;
			}
			if(j==0) 
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].y;
					v[k].y = v[k].z;
					v[k].z = v[k].w;
				}
			}
			else if(j==1) 
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].x;
					v[k].y = v[k].z;
					v[k].z = v[k].w;
				}
			} 
			else if(j==2) 
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].x;
					v[k].y = v[k].y;
					v[k].z = v[k].w;
				}
			} 
			else
			{
				for(k=0; k < 3; ++k)
				{
					v[k].x = v[k].x;
					v[k].y = v[k].y;
					v[k].z = v[k].z;
				}
			}
			minor.x.x = v[0].x;
			minor.x.y = v[0].y;
			minor.x.z = v[0].z;
			minor.y.x = v[1].x;
			minor.y.y = v[1].y;
			minor.y.z = v[1].z;
			minor.z.x = v[2].x;
			minor.z.y = v[2].y;
			minor.z.z = v[2].z;

			if((i ^ j) & 1)
			{
				adjoint.m[j][i] = -XFpuMtx33Determinant(&minor);
			} 
			else
			{
				adjoint.m[j][i] =  XFpuMtx33Determinant(&minor);
			}
		}
	}
	XFpuMtx44Copy(pm0, &adjoint);
	return pm0;
}

/**
 * @brief	tsԂ(s͉]+sړƉ)
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Inverse(IrisXFMtx44* pm0, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	IrisXFMtx44 m;
	IrisXFVec4 v;
	XFpuMtx44Copy(&m, pm1);
	v.x = m.w.x; v.y = m.w.y; v.z = m.w.z;
	m.w.x = m.w.y = m.w.z = XF32_ZERO;

	XFpuMtx44Transpose(pm0, &m);
	XFpuMtx44TransformXYZ(&v, pm0, &v);
	pm0->w.x = -v.x;
	pm0->w.y = -v.y;
	pm0->w.z = -v.z;
	return pm0;
}

/**
 * @brief	tsԂ
 * @param [out]	pm0				= o̓}gbNX
 * @param [out]	pDeterminant	= 
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44Inverse2(IrisXFMtx44* pm0, xf32* pDeterminant, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	xf32 d = XFpuMtx44Determinant(pm1);
	if( pDeterminant != nullptr ) *pDeterminant = d;
	if( d == XF32_ZERO ) return nullptr;
	XFpuMtx44Adjoint(pm0, pm1);
	XFpuMtx44Scale(pm0, pm0, XF32_Div(XF32_ONE, d) );
	return pm0;
}

/**
 * @brief	ƕʂ蓊e}gNX߂
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pv0				= xNg
 * @param [in]	pl0				= 
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44DropShadow(IrisXFMtx44* pm0, const IrisXFVec4* pv0, const IrisXFPlane* pl0)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pv0 );
	XFPU_NULLASSERT( pl0 );
	IrisUnion128 u;
	XFpuPlaneNormalize(&u.xfp, pl0);
	xf32 d = XFpuVec4InnerProductXYZ(&u.v.xfv, pv0);
	XFpuVec4Scale(&pm0->x, pv0, -u.xfp.a);
	XFpuVec4Scale(&pm0->y, pv0, -u.xfp.b);
	XFpuVec4Scale(&pm0->z, pv0, -u.xfp.c);
	XFpuVec4Scale(&pm0->w, pv0, -u.xfp.d);
	pm0->x.x += d;
	pm0->y.y += d;
	pm0->z.z += d;
	pm0->w.w += d;
	return pm0;
}

/**
 * @brief	}gNX̐K
 * @param [out]	pm0				= o̓}gbNX
 * @param [in]	pm1				= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44NormalizeXYZ(IrisXFMtx44* pm0, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	// XY̊OςZ𓾂
	XFpuVec4OuterProductXYZ(&pm0->z, &pm1->x, &pm1->y);
	// Oς瓾ZY̊OςX𓾂
	XFpuVec4OuterProductXYZ(&pm0->x, &pm1->y, &pm0->z);
	XFpuVec4Normalize(&pm0->x, &pm0->x);
	XFpuVec4Normalize(&pm0->y, &pm1->y);
	XFpuVec4Normalize(&pm0->z, &pm0->z);
	pm0->w = pm1->w;
	return pm0;
}

/**
 * @brief	8rbg̐x؂̂
 * @param [out]	pm0	= o̓}gbNX
 * @param [in]	pm1	= ̓}gbNX
 * @return	o̓}gbNX
*/
IRIS_XFPU_INLINE IrisXFMtx44*	XFpuMtx44TruncatePrecision24(IrisXFMtx44* pm0, const IrisXFMtx44* pm1)
{
	XFPU_NULLASSERT( pm0 );
	XFPU_NULLASSERT( pm1 );
	XFpuVec4TruncatePrecision24(&pm0->x, &pm1->x);
	XFpuVec4TruncatePrecision24(&pm0->y, &pm1->y);
	XFpuVec4TruncatePrecision24(&pm0->z, &pm1->z);
	XFpuVec4TruncatePrecision24(&pm0->w, &pm1->w);
	return pm0;
}

}	// end of namespace xf
}	// end of namespace iris

#endif
