//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathXFComplex.inl
 * @brief		fZt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_MathXFComplex_inl_
#define INCG_IRIS_MathXFComplex_inl_

namespace iris {
namespace xf
{

//======================================================================
// function
/**
 * @brief	f̐
 * @param [out]	pc0	= o͕f
 * @param [in]	re	= 
 * @param [in]	im	= 
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex* XFpuComplexSet(IrisXFComplex* pc0, xf32 re, xf32 im)
{
	XFPU_NULLASSERT( pc0 );
	pc0->re = re;
	pc0->im = im;
	return pc0;
}

/**
 * @brief	[f̐
 * @param [out]	pc0	= o͕f
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex* XFpuComplexZero(IrisXFComplex* pc0)
{
	XFPU_NULLASSERT( pc0 );
	pc0->re = XF32_ZERO;
	pc0->im = XF32_ZERO;
	return pc0;
}

/**
 * @brief	f̉Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= f
 * @param [in]	pc2	= f
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexAdd(IrisXFComplex* pc0, const IrisXFComplex* pc1, const IrisXFComplex* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->re = pc1->re + pc2->re;
	pc0->im = pc1->im + pc2->im;
	return pc0;
}

/**
 * @brief	f̌Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񌸕f
 * @param [in]	pc2	= f
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexSub(IrisXFComplex* pc0, const IrisXFComplex* pc1, const IrisXFComplex* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->re = pc1->re - pc2->re;
	pc0->im = pc1->im - pc2->im;
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 敡f
 * @param [in]	pc2	= 敡f
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexMul(IrisXFComplex* pc0, const IrisXFComplex* pc1, const IrisXFComplex* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	pc0->re = XF32_Mul(pc1->re, pc2->re) - XF32_Mul(pc1->im, pc2->im);
	pc0->im = XF32_Mul(pc1->im, pc2->re) + XF32_Mul(pc1->re, pc2->im);
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񏜕f
 * @param [in]	pc2	= f
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexDiv(IrisXFComplex* pc0, const IrisXFComplex* pc1, const IrisXFComplex* pc2)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	XFPU_NULLASSERT( pc2 );
	xf32 re = pc2->re;
	xf32 im = pc2->im;
	if( XFpuIsNaN(re) || XFpuIsNaN(im) )
	{
		pc0->re = XFpuPositiveSNaNF(pc2->re);
		pc0->im = pc1->re;
	}
	else if( (re < XF32_ZERO ? -re : +re) < (im < XF32_ZERO ? -im : +im) )
	{
		xf32 wr = XF32_Div(pc2->im, pc2->re);
		xf32 wd = pc2->re + XF32_Mul(wr, pc2->im);

		if( XFpuIsNaN(wd) || wd == XF32_ZERO )
		{
			pc0->re = XFpuPositiveSNaNF(pc2->re);
			pc0->im = pc1->re;
		}
		else
		{
			xf32 tmp = XF32_Div(pc1->re + XF32_Mul(pc1->im, wr), wd);
			pc0->im = XF32_Div(pc1->im - XF32_Mul(pc1->re, wr), wd);
			pc0->re = tmp;
		}
	}
	else if( im == XF32_ZERO )
	{
		pc0->re = XFpuPositiveSNaNF(pc2->re);
		pc0->im = pc1->re;
	}
	else
	{
		xf32 wr = XF32_Div(pc2->re, pc2->im);
		xf32 wd = pc2->im + XF32_Mul(wr, pc2->re);

		if( XFpuIsNaN(wd) || wd == XF32_ZERO )
		{
			pc0->re = XFpuPositiveSNaNF(pc2->re);
			pc0->im = pc1->re;
		}
		else
		{
			xf32 tmp = XF32_Div(XF32_Mul(pc1->re, wr) + pc1->im, wd);
			pc0->im = XF32_Div(XF32_Mul(pc1->im, wr) - pc1->re, wd);
			pc0->re = tmp;
		}
	}
	return pc0;
}

/**
 * @brief	f̉Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= f
 * @param [in]	re	= 
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexAddReal(IrisXFComplex* pc0, const IrisXFComplex* pc1, xf32 re)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->re = pc1->re + re;
	return pc0;
}

/**
 * @brief	f̌Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񌸕f
 * @param [in]	re	= 
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexSubReal(IrisXFComplex* pc0, const IrisXFComplex* pc1, xf32 re)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->re = pc1->re - re;
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 敡f
 * @param [in]	re	= 搔
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexMulReal(IrisXFComplex* pc0, const IrisXFComplex* pc1, xf32 re)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->re = XF32_Div(pc1->re, re);
	pc0->im = XF32_Div(pc1->im, re);
	return pc0;
}

/**
 * @brief	f̏Z
 * @param [out]	pc0	= o͕f
 * @param [in]	pc1	= 񏜕f
 * @param [in]	re	= 
 * @return	o͕f
*/
IRIS_XFPU_INLINE IrisXFComplex*	XFpuComplexDivReal(IrisXFComplex* pc0, const IrisXFComplex* pc1, xf32 re)
{
	XFPU_NULLASSERT( pc0 );
	XFPU_NULLASSERT( pc1 );
	pc0->re = XF32_Div(pc1->re, re);
	pc0->im = XF32_Div(pc1->im, re);
	return pc0;
}

}	// end of namespace xf
}	// end of namespace iris

#endif
