//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		iml_ratio.h
 * @brief		ratio t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_iml_ratio_HPP_
#define INCG_IRIS_iml_ratio_HPP_

//======================================================================
// include
#include "../../c++0x/cpp0x_static_assert.h"
#include "../detail/iml_detail.hpp"
#include "../iml_math.hpp"
#include "../iml_constant.hpp"

namespace iml
{

//======================================================================
// declare
template<intmax_t N, intmax_t D=1>class ratio;

//======================================================================
// typedef
typedef ratio<INTMAX_C(                  1), INTMAX_C(1000000000000000000)>	atto;
typedef ratio<INTMAX_C(                  1), INTMAX_C(   1000000000000000)>	femto;
typedef ratio<INTMAX_C(                  1), INTMAX_C(      1000000000000)>	pico;
typedef ratio<INTMAX_C(                  1), INTMAX_C(         1000000000)>	nano;
typedef ratio<INTMAX_C(                  1), INTMAX_C(            1000000)>	micro;
typedef ratio<INTMAX_C(                  1), INTMAX_C(               1000)>	milli;
typedef ratio<INTMAX_C(                  1), INTMAX_C(                100)>	centi;
typedef ratio<INTMAX_C(                  1), INTMAX_C(                 10)>	deci;
typedef ratio<INTMAX_C(                 10), INTMAX_C(                  1)>	deca;
typedef ratio<INTMAX_C(                100), INTMAX_C(                  1)>	hecto;
typedef ratio<INTMAX_C(               1000), INTMAX_C(                  1)>	kilo;
typedef ratio<INTMAX_C(            1000000), INTMAX_C(                  1)>	mega;
typedef ratio<INTMAX_C(         1000000000), INTMAX_C(                  1)>	giga;
typedef ratio<INTMAX_C(      1000000000000), INTMAX_C(                  1)>	tera;
typedef ratio<INTMAX_C(   1000000000000000), INTMAX_C(                  1)>	peta;
typedef ratio<INTMAX_C(1000000000000000000), INTMAX_C(                  1)>	exa;

//======================================================================
// class

template<intmax_t N, intmax_t D>
class ratio 
{
	STATIC_ASSERT( static_abs<N>::value >= 0 );
	STATIC_ASSERT( static_abs<D>::value > 0 );
	static const intmax_t	m_na = static_abs<N>::value;
	static const intmax_t	m_da = static_abs<D>::value;
	static const intmax_t	m_s = static_sign<N>::value * static_sign<D>::value;
	static const intmax_t	m_gcd = static_gcd<m_na, m_da>::value;
public:
	static const intmax_t	num = m_s * m_na / m_gcd;
	static const intmax_t	den = m_da / m_gcd;

	typedef typename ratio<num, den>	type;
private:
	ratio() {}
};

template<class R1, class R2>
class ratio_addition
{
	static const intmax_t	m_lcm_den = static_lcm<R1::den, R2::den>::value;
public:
	typedef typename ratio< R1::num * m_lcm_den / R1::den + R2::num * m_lcm_den / R2::den, m_lcm_den >::type	type;
};

template<class R1, class R2>
class ratio_substruct
{
	static const intmax_t	m_lcm_den = static_lcm<R1::den, R2::den>::value;
public:
	typedef typename ratio< R1::num * m_lcm_den / R1::den - R2::num * m_lcm_den / R2::den, m_lcm_den >::type	type;
};

template<class R1, class R2>
class ratio_multiply
{
	static const intmax_t	m_gcd1_2 = static_gcd<R1::num, R2::den>::value;
	static const intmax_t	m_gcd2_1 = static_gcd<R2::num, R1::den>::value;
public:
	typedef typename ratio< static_mul<R1::num / m_gcd1_2, R2::num / m_gcd2_1>::value
		, static_mul<R1::den / m_gcd2_1, R2::den / m_gcd1_2>::value >::type	type;
};

template<class R1, class R2>
class ratio_divide
{
	static const intmax_t	m_gcd1_2 = static_gcd<R1::num, R2::num>::value;
	static const intmax_t	m_gcd2_1 = static_gcd<R2::den, R1::den>::value;
public:
	typedef typename ratio< static_mul<R1::num / m_gcd1_2, R2::den / m_gcd2_1>::value
		, static_mul<R1::den / m_gcd2_1, R2::num / m_gcd1_2>::value >::type	type;
};

template<class R1, class R2>
class ratio_equal : public constant<bool, (R1::num == R2::num && R1::den == R2::den)>
{
};

template<class R1, class R2>
class ratio_not_equal : public constant<bool, !(ratio_equal<R1, R2>::value)>
{
};

template<class R1, class R2>
class ratio_less
{
	template<class TR1, class TR2
		, intmax_t I1 = TR1::num / TR1::den, intmax_t D1 = TR1::num % TR1::den
		, intmax_t I2 = TR2::num / TR2::den, intmax_t D2 = TR2::num % TR2::den>
	struct comp_less_impl : public constant<bool, I1 < I2>
	{
	};

	template<class TR1, class TR2, intmax_t I>
	struct comp_less_impl<TR1, TR2, I, 0, I, 0> : public constant<bool, false>
	{
	};

	template<class TR1, class TR2, intmax_t I, intmax_t D1>
	struct comp_less_impl<TR1, TR2, I, D1, I, 0> : public constant<bool, false>
	{
	};

	template<class TR1, class TR2, intmax_t I, intmax_t D2>
	struct comp_less_impl<TR1, TR2, I, 0, I, D2> : public constant<bool, true>
	{
	};

	template<class TR1, class TR2, intmax_t I, intmax_t D1, intmax_t D2>
	struct comp_less_impl<TR1, TR2, I, D1, I, D2> : public constant<bool, D1 < D2>
	{
	};

	template<class TR1, class TR2, intmax_t _S1 = static_sign<TR1::num>::value, intmax_t _S2 = static_sign<TR2::num>::value>
	struct less_impl : public constant<bool, (_S1 < _S2)>
	{
	};
	template<class TR1, class TR2>
	struct less_impl<TR1, TR2, 1, 1>
	{
		static const bool value = comp_less_impl<TR1, TR2>::value;
	};
	template<class TR1, class TR2>
	struct less_impl<TR1, TR2, -1, -1>
	{
		static const bool value = comp_less_impl< ratio<-TR2::num, TR2::den>, ratio<-TR1::num, TR1::den> >::value;
	};

public:
	static const bool value = less_impl<R1, R2>::value;
};

template<class R1, class R2>
class ratio_less_equal : public constant<bool, !(ratio_less<R2, R1>::value)>
{
};

template<class R1, class R2>
class ratio_greater : public constant<bool, ratio_less<R2, R1>::value>
{
};

template<class R1, class R2>
class ratio_greater_equal : public constant<bool, !(ratio_less<R1, R2>::value)>
{
};

template<typename TN>
class is_ratio 
{
	template<typename TT>
	struct is_ratio_impl : public iml::detail::false_type {};
	template<intmax_t N, intmax_t D>
	struct is_ratio_impl< ratio<N, D> > : public iml::detail::true_type {};
public:
	static const bool value = is_ratio_impl<TN>::value;
};

}	// end of namespace iml

#endif
