//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		iris_allegrex.h
 * @brief		t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_iris_allegrex_H_
#define INCG_IRIS_iris_allegrex_H_

//======================================================================
// include
#include <stdlib.h>
#include "iris_inchead.h"

//======================================================================
// define

EXTERN_C_BEGIN

//======================================================================
// inline function

/**
 * @brief	rbgĂ鑍Zo
 *			Count All One
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_cao(IrisU32 val)
{
	val  = (val & 0x55555555) + (val >> 1 & 0x55555555);
	val  = (val & 0x33333333) + (val >> 2 & 0x33333333);
	val  = (val & 0x0f0f0f0f) + (val >> 4 & 0x0f0f0f0f);
	val  = (val & 0x00ff00ff) + (val >> 8 & 0x00ff00ff);
	return (IrisS32)((val & 0x0000ffff) + (val >>16 & 0x0000ffff));
}

/**
 * @brief	rbgQĂ鑍Zo
 *			Count All Zero
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_caz(IrisU32 val)
{
	return iris_allegrex_cao(~val);
}

/**
 * @brief	MSB0Zo
 *			Count Leading Zero
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_clz(IrisArgU32 val)
{
	IrisS32 n = 1;
	IrisU32 ui = val;
	if( ui == 0 ) return 32;
#ifdef __BIG_ENDIAN__
	if( (ui&0xFFFF) == 0 ) { n += 16; ui >>= 16; }
	if( (ui&0x00FF) == 0 ) { n +=  8; ui >>=  8; }
	if( (ui&0x000F) == 0 ) { n +=  4; ui >>=  4; }
	if( (ui&0x0003) == 0 ) { n +=  2; ui >>=  2; }
	n -= ui & 0x1;
#else
	if( (ui>>16) == 0 ) { n += 16; ui <<= 16; }
	if( (ui>>24) == 0 ) { n +=  8; ui <<=  8; }
	if( (ui>>28) == 0 ) { n +=  4; ui <<=  4; }
	if( (ui>>30) == 0 ) { n +=  2; ui <<=  2; }
	n -= (IrisS32)((ui>>31) & 0x1);
#endif
	return n;
}

/**
 * @brief	MSB1Zo
 *			Count Leading One
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_clo(IrisArgU32 val)
{
	return iris_allegrex_clz(~val);
}

/**
 * @brief	LSB0Zo
 *			Count Trailing Zero
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_ctz(IrisArgU32 val)
{
	IrisS32 n = 1;
	IrisU32 ui = val;
	if( ui == 0 ) return 32;
#ifdef __BIG_ENDIAN__
	if( (ui>>16) == 0 ) { n += 16; ui <<= 16; }
	if( (ui>>24) == 0 ) { n +=  8; ui <<=  8; }
	if( (ui>>28) == 0 ) { n +=  4; ui <<=  4; }
	if( (ui>>30) == 0 ) { n +=  2; ui <<=  2; }
	n -= (ui>>31) & 0x1;
#else
	if( (ui&0xFFFF) == 0 ) { n += 16; ui >>= 16; }
	if( (ui&0x00FF) == 0 ) { n +=  8; ui >>=  8; }
	if( (ui&0x000F) == 0 ) { n +=  4; ui >>=  4; }
	if( (ui&0x0003) == 0 ) { n +=  2; ui >>=  2; }
	n -= (IrisS32)(ui & 0x1);
#endif
	return n;
}

/**
 * @brief	LSB1Zo
 *			Count Trailing One
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_cto(IrisArgU32 val)
{
	return iris_allegrex_ctz(~val);
}

/**
 * @brief	max
 * @param	val1 [in]	= l
 * @param	val2 [in]	= l
 * @return	傫̒l
*/
STATICINLINE	IrisS32		iris_allegrex_max(IrisArgS32 val1, IrisArgS32 val2)
{
	return (val1 > val2) ? val1 : val2;
}

/**
 * @brief	min
 * @param	val1 [in]	= l
 * @param	val2 [in]	= l
 * @return	̒l
*/
STATICINLINE	IrisS32		iris_allegrex_min(IrisArgS32 val1, IrisArgS32 val2)
{
	return (val1 < val2) ? val1 : val2;
}

/**
 * @brief	bit 7g
 *			Sign-Extend Byte
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_seb(IrisArgS8 val)
{
	return (IrisS32)val;
}

/**
 * @brief	bit 15g
 *			Sign-Extend Half
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisS32		iris_allegrex_seh(IrisArgS16 val)
{
	return (IrisS32)val;
}

/**
 * @brief	oCgPʂŃ[hXbv
 *			Word Swap Byte within Word
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisU32		iris_allegrex_wsbw(IrisArgU32 val)
{
	return (IrisU32)( ((val&0xFF) << 24) | ((val&0xFF00) << 8) | ((val>>8) & 0xFF00) | ((val>>24) & 0xFF) );
}

/**
 * @brief	oCgPʂŃn[t[hXbv
 *			Word Swap Byte within Halfword
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisU32		iris_allegrex_wsbh(IrisArgU32 val)
{
	return (IrisU32)( ((val<<8) & 0xFF00FF00) | ((val>>8) & 0x00FF00FF) );
}

/**
 * @brief	rbgPʂŃ[hXbv
 *			Bit Reverse
 * @param	val [in]	= l
 * @return	
*/
STATICINLINE	IrisU32		iris_allegrex_bitrev(IrisArgU32 val)
{
	IrisU32 n = 0;
	for( int i=0; i < 32; ++i ) n |= ((val>>i) & 0x1) << (31-i);
	return n;
}

IRIS_PRAGMA_INTRINSIC_BEGIN(_lrotl)
IRIS_PRAGMA_INTRINSIC_BEGIN(_lrotr)

/**
 * @brief	Enrbg]
 * @param [in] x = l
 * @param [in] n = Vtg
 * @return	]̒l
*/
STATICINLINE	IrisU32		iris_allegrex_rotr(IrisArgU32 x, IrisArgInt n)
{
#ifdef _MSC_VER
	return ::_lrotr(x, n);
#else
	return IRIS_rotr32(x, n);
#endif
}

/**
 * @brief	nrbg]
 * @param [in] x = l
 * @param [in] n = Vtg
 * @return	]̒l
*/
STATICINLINE	IrisU32		iris_allegrex_rotl(IrisArgU32 x, IrisArgInt n)
{
#ifdef _MSC_VER
	return ::_lrotl(x, n);
#else
	return IRIS_rotl32(x, n);
#endif
}

IRIS_PRAGMA_INTRINSIC_END(_lrotl)
IRIS_PRAGMA_INTRINSIC_END(_lrotr)

/**
 * @brief	rbg̔o
 * @param [in] val	= l
 * @param [in] pos	= oJnʒu
 * @param [in] size	= oJnʒu
 * @return	valposrbgʒusizerbg𔲂oAEl
*/
STATICINLINE	IrisU32		iris_allegrex_ext(IrisArgU32 val, IrisArgU32 pos, IrisArgU32 size)
{
	return (IrisU32)( ((val>>pos)<<(32 - size)) >> (32-size) );
}

/**
 * @brief	rbg̑}
 * @param [in] val1	= l
 * @param [in] val2	= }l
 * @param [in] pos	= oJnʒu
 * @param [in] size	= oJnʒu
 * @return	val2̉sizerbgval1posɑ}
*/
STATICINLINE	IrisU32		iris_allegrex_ins(IrisArgU32 val1, IrisArgU32 val2, IrisArgU32 pos, IrisArgU32 size)
{
	IrisU32 mask = (IrisU32)(((0x1 << (size+1)) - 1) << pos);
	return (IrisU32)( (val1 & ~mask) | ((val2 << pos) & mask) );
}

/**
 * @brief	2ׂ̂悩f
 * @param [in]	val	= l
 * @return	^Ul
*/
STATICINLINE	IrisBool	iris_allegrex_ispow2(IrisArgU32 val)
{
	return (val & (val - 1)) != 0 ? IRIS_TRUE : IRIS_FALSE;
}

/**
 * @brief	t
 * @param [in]	fs	= 
 * @return	
*/
STATICINLINE	IrisF32		iris_allegrex_rcpf(IrisArgF32 fs)
{
	IrisFInt fi;
	if( (*(IrisS32*)&fs & 0x7FFFFFFF) == 0 ) return MATH_F32_P_INF_BITS;
	fi.iv = (IrisU32)(0x7F000000 - *(IrisS32*)&fs);
	fi.fv *= (2.0f - (fi.fv * fs));
	fi.fv *= (2.0f - (fi.fv * fs));
	return fi.fv * (2.0f - (fi.fv * fs));
}

/**
 * @brief	floorf
 * @param [in]	fs	= 
 * @return	
*/
STATICINLINE	IrisF32		iris_allegrex_floorf(IrisArgF32 fs)
{
#if 1
	return (IrisF32)(IrisS32)fs;
#else
	IrisS32	bias = 0xBF7FFFFF & ((*(IrisS32*)&fs) >> 31);
	return (IrisF32)( (IrisS32)( fs + (*(IrisF32*)&bias) ) );
#endif
}

/**
 * @brief	ceilf
 * @param [in]	fs	= 
 * @return	
*/
STATICINLINE	IrisF32		iris_allegrex_ceilf(IrisArgF32 fs)
{
	IrisS32	bias = 0x3F7FFFFF & (((*(IrisS32*)&fs) ^ (IrisS32)0x80000000) >> 31);
	return (IrisF32)( (IrisS32)( fs + (*(IrisF32*)&bias) ) );
}

/**
 * @brief	sqrtf
 * @param	[in]	fs	= 
 * @return	
*/
STATICINLINE	IrisF32		iris_allegrex_sqrtf(IrisArgF32 fs)
{
	if( *(IrisS32*)&fs <= 0 )
	{
		if( (*(IrisS32*)&fs & 0x7FFFFFFF) == 0 ) return fs;
		IrisFInt r;
		r.iv = MATH_F32_N_QNAN_BITS;
		return r.fv;
	}
	IrisFInt x, y;
	x.iv = *(IrisU32*)&fs + 0xFF800000;
	y.iv = 0x5F3759DF - (*(IrisS32*)&fs >> 1);

	y.fv *= (1.5f - ( y.fv * y.fv * x.fv ));
	y.fv *= (1.5f - ( y.fv * y.fv * x.fv ));
	return (y.fv * (1.5f - (y.fv * y.fv * x.fv))) * fs;
}

/**
 * @brief	rsqrtf
 * @param	[in]	fs	= 
 * @return	
*/
STATICINLINE	IrisF32		iris_allegrex_rsqrtf(IrisArgF32 fs)
{
	if( *(IrisS32*)&fs & 0x80000000 )
	{
		IrisFInt r;
		r.iv = MATH_F32_N_INF_BITS;
		return r.fv;
	}
	IrisFInt x, y;
	x.iv = *(IrisU32*)&fs + 0xFF800000;
	y.iv = (IrisU32)(0x5F3759DF - (*(IrisS32*)&fs >> 1));

	y.fv *= (1.5f - ( y.fv * y.fv * x.fv ));
	y.fv *= (1.5f - ( y.fv * y.fv * x.fv ));
	return (y.fv * (1.5f - (y.fv * y.fv * x.fv)));
}

/**
 * @brief	hypot
 *			Moler-Morrison Algorithm ł̎
 * @param	[in]	fa	= 
 * @param	[in]	fb	= 
 * @return	a*a + b*b
*/
STATICINLINE	IrisF32		iris_allegrex_hypotf(IrisArgF32 fa, IrisArgF32 fb)
{
	//IrisF32 a = fabsf(fa);
	//IrisF32 b = fabsf(fb);
	IrisS32 ia = *(IrisS32*)&fa & 0x7FFFFFFF;
	IrisS32 ib = *(IrisS32*)&fb & 0x7FFFFFFF;
	if( ia < ib )
	{
		IrisS32 x = ia;
		ia = ib;
		ib = x;
	}
	IrisF32 a = *(IrisF32*)&ia;
	IrisF32 b = *(IrisF32*)&ib;
	if( ib == 0 ) return a;
	IrisF32 s;
	for( int i=0; i < 3; ++i )
	{
		s = b/a;
		s *= s;
		s /= 4+s;
		a += 2*a*s;
		b *= s;
	}
	return a;
}


EXTERN_C_END

#endif
