//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathSHA512.cpp
 * @brief		SHA512 t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_MathSHA512_CPP_
#define INCG_IRIS_MathSHAPrivate_CPP_	// private֐gp

//======================================================================
// include
#include "MathSHA512.h"
#include "MathSHACommon.h"
#include "../../misc/iris_allegrex.h"
#include "../../iris_debug.h"

namespace iris {
namespace math
{

//======================================================================
// define
#define MESSAGE_BLK_SIZE	SHA512_MSGBLKSIZE
#define MESSAGE_PAD_SIZE	112	// SHA512_MSGBLKSIZE-SHA512_LENGTHBUFSIZE

#ifdef _IRIS_NOT_SUPPORT_LLONG
// 32bit
#define SHTR(lh, x, n)	( \
	(lh)[0] = (((n) < 32) && ((n) >= 0)) ?				\
	((x)[0] >> (n)) : 0,								\
	(lh)[1] = ((n) > 32) ? ((x)[0] >> ((n) - 32)) :		\
	((n) == 32) ? (x)[0] :								\
	((n) >= 0) ?										\
	(((x)[0] << (32 - (n))) |							\
	((x)[1] >> (n))) : 0 )
#define SHTL(lh, x, n)	( \
	(lh)[0] = ((n) > 32) ? ((x)[1] << ((n) - 32)) :		\
	((n) == 32) ? (x)[1] :								\
	((n) >= 0) ?										\
	(((x)[0] << (n)) |									\
	((x)[1] >> (32 - (n)))) :							\
	0,													\
	(lh)[1] = (((n) < 32) && ((n) >= 0)) ?				\
	((x)[1] << (n)) : 0 )
	
#define SHA512_OR(lh, rh1, rh2)	(	\
	(lh)[0] = (rh1)[0] | (rh2)[0],	\
	(lh)[1] = (rh1)[1] | (rh2)[1] )

#define SHA512_XOR(lh, rh1, rh2) (	\
	(lh)[0] = (rh1)[0] ^ (rh2)[0],	\
	(lh)[1] = (rh1)[1] ^ (rh2)[1] )

#define ROTR(lh, rh, n, t1, t2)		\
	SHTR(t1, (rh), (n));			\
	SHTL(t2, (rh), (64-(n)));		\
	SHA512_OR((lh), t1, t2)

#define _SHA512Encode	_SHAEncode32

#define SHA512AddLength(ctx, len)		\
	(ctx)->uCount[3] += len				\
	, ((ctx)->uCount[3] != 0 ) ? 0 :	\
	(++(ctx)->uCount[2]					\
	, ((ctx)->uCount[2] != 0 ) ? 0 :	\
	(++(ctx)->uCount[1]					\
	, ((ctx)->uCount[1] != 0 ) ? 0 :	\
	(++(ctx)->uCount[0]					\
	, ((ctx)->uCount[0] != 0 ) ? 0 : 1 )))
#else

// 64bit
#define SHTR(x, n)	((x) >> n)
#define ROTL	IRIS_rotl
#define ROTR	IRIS_rotr

#define _SHA512Encode	_SHAEncode64

#define SHA512AddLength(ctx, len)		\
	(ctx)->uCount[1] += len				\
	, ((ctx)->uCount[1] != 0 ) ? 0 :	\
	(++(ctx)->uCount[0]					\
	, ((ctx)->uCount[0] != 0 ) ? 0 : 1)
#endif


//======================================================================
// declare
// gXtH[֐
#ifdef _IRIS_NOT_SUPPORT_LLONG
static void _SHA512Transform(u32* lpState, const u8* lpBuffer);
#else
static void _SHA512Transform(u64* lpState, const u8* lpBuffer);
#endif

//======================================================================
// function
/**********************************************************************//**
 *
 * SHA512pReLXg̏
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @return	
*//***********************************************************************/
void	SHA512InitContext(LPSHA512CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	lpContext->uIndex = 0;
#ifdef _IRIS_NOT_SUPPORT_LLONG
	lpContext->uCount[0] = lpContext->uCount[1] = 
		lpContext->uCount[2] = lpContext->uCount[3] = 0;
	lpContext->uState[ 0] = 0x6A09E667;
	lpContext->uState[ 1] = 0xF3BCC908;
	lpContext->uState[ 2] = 0xBB67AE85;
	lpContext->uState[ 3] = 0x84CAA73B;
	lpContext->uState[ 4] = 0x3C6EF372;
	lpContext->uState[ 5] = 0xFE94F82B;
	lpContext->uState[ 6] = 0xA54FF53A;
	lpContext->uState[ 7] = 0x5F1D36F1;
	lpContext->uState[ 8] = 0x510E527F;
	lpContext->uState[ 9] = 0xADE682D1;
	lpContext->uState[10] = 0x9B05688C;
	lpContext->uState[11] = 0x2B3E6C1F;
	lpContext->uState[12] = 0x1F83D9AB;
	lpContext->uState[13] = 0xFB41BD6B;
	lpContext->uState[14] = 0x5BE0CD19;
	lpContext->uState[15] = 0x137E2179;
#else
	lpContext->uCount[0] = lpContext->uCount[1] = 0;
	lpContext->uState[ 0] = 0x6A09E667F3BCC908ll;
	lpContext->uState[ 1] = 0xBB67AE8584CAA73Bll;
	lpContext->uState[ 2] = 0x3C6EF372FE94F82Bll;
	lpContext->uState[ 3] = 0xA54FF53A5F1D36F1ll;
	lpContext->uState[ 4] = 0x510E527FADE682D1ll;
	lpContext->uState[ 5] = 0x9B05688C2B3E6C1Fll;
	lpContext->uState[ 6] = 0x1F83D9ABFB41BD6Bll;
	lpContext->uState[ 7] = 0x5BE0CD19137E2179ll;
#endif
	//ZeroMemory(lpContext->uBuffer, sizeof(lpContext->uBuffer));
}

/**********************************************************************//**
 *
 * SHA512pReLXg̃NA
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @return	
*//***********************************************************************/
void	SHA512ClearContext(LPSHA512CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	SHA512InitContext(lpContext);
	ZeroMemory(lpContext->uBuffer, sizeof(lpContext->uBuffer));
}

/**********************************************************************//**
 *
 * SHA512vZ
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA512Update(LPSHA512CONTEXT lpContext, const u8* lpBuffer, size_t uLength)
{
	IRIS_ASSERT( lpContext != nullptr );
	IRIS_ASSERT( lpBuffer != nullptr );
	const u8* buf = lpBuffer;
	u16 index = lpContext->uIndex;
	while( uLength-- )
	{
		lpContext->uBuffer[index++] = *buf;
		if( index == MESSAGE_BLK_SIZE )
		{
			_SHA512Transform(lpContext->uState, lpContext->uBuffer);
			index = 0;
		}
		if( SHA512AddLength(lpContext, 8) )
		{
			IRIS_WARNING("message is too long.");
			break;
		}
		++buf;
	}
	lpContext->uIndex = index;
}

/**********************************************************************//**
 *
 * ŏIISHA512vZ
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
*//***********************************************************************/
void	SHA512Final(LPSHA512CONTEXT lpContext)
{
	IRIS_ASSERT( lpContext != nullptr );
	u16 index = lpContext->uIndex;
	if( index >= MESSAGE_PAD_SIZE )
	{
		lpContext->uBuffer[index++] = 0x80;
		while( index < MESSAGE_BLK_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
		_SHA512Transform(lpContext->uState, lpContext->uBuffer);
		index = 0;
		while( index < MESSAGE_PAD_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
	}
	else
	{
		lpContext->uBuffer[index++] = 0x80;
		while( index < MESSAGE_PAD_SIZE )
		{
			lpContext->uBuffer[index++] = 0;
		}
	}
	_SHA512Encode(lpContext->uBuffer+MESSAGE_PAD_SIZE, lpContext->uCount, SHA512_LENGTHBUFSIZE);
	_SHA512Transform(lpContext->uState, lpContext->uBuffer);
	lpContext->uIndex = 0;
}

/**********************************************************************//**
 *
 * SHA512o
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [out]	lpBuffer	= o̓obt@
*//***********************************************************************/
void	SHA512Output(LPCSHA512CONTEXT lpContext, u8* lpBuffer)
{
	_SHA512Encode(lpBuffer, lpContext->uState, SHA512_HASHSIZE);
}

/**********************************************************************//**
 *
 * SHA512܂Ƃ߂ČvZ(Init ` Final)
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpContext	= ReLXg
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA512Encode(LPSHA512CONTEXT lpContext, const u8* lpBuffer, size_t uLength)
{
	SHA512InitContext(lpContext);
	SHA512Update(lpContext, lpBuffer, uLength);
	SHA512Final(lpContext);
}

/**********************************************************************//**
 *
 * SHA512܂Ƃ߂ČvZ(Init ` Output)
 * 
 ----------------------------------------------------------------------
 * @param [io]	lpDst		= o̓obt@
 * @param [in]	lpBuffer	= ̓obt@
 * @param [in]	uLength		= ̓obt@TCY
*//***********************************************************************/
void	SHA512Encode(u8* lpDst, const u8* lpBuffer, size_t uLength)
{
	SHA512CONTEXT ctx;
	SHA512InitContext(&ctx);
	SHA512Update(&ctx, lpBuffer, uLength);
	SHA512Final(&ctx);
	SHA512Output(&ctx, lpDst);
}

/**********************************************************************//**
 *
 * SHA512o͒l𕶎ɕϊ
 * 
 ----------------------------------------------------------------------
 * @param [out]	lpString	= o̓obt@
 * @param [in]	uSize		= o̓obt@TCY
 * @param [in]	lpMD5		= ̓obt@(MD5)
*//***********************************************************************/
LPTSTR	SHA512ToString (LPTSTR lpString, size_t uSize, const u8* lpSHA512)
{
	return SHAToString(lpString, uSize, lpSHA512, SHA512_HASHSIZE);
}
LPSTR	SHA512ToStringA(LPSTR  lpString, size_t uSize, const u8* lpSHA512)
{
	return SHAToStringA(lpString, uSize, lpSHA512, SHA512_HASHSIZE);
}
LPWSTR	SHA512ToStringW(LPWSTR lpString, size_t uSize, const u8* lpSHA512)
{
	return SHAToStringW(lpString, uSize, lpSHA512, SHA512_HASHSIZE);
}

// transform
#ifdef _IRIS_NOT_SUPPORT_LLONG
void _SHA512Transform(u32* lpState, const u8* lpBuffer)
{
	static const u32 K[160] = 
	{
		  0x428A2F98, 0xD728AE22, 0x71374491, 0x23EF65CD, 0xB5C0FBCF, 0xEC4D3B2F, 0xE9B5DBA5, 0x8189DBBC
		, 0x3956C25B, 0xF348B538, 0x59F111F1, 0xB605D019, 0x923F82A4, 0xAF194F9B, 0xAB1C5ED5, 0xDA6D8118
		, 0xD807AA98, 0xA3030242, 0x12835B01, 0x45706FBE, 0x243185BE, 0x4EE4B28C, 0x550C7DC3, 0xD5FFB4E2
		, 0x72BE5D74, 0xF27B896F, 0x80DEB1FE, 0x3B1696B1, 0x9BDC06A7, 0x25C71235, 0xC19BF174, 0xCF692694
		, 0xE49B69C1, 0x9EF14AD2, 0xEFBE4786, 0x384F25E3, 0x0FC19DC6, 0x8B8CD5B5, 0x240CA1CC, 0x77AC9C65
		, 0x2DE92C6F, 0x592B0275, 0x4A7484AA, 0x6EA6E483, 0x5CB0A9DC, 0xBD41FBD4, 0x76F988DA, 0x831153B5
		, 0x983E5152, 0xEE66DFAB, 0xA831C66D, 0x2DB43210, 0xB00327C8, 0x98FB213F, 0xBF597FC7, 0xBEEF0EE4
		, 0xC6E00BF3, 0x3DA88FC2, 0xD5A79147, 0x930AA725, 0x06CA6351, 0xE003826F, 0x14292967, 0x0A0E6E70
		, 0x27B70A85, 0x46D22FFC, 0x2E1B2138, 0x5C26C926, 0x4D2C6DFC, 0x5AC42AED, 0x53380D13, 0x9D95B3DF
		, 0x650A7354, 0x8BAF63DE, 0x766A0ABB, 0x3C77B2A8, 0x81C2C92E, 0x47EDAEE6, 0x92722C85, 0x1482353B
		, 0xA2BFE8A1, 0x4CF10364, 0xA81A664B, 0xBC423001, 0xC24B8B70, 0xD0F89791, 0xC76C51A3, 0x0654BE30
		, 0xD192E819, 0xD6EF5218, 0xD6990624, 0x5565A910, 0xF40E3585, 0x5771202A, 0x106AA070, 0x32BBD1B8
		, 0x19A4C116, 0xB8D2D0C8, 0x1E376C08, 0x5141AB53, 0x2748774C, 0xDF8EEB99, 0x34B0BCB5, 0xE19B48A8
		, 0x391C0CB3, 0xC5C95A63, 0x4ED8AA4A, 0xE3418ACB, 0x5B9CCA4F, 0x7763E373, 0x682E6FF3, 0xD6B2B8A3
		, 0x748F82EE, 0x5DEFB2FC, 0x78A5636F, 0x43172F60, 0x84C87814, 0xA1F0AB72, 0x8CC70208, 0x1A6439EC
		, 0x90BEFFFA, 0x23631E28, 0xA4506CEB, 0xDE82BDE9, 0xBEF9A3F7, 0xB2C67915, 0xC67178F2, 0xE372532B
		, 0xCA273ECE, 0xEA26619C, 0xD186B8C7, 0x21C0C207, 0xEADA7DD6, 0xCDE0EB1E, 0xF57D4F7F, 0xEE6ED178
		, 0x06F067AA, 0x72176FBA, 0x0A637DC5, 0xA2C898A6, 0x113F9804, 0xBEF90DAE, 0x1B710B35, 0x131C471B
		, 0x28DB77F5, 0x23047D84, 0x32CAAB7B, 0x40C72493, 0x3C9EBE0A, 0x15C9BEBC, 0x431D67C4, 0x9C100D4C
		, 0x4CC5D4BE, 0xCB3E42B6, 0x597F299C, 0xFC657E2A, 0x5FCB6FAB, 0x3AD6FAEC, 0x6C44198C, 0x4A475817
	};
#define SIGMA0(lh, w, t1, t2, t3, t4)	\
	ROTR(lh, w, 28, t2, t3);			\
	ROTR(t1, w, 34, t2, t3);			\
	ROTR(t2, w, 39, t3, t4);			\
	SHA512_XOR(t3, lh, t1);				\
	SHA512_XOR(lh, t2, t3)

#define SIGMA1(lh, w, t1, t2, t3, t4)	\
	ROTR(lh, w, 14, t2, t3);			\
	ROTR(t1, w, 18, t2, t3);			\
	ROTR(t2, w, 41, t3, t4);			\
	SHA512_XOR(t3, lh, t1);				\
	SHA512_XOR(lh, t2, t3)

#define sigma0(lh, w, t1, t2, t3)	\
	ROTR(lh, w,  1, t2, t3);		\
	ROTR(t1, w,  8, t2, t3);		\
	SHTR(t2, w,  7);				\
	SHA512_XOR(t3, lh, t1);			\
	SHA512_XOR(lh, t2, t3)

#define sigma1(lh, w, t1, t2, t3)	\
	ROTR(lh, w, 19, t2, t3);		\
	ROTR(t1, w, 61, t2, t3);		\
	SHTR(t2, w,  6);				\
	SHA512_XOR(t3, lh, t1);			\
	SHA512_XOR(lh, t2, t3)

#define F1(lh, x, y, z)	\
	(lh)[0] = (((x)[0] & ((y)[0] ^ (z)[0])) ^ (z)[0]);	\
	(lh)[1] = (((x)[1] & ((y)[1] ^ (z)[1])) ^ (z)[1])
#define F2(lh, x, y, z)	\
	(lh)[0] = (((x)[0] & ((y)[0] | (z)[0])) | ((y)[0] & (z)[0]));	\
	(lh)[1] = (((x)[1] & ((y)[1] | (z)[1])) | ((y)[1] & (z)[1]))

#define SHA512_COPY2(lh, rh)			\
	(lh)[0] = (rh)[0];					\
	(lh)[1] = (rh)[1]
#define SHA512_ADD(lh, rh, add)			\
	(lh)[1] = (rh)[1] + (add)[1];		\
	(lh)[0] = (rh)[0] + (add)[0] + ((lh)[1] < (rh)[1])

#define SHA512_ADDTO2(lh, rh)			\
	tmp = (lh)[1];						\
	(lh)[1] += rh[1];					\
	(lh)[0] += rh[0] + ((lh)[1] < tmp)

#pragma IRIS_WARNING_DISABLE(4293)
	int i=0, i2=0;
	u32 W[160];
	u32 A[2] = { *(lpState   ), *(lpState+ 1) };
	u32 B[2] = { *(lpState+ 2), *(lpState+ 3) };
	u32 C[2] = { *(lpState+ 4), *(lpState+ 5) };
	u32 D[2] = { *(lpState+ 6), *(lpState+ 7) };
	u32 E[2] = { *(lpState+ 8), *(lpState+ 9) };
	u32 F[2] = { *(lpState+10), *(lpState+11) };
	u32 G[2] = { *(lpState+12), *(lpState+13) };
	u32 H[2] = { *(lpState+14), *(lpState+15) };
	u32 tmp1[2], tmp2[2], tmp3[2], tmp;
	u32 stmp1[2], stmp2[2], stmp3[2], stmp4[2];

	for( ; i < 16; ++i, ++i2 )
	{
		W[i2  ] = IRIS_ByteArray2DWordBE(lpBuffer+i2*4);
		W[++i2] = IRIS_ByteArray2DWordBE(lpBuffer+i2*4);
	}
	for( ; i < 80; ++i, i2+=2 )
	{
		u32* W2  = &W[i2-4];
		u32* W7  = &W[i2-14];
		u32* W15 = &W[i2-30];
		u32* W16 = &W[i2-32];
		sigma1(tmp1, W2, stmp1, stmp2, stmp3);
		SHA512_ADD(tmp2, tmp1, W7);
		sigma0(tmp1, W15, stmp1, stmp2, stmp3);
		SHA512_ADD(tmp3, tmp1, W16);
		SHA512_ADD(&W[i2], tmp2, tmp3);
	}

	for( i=0, i2=0; i < 80; ++i, i2+=2 )
	{
		//tmp1 = H + SIGMA1(E) + F1(E, F, G) + W[i] + K[i];
		SIGMA1(tmp2, E, stmp1, stmp2, stmp3, stmp4);
		SHA512_ADD(tmp3, H, tmp2);
		F1(tmp2, E, F, G);
		SHA512_ADD(stmp1, &W[i2], &K[i2]);
		SHA512_ADD(stmp2, tmp2, tmp3);
		SHA512_ADD(tmp1, stmp1, stmp2);

		//tmp2 = SIGMA0(A) + F2(A, B, C);
		SIGMA0(tmp3, A, stmp1, stmp2, stmp3, stmp4);
		F2(stmp1, A, B, C);
		SHA512_ADD(tmp2, tmp3, stmp1);

		SHA512_COPY2(H, G);
		SHA512_COPY2(G, F);
		SHA512_COPY2(F, E);
		SHA512_ADD(E, D, tmp1);
		SHA512_COPY2(D, C);
		SHA512_COPY2(C, B);
		SHA512_COPY2(B, A);
		SHA512_ADD(A, tmp1, tmp2);
	}

	SHA512_ADDTO2((lpState   ), A);
	SHA512_ADDTO2((lpState+ 2), B);
	SHA512_ADDTO2((lpState+ 4), C);
	SHA512_ADDTO2((lpState+ 6), D);
	SHA512_ADDTO2((lpState+ 8), E);
	SHA512_ADDTO2((lpState+10), F);
	SHA512_ADDTO2((lpState+12), G);
	SHA512_ADDTO2((lpState+14), H);

#pragma IRIS_WARNING_DEFAULT(4293)
}
#else
void _SHA512Transform(u64* lpState, const u8* lpBuffer)
{
	static const u64 K[80] = 
	{
		  0x428A2F98D728AE22ll, 0x7137449123EF65CDll, 0xB5C0FBCFEC4D3B2Fll, 0xE9B5DBA58189DBBCll
		, 0x3956C25BF348B538ll, 0x59F111F1B605D019ll, 0x923F82A4AF194F9Bll, 0xAB1C5ED5DA6D8118ll
		, 0xD807AA98A3030242ll, 0x12835B0145706FBEll, 0x243185BE4EE4B28Cll, 0x550C7DC3D5FFB4E2ll
		, 0x72BE5D74F27B896Fll, 0x80DEB1FE3B1696B1ll, 0x9BDC06A725C71235ll, 0xC19BF174CF692694ll
		, 0xE49B69C19EF14AD2ll, 0xEFBE4786384F25E3ll, 0x0FC19DC68B8CD5B5ll, 0x240CA1CC77AC9C65ll
		, 0x2DE92C6F592B0275ll, 0x4A7484AA6EA6E483ll, 0x5CB0A9DCBD41FBD4ll, 0x76F988DA831153B5ll
		, 0x983E5152EE66DFABll, 0xA831C66D2DB43210ll, 0xB00327C898FB213Fll, 0xBF597FC7BEEF0EE4ll
		, 0xC6E00BF33DA88FC2ll, 0xD5A79147930AA725ll, 0x06CA6351E003826Fll, 0x142929670A0E6E70ll
		, 0x27B70A8546D22FFCll,	0x2E1B21385C26C926ll, 0x4D2C6DFC5AC42AEDll, 0x53380D139D95B3DFll
		, 0x650A73548BAF63DEll, 0x766A0ABB3C77B2A8ll, 0x81C2C92E47EDAEE6ll, 0x92722C851482353Bll
		, 0xA2BFE8A14CF10364ll, 0xA81A664BBC423001ll, 0xC24B8B70D0F89791ll, 0xC76C51A30654BE30ll
		, 0xD192E819D6EF5218ll, 0xD69906245565A910ll, 0xF40E35855771202All, 0x106AA07032BBD1B8ll
		, 0x19A4C116B8D2D0C8ll, 0x1E376C085141AB53ll, 0x2748774CDF8EEB99ll, 0x34B0BCB5E19B48A8ll
		, 0x391C0CB3C5C95A63ll, 0x4ED8AA4AE3418ACBll, 0x5B9CCA4F7763E373ll, 0x682E6FF3D6B2B8A3ll
		, 0x748F82EE5DEFB2FCll,	0x78A5636F43172F60ll, 0x84C87814A1F0AB72ll, 0x8CC702081A6439ECll
		, 0x90BEFFFA23631E28ll, 0xA4506CEBDE82BDE9ll, 0xBEF9A3F7B2C67915ll, 0xC67178F2E372532Bll
		, 0xCA273ECEEA26619Cll, 0xD186B8C721C0C207ll, 0xEADA7DD6CDE0EB1Ell, 0xF57D4F7FEE6ED178ll
		, 0x06F067AA72176FBAll, 0x0A637DC5A2C898A6ll, 0x113F9804BEF90DAEll, 0x1B710B35131C471Bll
		, 0x28DB77F523047D84ll, 0x32CAAB7B40C72493ll, 0x3C9EBE0A15C9BEBCll,	0x431D67C49C100D4Cll
		, 0x4CC5D4BECB3E42B6ll, 0x597F299CFC657E2All, 0x5FCB6FAB3AD6FAECll, 0x6C44198C4A475817ll
	};
#define SIGMA0(w)	(ROTR(w, 28) ^ ROTR(w, 34) ^ ROTR(w, 39))
#define SIGMA1(w)	(ROTR(w, 14) ^ ROTR(w, 18) ^ ROTR(w, 41))
#define sigma0(w)	(ROTR(w,  1) ^ ROTR(w,  8) ^ SHTR(w,  7))
#define sigma1(w)	(ROTR(w, 19) ^ ROTR(w, 61) ^ SHTR(w,  6))

#define F1(x, y, z)	_SHA_Ch(x, y, z)
#define F2(x, y, z)	_SHA_Maj(x, y, z)

	int i=0;
	u64 W[80];
	u64 A = *(lpState);
	u64 B = *(lpState+1);
	u64 C = *(lpState+2);
	u64 D = *(lpState+3);
	u64 E = *(lpState+4);
	u64 F = *(lpState+5);
	u64 G = *(lpState+6);
	u64 H = *(lpState+7);
	u64 tmp1, tmp2;

	for( ; i < 16; ++i )
	{
		W[i] = IRIS_ByteArray2QWordBE(lpBuffer+i*8);
	}
	for( ; i < 80; ++i )
	{
		W[i] = sigma1(W[i-2]) + W[i-7] + sigma0(W[i-15]) + W[i-16];
	}

	for( i=0; i < 80; ++i )
	{
		tmp1 = H + SIGMA1(E) + F1(E, F, G) + W[i] + K[i];
		tmp2 = SIGMA0(A) + F2(A, B, C);
		H = G;
		G = F;
		F = E;
		E = D + tmp1;
		D = C;
		C = B;
		B = A;
		A = tmp1 + tmp2;
	}

	*(lpState)   += A;
	*(lpState+1) += B;
	*(lpState+2) += C;
	*(lpState+3) += D;
	*(lpState+4) += E;
	*(lpState+5) += F;
	*(lpState+6) += G;
	*(lpState+7) += H;
}
#endif

}	// end of namespace math
}	// end of namespace iris


#if	(defined(_IRIS_SUPPORT_GOOGLETEST) || defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/gt/gt_inchead.h"
#include "../../iris_using.h"

TEST(CMathSHA512Test, Function)
{
	char comm[256] = "";
	u8 out[SHA512_HASHSIZE];

	SHA512CONTEXT ctx;

	// TEST1
	strcpy_s(comm, 256, "abc");
	SHA512Encode(out, comm, strlen(comm));
	SHA512ToStringA(comm, 256, out);
	ASSERT_STREQ( 
		"ddaf35a193617aba" "cc417349ae204131"
		"12e6fa4e89a97ea2" "0a9eeee64b55d39a"
		"2192992a274fc1a8" "36ba3c23a3feebbd"
		"454d4423643ce80e" "2a9ac94fa54ca49f"
		, comm );

	// TEST2
	strcpy_s(comm, 256, "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmn"
		"hijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu");
	SHA512Encode(out, comm, strlen(comm));
	SHA512ToStringA(comm, 256, out);
	ASSERT_STREQ( 
		"8e959b75dae313da" "8cf4f72814fc143f"
		"8f7779c6eb9f7fa1" "7299aeadb6889018"
		"501d289e4900f7e4" "331b99dec4b5433a"
		"c7d329eeb6dd2654" "5e96e55b874be909"
		, comm );

	// TEST3
	strcpy_s(comm, 256, "a");
	SHA512InitContext(&ctx);
	for( int i=0; i < 1000000; ++i )
		SHA512Update(&ctx, comm, strlen(comm));
	SHA512Final(&ctx);
	SHA512Output(&ctx, out);
	SHA512ToStringA(comm, 256, out);
	ASSERT_STREQ( 
		"e718483d0ce76964" "4e2e42c7bc15b463"
		"8e1f98b13b204428" "5632a803afa973eb"
		"de0ff244877ea60a" "4cb0432ce577c31b"
		"eb009c5c2c49aa2e" "4eadb217ad8cc09b"
		, comm );

	// TEST4
	strcpy_s(comm, 256, "01234567012345670123456701234567""01234567012345670123456701234567");
	SHA512InitContext(&ctx);
	for( int i=0; i < 10; ++i )
		SHA512Update(&ctx, comm, strlen(comm));
	SHA512Final(&ctx);
	SHA512Output(&ctx, out);
	SHA512ToStringA(comm, 256, out);
	ASSERT_STREQ( 
		"89d05ba632c699c3" "1231ded4ffc127d5"
		"a894dad412c0e024" "db872d1abd2ba814"
		"1a0f85072a9be1e2" "aa04cf33c765cb51"
        "0813a39cd5a84c4a" "caa64d3f3fb7bae9"
		, comm );
}

#endif

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_using.h"
#include "../../iris_iostream.h"

//======================================================================
// test
IRIS_UNITTEST(CMathSHA512UnitTest, MathSHA512UnitTest)
{
	char comm[256] = "";
	u8 out[SHA512_HASHSIZE];

	std::clog << "SHA512GR[h܂B" << std::endl;
	std::safe_cin >> comm;

	SHA512Encode(out, comm, strlen(comm));

	for( int i=0; i < SHA512_HASHSIZE; ++i )
		printf("%.2x", out[i]);
	std::cout << std::endl;
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

