//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		MathPrimeNumber.cpp
 * @brief		ft@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_MathPrimeNumber_CPP_

//======================================================================
// include
#include "MathPrimeNumber.h"

namespace iris {
namespace math
{

//======================================================================
// function

	
//======================================================================
// static
#if		defined(_IRIS_SUPPORT_PRIMENUMBER_TABLE_u32)
static u32	s_PrimeNum[] = {
	#include "prime_number16.inc"
	#include "prime_number32.inc"
};
#elif	defined(_IRIS_SUPPORT_PRIMENUMBER_TABLE_u16)
static u32	s_PrimeNum[] = { 
	#include "prime_number16.inc"
};
#endif

//======================================================================
// function

/**********************************************************************//**
 *
 * fe[u̎擾
 *
 -----------------------------------------------------------------------
 * @return fe[ũAhX
*//***********************************************************************/
u32* GetPrimeNumberTable(void)
{
#ifdef _IRIS_HAS_PRIME_TABLE
	return s_PrimeNum;
#else
	return nullptr;
#endif
}

/**********************************************************************//**
 *
 * fe[u̗vf擾
 *
 -----------------------------------------------------------------------
 * @return fe[u̗vf
*//***********************************************************************/
s32 GetPrimeNumberTableElements(void)
{
#ifdef _IRIS_HAS_PRIME_TABLE
	return IRIS_NumOfElements(s_PrimeNum);
#else
	return 0;
#endif
}

/**********************************************************************//**
 *
 * f̎擾
 *
 -----------------------------------------------------------------------
 * @param [in]	nIndex	= e[ũCfbNX
 * @return f
*//***********************************************************************/
u32	GetPrimeNumber(u32 nIndex)
{
#ifdef _IRIS_HAS_PRIME_TABLE
	if( nIndex > IRIS_NumOfElements(s_PrimeNum) ) return 0;
	return s_PrimeNum[nIndex];
#else
	IRIS_UNUSED_VARIABLE(nIndex);
	return 0;
#endif
}

/**********************************************************************//**
 *
 * f̌
 *
 -----------------------------------------------------------------------
 * @param [in]	num	= f
 * @return ꍇnumԂBꍇ͋ߎ̑fԂ
*//***********************************************************************/
u32	FindPrimeNumber(u32 num)
{
#ifdef _IRIS_HAS_PRIME_TABLE
	int size = IRIS_NumOfElements(s_PrimeNum);
	int s=0, e=size, n=0;
	while(s<=e)
	{
		n = s + ((e-s+1)>>1);
		const u32 temp = *(s_PrimeNum + n);
		if( temp == num )	return temp;
		if( temp <  num )	s = n+1;
		else				e = n-1;
	}
	return s_PrimeNum[e];
#else
	IRIS_UNUSED_VARIABLE(num);
	return 0;
#endif
}

/**********************************************************************//**
 *
 * f̔
 *
 -----------------------------------------------------------------------
 * @param [in]	num	= l
 * @return ^Ul
*//***********************************************************************/
BOOL IsPrimeNumber(u32 num)
{
	u32 n = num;
	u32 m = 3;
	if( n <= 1 ) return FALSE;
#ifdef _IRIS_HAS_PRIME_TABLE
	for( int i=0, sz=IRIS_NumOfElements(s_PrimeNum); i < sz; ++i )
	{
		m = s_PrimeNum[i];
		if( n == m ) return TRUE;
		if( n % m == 0 ) return FALSE;
		if( m * m > n ) return TRUE;
	}
	m += 2;
#else
	if( n == 2 ) return TRUE;
	if( n % 2 == 0 ) return FALSE;
#endif
	for( ; m * m <= n; m += 2 )
	{
		if( n % m == 0 ) return FALSE;
	}
	return TRUE;
}

}	// end of namespace math
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))
#include "../unit/UnitCore.h"
#include "../iris_iostream.h"
#include "../iris_using.h"

//#define SPEED_TEST	// xeXg
#ifdef SPEED_TEST
#include "../mpf/os/MpfSysTime.h"
#endif

//======================================================================
// test
IRIS_UNITTEST(CMathPrimeNumberUnitTest, MathPrimeNumberUnitTest)
{
#ifndef SPEED_TEST
	std::cout << "f܂B" << std::endl;
	while(1)
	{
		u32 x;
		std::cout << "l͂ĂB(0:ŏI)" << std::endl;
#ifdef _IRIS_SUPPORT_SAFE_ISTREAM
		std::safe_cin >> x;
#else
		std::cin >> x;
		if( std::cin.fail() )
		{
			std::cin.clear();
			std::cin.ignore(INT_MAX, '\n');
			std::cout << "lȊO͂܂B" << std::endl;
			continue;
		}
#endif

		if( x == 0 ) break;
		BOOL b = IsPrimeNumber(x);
		if( b )	std::cout << "prime" << std::endl;
		else	std::cout << "not prime" << std::endl;
	}
#else
	static const u32 MIN = 0x10001;
	static const u32 MAX = 0x100001;
	u64 cnt = 0;
	u64 min = (u64)-1;
	u64 max = 0;
	for(u32 i=MIN, j=0, k=0; i < MAX; i+=2, ++j )
	{
		u64	tick0 = SysGetTick();
		BOOL b = IsPrimeNumber(i);
		u64	tick1 = SysGetTick();
		u64 diff = SysDiffTick(tick1, tick0);
		cnt += diff;
		if( diff > max ) max = diff;
		if( diff < min ) min = diff;
		if( j > (MAX - MIN)/(100*20) ) { j = 0; ++k; printf("."); }
		if( k >= 20 ) { k = 0; printf("\n"); }
	}
	printf("\n");
	printf("ave %f microb\n", SysTicksToMicroSeconds(cnt) / (f32)(MAX-MIN) );
	printf("min %d microb\n", SysTicksToMicroSeconds(min) );
	printf("max %d microb\n", SysTicksToMicroSeconds(max) );
#endif
}

#endif
