//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndColor.h
 * @brief		J[ template t@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2010-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndColor_H_
#define INCG_IRIS_FndColor_H_

//======================================================================
// include
#include "../../iris_object.h"
#include "../../iris_debug.h"

namespace iris {
namespace fnd
{

//======================================================================
// declare
template<typename _TN, int _RBITS, int _GBITS, int _BBITS, int _ABITS
			, int _RSHIFT, int _GSHIFT, int _BSHIFT, int _ASHIFT>
class CColor;

//======================================================================
// typedef
typedef CColor<u32, 8, 8, 8, 8, 16, 8, 0, 24>	CARGB8888;
typedef CColor<u32, 8, 8, 8, 8, 24, 16, 8, 0>	CRGBA8888;
typedef CColor<u32, 8, 8, 8, 0, 0, 8, 16, 24>	CCOLORREF;

//======================================================================
// class
/**
 * @brief	J[ev[g
 * @details	(@ref section_autoexp_CColor "autoexp")
 * @tparam	_TN		= RGBA̐lێ^
 * @tparam	_RBITS	= R lbit
 * @tparam	_GBITS	= G lbit
 * @tparam	_BBITS	= B lbit
 * @tparam	_ABITS	= A lbit
 * @tparam	_RSHIFT	= R lւ̉EVtg
 * @tparam	_GSHIFT	= G lւ̉EVtg
 * @tparam	_BSHIFT	= B lւ̉EVtg
 * @tparam	_ASHIFT	= A lւ̉EVtg
*/
template<typename _TN, int _RBITS, int _GBITS, int _BBITS, int _ABITS
			, int _RSHIFT, int _GSHIFT, int _BSHIFT, int _ASHIFT>
class CColor : public IIrisObject
{
	typedef CColor<_TN, _RBITS, _GBITS, _BBITS, _ABITS, _RSHIFT, _GSHIFT, _BSHIFT, _ASHIFT>	_Myt;
public:
	typedef _TN		value_type;	//!< RGBA̐lێ^
	typedef enum
	{
		BSZ_R = _RBITS,			//!< R lbit
		BSZ_G = _GBITS,			//!< G lbit
		BSZ_B = _BBITS,			//!< B lbit
		BSZ_A = _ABITS,			//!< A lbit
		BSZ_ALL = BSZ_R + BSZ_G + BSZ_B + BSZ_A	//!< bit
	} BITSIZE;
	typedef enum
	{
		SFT_R	= _RSHIFT,		//!< R lւ̉EVtg
		SFT_G	= _GSHIFT,		//!< G lւ̉EVtg
		SFT_B	= _BSHIFT,		//!< B lւ̉EVtg
		SFT_A	= _ASHIFT		//!< A lւ̉EVtg
	} SHIFT;

private:
	value_type		m_col;	//!< J[l

	IRIS_STATIC_ASSERT( sizeof(value_type)*8 >= BSZ_ALL );

public:
	/**
	 * @name RXgN^
	 * @{
	*/
	CColor(void) : m_col(0) {}
	CColor(value_type col) : m_col(col) {}
	CColor(value_type _r, value_type _g, value_type _b, value_type _a) { r(_r); g(_g); b(_b); a(_a); }
	CColor(const IrisFColor& fcol) { copy(fcol); }
	template<typename _OTN, int _ORBITS, int _OGBITS, int _OBBITS, int _OABITS
				, int _ORSHIFT, int _OGSHIFT, int _OBSHIFT, int _OASHIFT>
	CColor(const CColor<_OTN, _ORBITS, _OGBITS, _OBBITS, _OABITS, _ORSHIFT, _OGSHIFT, _OBSHIFT, _OASHIFT>& rObj)
	{
		r(rObj.r());
		g(rObj.g());
		b(rObj.b());
		a(rObj.a());
	}
	/**
	 * @}
	*/

public:
	/**
	 * @name	e̍ől
	 * @{
	*/
	value_type		max_r(void)	const	{ return (((value_type)1<<BSZ_R)-1); }
	value_type		max_g(void)	const	{ return (((value_type)1<<BSZ_G)-1); }
	value_type		max_b(void)	const	{ return (((value_type)1<<BSZ_B)-1); }
	value_type		max_a(void)	const	{ return (((value_type)1<<BSZ_A)-1); }
	/**
	 * @}
	*/

public:
	/**
	 * @name	e̎擾
	 * @{
	*/
	value_type		r(void)		const	{ return (m_col >> SFT_R) & max_r(); }
	value_type		g(void)		const	{ return (m_col >> SFT_G) & max_g(); }
	value_type		b(void)		const	{ return (m_col >> SFT_B) & max_b(); }
	value_type		a(void)		const	{ return (m_col >> SFT_A) & max_a(); }
	/**
	 * @}
	*/

	/**
	 * @name	e̐ݒ
	 * @{
	*/
	void			r(value_type v)		{ value_type max = max_r(); m_col &= ~(max << SFT_R); m_col |= (v > max ? max : v) << SFT_R; }
	void			g(value_type v)		{ value_type max = max_g(); m_col &= ~(max << SFT_G); m_col |= (v > max ? max : v) << SFT_G; }
	void			b(value_type v)		{ value_type max = max_b(); m_col &= ~(max << SFT_B); m_col |= (v > max ? max : v) << SFT_B; }
	void			a(value_type v)		{ value_type max = max_a(); m_col &= ~(max << SFT_A); m_col |= (v > max ? max : v) << SFT_A; }
	/**
	 * @}
	*/

public:
	/// J[^ɕϊ
	void			cast(IrisFColor& rfcol)	const
	{
		rfcol.r = BSZ_R ? static_cast<f32>(r()) / max_r() : 0.f;
		rfcol.g = BSZ_G ? static_cast<f32>(g()) / max_g() : 0.f;
		rfcol.b = BSZ_B ? static_cast<f32>(b()) / max_b() : 0.f;
		rfcol.a = BSZ_A ? static_cast<f32>(a()) / max_a() : 0.f;	// 1.f;
	}

	/// J[^Rs[
	void			copy(const IrisFColor& rfcol)
	{
		r( (rfcol.r > 1.0f) ? max_r() : (rfcol.r < 0.0f) ? 0 : static_cast<value_type>(rfcol.r * max_r()) );
		g( (rfcol.g > 1.0f) ? max_g() : (rfcol.g < 0.0f) ? 0 : static_cast<value_type>(rfcol.g * max_g()) );
		b( (rfcol.b > 1.0f) ? max_b() : (rfcol.b < 0.0f) ? 0 : static_cast<value_type>(rfcol.b * max_b()) );
		a( (rfcol.a > 1.0f) ? max_a() : (rfcol.a < 0.0f) ? 0 : static_cast<value_type>(rfcol.a * max_a()) );
	}

public:
	operator value_type	(void)			{ return m_col; }
	operator IrisFColor	(void)			{ IrisFColor fcol; cast(fcol); return fcol; }
};

}	// end of namespace fnd
}	// end of namespace iris

#endif

/**
 * @addtogroup	autoexp
 * @{
 * @addtogroup	Visualizer
 * @section		section_autoexp_CColor	; iris::fnd::CColor
 * @code
;------------------------------------------------------------------------------
; iris::fnd::CColor
;------------------------------------------------------------------------------ 
iris::fnd::CColor<*,*,*,*,*,*,*,*,*> {
	children
	(
		#(
			R : (($e.m_col >> ($e.SFT_R)) & ((1 << ($e.BSZ_R))-1)),
			G : (($e.m_col >> ($e.SFT_G)) & ((1 << ($e.BSZ_G))-1)),
			B : (($e.m_col >> ($e.SFT_B)) & ((1 << ($e.BSZ_B))-1)),
			A : (($e.m_col >> ($e.SFT_A)) & ((1 << ($e.BSZ_A))-1)),
			m_col : $e.m_col
		)
	)
	preview
	(
		#(
			$e.m_col
		)
	)
}
 * @endcode
 * @}
*/
