//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndAllocator.h
 * @brief		AllocatorNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#ifndef INCG_IRIS_FndAllocator_H_
#define INCG_IRIS_FndAllocator_H_

//======================================================================
// include
#include "../../iris_object.h"
#include "../../iris_debug.h"
#include "stdlib.h"

namespace iris {
namespace fnd
{

//======================================================================
// declare
template<typename _TN>class CNewAllocator;
template<typename _TN>class CMemAllocator;
template<typename _TN, u32 MAX_INST>class CFixedAllocator;
template<typename _TN>class COperatorNewObject;

//======================================================================
// class
/**
 * @brief	AllocatorNX̃CX^XNX
 * @tparam	_Allocator = AP[^^Cv
*/
template<class _Allocator>
class CAllocatorInstance : public IIrisObject
{
public:
	typedef _Allocator	allocator_type;	//!< AP[^^Cv
public:
	//! AP[^CX^X̒
	static allocator_type& allocator(void) 
	{
		static allocator_type	this_allocator;
		return this_allocator;
	}
};

//! AllocatorC^[tFCX
class IAllocator : public INonCopyable<IAllocator>
{
public:
	virtual	void*	alloc(u32 size = 1, s32 arg=0)	= 0;	//!< m
	virtual	void	dealloc(void* ptr)	= 0;	//!< 
};

/**
 * @brief	new AP[^NX
 * @tparam	_TN	= AP[g^
*/
template<typename _TN>
class CNewAllocator : public IAllocator
{
public:
	typedef _TN				value_type;	//!< new ^
	typedef value_type		*value_ptr;	//!< |C^^
public:
	//! rebind
	template<typename _TN2> struct rebind 
	{
		typedef CNewAllocator<_TN2> other;
	};
public:
	/// m
	virtual	void*	alloc(u32 size = 1, s32 arg=0)	
	{ 
		(void)arg;
		return new value_type [size];
	}
	/// 
	virtual	void	dealloc(void* ptr)
	{ 
		delete [] (static_cast<value_ptr>(ptr));
	}
};

/**
 * @brief	malloc AP[^NX
 * @tparam	_TN	= AP[g^
*/
template<typename _TN>
class CMemAllocator : public IAllocator
{
public:
	typedef _TN				value_type;	//!< malloc ^
	typedef value_type		*value_ptr;	//!< |C^^
public:
	//! rebind
	template<typename _TN2> struct rebind 
	{
		typedef CMemAllocator<_TN2> other;
	};
public:
	/// m
	virtual	void*	alloc(u32 size = 1, s32 arg=0)	
	{ 
		(void)arg;
		return malloc(size*sizeof(value_type)); 
	}
	/// 
	virtual	void	dealloc(void* ptr)	
	{ 
		free(ptr);
	}
};

/**
 * @brief	fixed AP[^NX
 * @tparam	_TN			= AP[g^
 * @tparam	MAX_INST	= AP[g\ȍő吔
*/
template<typename _TN, u32 MAX_INST=32>
class CFixedAllocator : public IAllocator
{
public:
	typedef _TN				value_type;	//!< new ^
	typedef value_type		*value_ptr;	//!< |C^^
public:
	//! rebind
	template<typename _TN2> struct rebind 
	{
		typedef CFixedAllocator<_TN2, MAX_INST> other;
	};
protected:
	/**
	 * @private
	*/
	enum
	{
		PARTIAL_BITS	= 32,
		MAX_REFBIT		= (MAX_INST+PARTIAL_BITS-1)/PARTIAL_BITS,
		MAX_INNER_INST	= MAX_REFBIT*PARTIAL_BITS
	};
public:
	/**
	 * @brief	m
	 * @note	z̊mۂɑΉĂ܂BPÂmۂĂB
	*/
	void*	alloc(u32 size = 1, s32 arg=0)
	{
		IRIS_UNUSED_VARIABLE( size );
		IRIS_ASSERT( size == 1 );
		(void)arg;
		for( u32 i=0; i < MAX_REFBIT; ++i ) 
		{
			u32	bits = refbit(i);
			if( bits != 0xFFFFFFFF ) 
			{
				s32	shift_bit	= 0;
				if( !((bits & 0x0000FFFF) ^ 0x0000FFFF) ) 
				{
					shift_bit += 16;
					bits = (bits >> 16);
				}
				if( !((bits & 0x000000FF) ^ 0x000000FF) )
				{
					shift_bit += 8;
					bits = (bits >> 8);
				}
				if( !((bits & 0x0000000F) ^ 0x0000000F) ) 
				{
					shift_bit += 4;
					bits = (bits >> 4);
				}
				if( !((bits & 0x00000003) ^ 0x00000003) ) 
				{
					shift_bit += 2;
					bits = (bits >> 2);
				}
				if( !((bits & 0x00000001) ^ 0x00000001) ) 
				{
					shift_bit += 1;
				}

				refbit(i) |= (0x00000001 << shift_bit);
				return inst_ptr(i*PARTIAL_BITS+shift_bit);
			}
		}

		// Œ~MAX_INST𑝂₷
#ifdef _IRIS_DEBUG
		IRIS_ASSERTMSG( 0, "cant find free work!!" );
#endif
		return nullptr;
	}

	/// 
	void	dealloc(void* ptr) 
	{
		IRIS_ASSERT( (u32)(u32w64)ptr >= (u32)(u32w64)inst_ptr(0) );

		u32		inst_num = ((u32)(u32w64)ptr - (u32)(u32w64)inst_ptr(0)) / sizeof(value_type);

		refbit(inst_num/PARTIAL_BITS) &= ~(0x00000001<<(inst_num%PARTIAL_BITS));
	}
private:
	static	value_ptr	inst_ptr(u32 idx) 
	{
		static	value_type inst[MAX_INST];
		IRIS_ASSERT( idx < MAX_INST );
		return &inst[idx];
	}
	static	u32&		refbit( u32 idx ) 
	{
		static	u32 bits[MAX_REFBIT];
		IRIS_ASSERT( idx < MAX_REFBIT );
		return bits[idx];
	}
};

//! Global AP[^NX
class CGlobalAllocator : public IAllocator
{
private:
	IAllocator*		m_pAllocator;
public:
	/// RXgN^
	CGlobalAllocator(IAllocator* pDefAllocator) : m_pAllocator(pDefAllocator) {}
public:
	/// AP[^̕ύX
	void	SetAllocator(IAllocator* pAllocator)	{ m_pAllocator = pAllocator; }
public:
	/// m
	virtual	void*	alloc(u32 size = 1, s32 arg=0)	{ return m_pAllocator->alloc(size, arg); }
	/// 
	virtual	void	dealloc(void* ptr)				{ m_pAllocator->dealloc(ptr); }
};

//! operator new NX
class COperatorNew : public IIrisObject
{
	static const u32	ALIGNSIZE	= 32;
	static const u32	NEW_MAGIC	= 'NMGC';
public:
	void*	operator new(size_t size) { return ::operator new( size ); }
	void*	operator new(size_t size, IAllocator* allocator) 
	{
		u8* ptr = reinterpret_cast<u8*>(allocator->alloc( (u32)(size+ALIGNSIZE) ));
		*(u32*)&ptr[0] = NEW_MAGIC;
		*(IAllocator**)&ptr[4] = allocator;
		return ptr+ALIGNSIZE;
	}
	void*	operator new[](size_t size, IAllocator* allocator) 
	{
		u8* ptr = reinterpret_cast<u8*>(allocator->alloc( (u32)(size+ALIGNSIZE) ));
		*(u32*)&ptr[0] = NEW_MAGIC;
		*(IAllocator**)&ptr[4] = allocator;
		return ptr+ALIGNSIZE;
	}

	void	operator delete(void* ptr) 
	{
		if( ptr != nullptr )
		{
			void *head = reinterpret_cast<void*>(reinterpret_cast<u8*>(ptr)-ALIGNSIZE);
			u32  magic = *reinterpret_cast<u32*>(head);
			if( magic == NEW_MAGIC ) 
			{
				IAllocator* allocator = *reinterpret_cast<IAllocator**>(
					reinterpret_cast<u8*>(ptr)-ALIGNSIZE+sizeof(NEW_MAGIC) );
				allocator->dealloc(head);
			}
		}
	}
	void	operator delete(void* ptr, IAllocator*) 
	{
		operator delete( ptr );
	}
	void	operator delete[](void* ptr) 
	{
		if( ptr )
		{
			void *head = reinterpret_cast<void*>(reinterpret_cast<u8*>(ptr)-ALIGNSIZE);
			u32  magic = *(u32*)head;
			if( magic == NEW_MAGIC ) 
			{
				IAllocator* allocator = *reinterpret_cast<IAllocator**>(
					reinterpret_cast<u8*>(ptr)-ALIGNSIZE+sizeof(NEW_MAGIC) );
				allocator->dealloc(head);
			}
		}
	}
	void	operator delete[](void* ptr, IAllocator*) 
	{
		operator delete[]( ptr );
	}
};

/**
 * @brief	operator new (IAllocator* ) IuWFNgNX
 * @tparam	_TN	= AP[^IuWFNgɑΉNX
*/
template<class _TC>
class COperatorNewObject : public _TC, public COperatorNew
{
};

}	// end of namespace fnd
}	// end of namespace iris

#endif
