/*
 -------------------------------------------------------------------------
 Copyright (c) 2001, Dr Brian Gladman <                 >, Worcester, UK.
 All rights reserved.

 LICENSE TERMS

 The free distribution and use of this software in both source and binary 
 form is allowed (with or without changes) provided that:

   1. distributions of this source code include the above copyright 
      notice, this list of conditions and the following disclaimer;

   2. distributions in binary form include the above copyright
      notice, this list of conditions and the following disclaimer
      in the documentation and/or other associated materials;

   3. the copyright holder's name is not used to endorse products 
      built using this software without specific written permission. 

 DISCLAIMER

 This software is provided 'as is' with no explicit or implied warranties
 in respect of its properties, including, but not limited to, correctness 
 and fitness for purpose.
 -------------------------------------------------------------------------
 Issue Date: 29/07/2002
*/
//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndRijndael.cpp
 * @brief		RijndaelÍNXt@C
 *
 * @note		Brian Gladmanaessrc.zip̃R[hςĂ܂B
 *				http://www.gladman.me.uk/
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndRijndael_CPP_

//======================================================================
// include
#include "FndRijndael.h"
#include "../../iris_debug.h"

//======================================================================
// config
#define FM_TABLE	0	//!< fm_tbl
#define FT_TABLE	1	//!< ft_tbl
#define FL_TABLE	1	//!< fl_tbl
#define IM_TABLE	1	//!< im_tbl
#define IT_TABLE	1	//!< it_tbl
#define IL_TABLE	1	//!< il_tbl

namespace iris {
namespace fnd
{

//======================================================================
// define
// config
#ifdef _IRIS_RIJNDAEL_TABLE

#if	(_IRIS_RIJNDAEL_TABLE == 0 || _IRIS_RIJNDAEL_TABLE == 1 || _IRIS_RIJNDAEL_TABLE == 4)

#undef IM_TABLE
#define IM_TABLE	_IRIS_RIJNDAEL_TABLE

#undef FT_TABLE
#define FT_TABLE	_IRIS_RIJNDAEL_TABLE
#undef FL_TABLE
#define FL_TABLE	_IRIS_RIJNDAEL_TABLE
#undef IT_TABLE
#define IT_TABLE	_IRIS_RIJNDAEL_TABLE
#undef IL_TABLE
#define IL_TABLE	_IRIS_RIJNDAEL_TABLE

#else
#  undef _IRIS_RIJNDAEL_TABLE
#endif

#endif

#if	(FT_TABLE == 0 || FL_TABLE == 0)
#  define SBOX_TABLE	1
#else
#  undef FM_TABLE		// gȂ
#  define FM_TABLE	0
#endif

#if	(IT_TABLE == 0 || IL_TABLE == 0)
#  define ISBOX_TABLE	1
#endif


// 
// original
//#define WPOLY   0x011b	
//#define BPOLY     0x1b	
#define WPOLY   0x011d	//!< GF(2^8) primitive polynomial
#define BPOLY     0x1d	//!< 

// multiply four bytes in GF(2^8) by 'x' {02} in parallel
#define m1  0x80808080
#define m2  0x7f7f7f7f
#define FFmulX(x)  ((((x) & m2) << 1) ^ ((((x) & m1) >> 7) * BPOLY))

// upr
#ifdef __BIG_ENDIAN__
#define upr(x, n)	IRIS_rotr32(x, n<<3)
#define ups(x, n)	((x) >> (n<<3))
#else
#define upr(x, n)	IRIS_rotl32(x, n<<3)
#define ups(x, n)	((x) << (n<<3))
#endif
// byte -> word
#define b2w(b0, b1, b2, b3)	IRIS_Bytes2DWord(b0, b1, b2, b3)
// byte -> word
#define b2wary(a)	IRIS_ByteArray2DWord(a)
// word -> byte
#define w2b(x, n)	IRIS_DWord2Byte(x, n)
// array
#define ary(x, c)	(x)[c]

// S-Boxx[X
// TODO:KÂ̋t狁߂̂ƂBBBB
#define sb_data(w) \
    w(0x63), w(0x7c), w(0x77), w(0x7b), w(0xf2), w(0x6b), w(0x6f), w(0xc5),\
    w(0x30), w(0x01), w(0x67), w(0x2b), w(0xfe), w(0xd7), w(0xab), w(0x76),\
    w(0xca), w(0x82), w(0xc9), w(0x7d), w(0xfa), w(0x59), w(0x47), w(0xf0),\
    w(0xad), w(0xd4), w(0xa2), w(0xaf), w(0x9c), w(0xa4), w(0x72), w(0xc0),\
    w(0xb7), w(0xfd), w(0x93), w(0x26), w(0x36), w(0x3f), w(0xf7), w(0xcc),\
    w(0x34), w(0xa5), w(0xe5), w(0xf1), w(0x71), w(0xd8), w(0x31), w(0x15),\
    w(0x04), w(0xc7), w(0x23), w(0xc3), w(0x18), w(0x96), w(0x05), w(0x9a),\
    w(0x07), w(0x12), w(0x80), w(0xe2), w(0xeb), w(0x27), w(0xb2), w(0x75),\
    w(0x09), w(0x83), w(0x2c), w(0x1a), w(0x1b), w(0x6e), w(0x5a), w(0xa0),\
    w(0x52), w(0x3b), w(0xd6), w(0xb3), w(0x29), w(0xe3), w(0x2f), w(0x84),\
    w(0x53), w(0xd1), w(0x00), w(0xed), w(0x20), w(0xfc), w(0xb1), w(0x5b),\
    w(0x6a), w(0xcb), w(0xbe), w(0x39), w(0x4a), w(0x4c), w(0x58), w(0xcf),\
    w(0xd0), w(0xef), w(0xaa), w(0xfb), w(0x43), w(0x4d), w(0x33), w(0x85),\
    w(0x45), w(0xf9), w(0x02), w(0x7f), w(0x50), w(0x3c), w(0x9f), w(0xa8),\
    w(0x51), w(0xa3), w(0x40), w(0x8f), w(0x92), w(0x9d), w(0x38), w(0xf5),\
    w(0xbc), w(0xb6), w(0xda), w(0x21), w(0x10), w(0xff), w(0xf3), w(0xd2),\
    w(0xcd), w(0x0c), w(0x13), w(0xec), w(0x5f), w(0x97), w(0x44), w(0x17),\
    w(0xc4), w(0xa7), w(0x7e), w(0x3d), w(0x64), w(0x5d), w(0x19), w(0x73),\
    w(0x60), w(0x81), w(0x4f), w(0xdc), w(0x22), w(0x2a), w(0x90), w(0x88),\
    w(0x46), w(0xee), w(0xb8), w(0x14), w(0xde), w(0x5e), w(0x0b), w(0xdb),\
    w(0xe0), w(0x32), w(0x3a), w(0x0a), w(0x49), w(0x06), w(0x24), w(0x5c),\
    w(0xc2), w(0xd3), w(0xac), w(0x62), w(0x91), w(0x95), w(0xe4), w(0x79),\
    w(0xe7), w(0xc8), w(0x37), w(0x6d), w(0x8d), w(0xd5), w(0x4e), w(0xa9),\
    w(0x6c), w(0x56), w(0xf4), w(0xea), w(0x65), w(0x7a), w(0xae), w(0x08),\
    w(0xba), w(0x78), w(0x25), w(0x2e), w(0x1c), w(0xa6), w(0xb4), w(0xc6),\
    w(0xe8), w(0xdd), w(0x74), w(0x1f), w(0x4b), w(0xbd), w(0x8b), w(0x8a),\
    w(0x70), w(0x3e), w(0xb5), w(0x66), w(0x48), w(0x03), w(0xf6), w(0x0e),\
    w(0x61), w(0x35), w(0x57), w(0xb9), w(0x86), w(0xc1), w(0x1d), w(0x9e),\
    w(0xe1), w(0xf8), w(0x98), w(0x11), w(0x69), w(0xd9), w(0x8e), w(0x94),\
    w(0x9b), w(0x1e), w(0x87), w(0xe9), w(0xce), w(0x55), w(0x28), w(0xdf),\
    w(0x8c), w(0xa1), w(0x89), w(0x0d), w(0xbf), w(0xe6), w(0x42), w(0x68),\
    w(0x41), w(0x99), w(0x2d), w(0x0f), w(0xb0), w(0x54), w(0xbb), w(0x16)

#define isb_data(w) \
    w(0x52), w(0x09), w(0x6a), w(0xd5), w(0x30), w(0x36), w(0xa5), w(0x38),\
    w(0xbf), w(0x40), w(0xa3), w(0x9e), w(0x81), w(0xf3), w(0xd7), w(0xfb),\
    w(0x7c), w(0xe3), w(0x39), w(0x82), w(0x9b), w(0x2f), w(0xff), w(0x87),\
    w(0x34), w(0x8e), w(0x43), w(0x44), w(0xc4), w(0xde), w(0xe9), w(0xcb),\
    w(0x54), w(0x7b), w(0x94), w(0x32), w(0xa6), w(0xc2), w(0x23), w(0x3d),\
    w(0xee), w(0x4c), w(0x95), w(0x0b), w(0x42), w(0xfa), w(0xc3), w(0x4e),\
    w(0x08), w(0x2e), w(0xa1), w(0x66), w(0x28), w(0xd9), w(0x24), w(0xb2),\
    w(0x76), w(0x5b), w(0xa2), w(0x49), w(0x6d), w(0x8b), w(0xd1), w(0x25),\
    w(0x72), w(0xf8), w(0xf6), w(0x64), w(0x86), w(0x68), w(0x98), w(0x16),\
    w(0xd4), w(0xa4), w(0x5c), w(0xcc), w(0x5d), w(0x65), w(0xb6), w(0x92),\
    w(0x6c), w(0x70), w(0x48), w(0x50), w(0xfd), w(0xed), w(0xb9), w(0xda),\
    w(0x5e), w(0x15), w(0x46), w(0x57), w(0xa7), w(0x8d), w(0x9d), w(0x84),\
    w(0x90), w(0xd8), w(0xab), w(0x00), w(0x8c), w(0xbc), w(0xd3), w(0x0a),\
    w(0xf7), w(0xe4), w(0x58), w(0x05), w(0xb8), w(0xb3), w(0x45), w(0x06),\
    w(0xd0), w(0x2c), w(0x1e), w(0x8f), w(0xca), w(0x3f), w(0x0f), w(0x02),\
    w(0xc1), w(0xaf), w(0xbd), w(0x03), w(0x01), w(0x13), w(0x8a), w(0x6b),\
    w(0x3a), w(0x91), w(0x11), w(0x41), w(0x4f), w(0x67), w(0xdc), w(0xea),\
    w(0x97), w(0xf2), w(0xcf), w(0xce), w(0xf0), w(0xb4), w(0xe6), w(0x73),\
    w(0x96), w(0xac), w(0x74), w(0x22), w(0xe7), w(0xad), w(0x35), w(0x85),\
    w(0xe2), w(0xf9), w(0x37), w(0xe8), w(0x1c), w(0x75), w(0xdf), w(0x6e),\
    w(0x47), w(0xf1), w(0x1a), w(0x71), w(0x1d), w(0x29), w(0xc5), w(0x89),\
    w(0x6f), w(0xb7), w(0x62), w(0x0e), w(0xaa), w(0x18), w(0xbe), w(0x1b),\
    w(0xfc), w(0x56), w(0x3e), w(0x4b), w(0xc6), w(0xd2), w(0x79), w(0x20),\
    w(0x9a), w(0xdb), w(0xc0), w(0xfe), w(0x78), w(0xcd), w(0x5a), w(0xf4),\
    w(0x1f), w(0xdd), w(0xa8), w(0x33), w(0x88), w(0x07), w(0xc7), w(0x31),\
    w(0xb1), w(0x12), w(0x10), w(0x59), w(0x27), w(0x80), w(0xec), w(0x5f),\
    w(0x60), w(0x51), w(0x7f), w(0xa9), w(0x19), w(0xb5), w(0x4a), w(0x0d),\
    w(0x2d), w(0xe5), w(0x7a), w(0x9f), w(0x93), w(0xc9), w(0x9c), w(0xef),\
    w(0xa0), w(0xe0), w(0x3b), w(0x4d), w(0xae), w(0x2a), w(0xf5), w(0xb0),\
    w(0xc8), w(0xeb), w(0xbb), w(0x3c), w(0x83), w(0x53), w(0x99), w(0x61),\
    w(0x17), w(0x2b), w(0x04), w(0x7e), w(0xba), w(0x77), w(0xd6), w(0x26),\
    w(0xe1), w(0x69), w(0x14), w(0x63), w(0x55), w(0x21), w(0x0c), w(0x7d),

#define mm_data(w) \
    w(0x00), w(0x01), w(0x02), w(0x03), w(0x04), w(0x05), w(0x06), w(0x07),\
    w(0x08), w(0x09), w(0x0a), w(0x0b), w(0x0c), w(0x0d), w(0x0e), w(0x0f),\
    w(0x10), w(0x11), w(0x12), w(0x13), w(0x14), w(0x15), w(0x16), w(0x17),\
    w(0x18), w(0x19), w(0x1a), w(0x1b), w(0x1c), w(0x1d), w(0x1e), w(0x1f),\
    w(0x20), w(0x21), w(0x22), w(0x23), w(0x24), w(0x25), w(0x26), w(0x27),\
    w(0x28), w(0x29), w(0x2a), w(0x2b), w(0x2c), w(0x2d), w(0x2e), w(0x2f),\
    w(0x30), w(0x31), w(0x32), w(0x33), w(0x34), w(0x35), w(0x36), w(0x37),\
    w(0x38), w(0x39), w(0x3a), w(0x3b), w(0x3c), w(0x3d), w(0x3e), w(0x3f),\
    w(0x40), w(0x41), w(0x42), w(0x43), w(0x44), w(0x45), w(0x46), w(0x47),\
    w(0x48), w(0x49), w(0x4a), w(0x4b), w(0x4c), w(0x4d), w(0x4e), w(0x4f),\
    w(0x50), w(0x51), w(0x52), w(0x53), w(0x54), w(0x55), w(0x56), w(0x57),\
    w(0x58), w(0x59), w(0x5a), w(0x5b), w(0x5c), w(0x5d), w(0x5e), w(0x5f),\
    w(0x60), w(0x61), w(0x62), w(0x63), w(0x64), w(0x65), w(0x66), w(0x67),\
    w(0x68), w(0x69), w(0x6a), w(0x6b), w(0x6c), w(0x6d), w(0x6e), w(0x6f),\
    w(0x70), w(0x71), w(0x72), w(0x73), w(0x74), w(0x75), w(0x76), w(0x77),\
    w(0x78), w(0x79), w(0x7a), w(0x7b), w(0x7c), w(0x7d), w(0x7e), w(0x7f),\
    w(0x80), w(0x81), w(0x82), w(0x83), w(0x84), w(0x85), w(0x86), w(0x87),\
    w(0x88), w(0x89), w(0x8a), w(0x8b), w(0x8c), w(0x8d), w(0x8e), w(0x8f),\
    w(0x90), w(0x91), w(0x92), w(0x93), w(0x94), w(0x95), w(0x96), w(0x97),\
    w(0x98), w(0x99), w(0x9a), w(0x9b), w(0x9c), w(0x9d), w(0x9e), w(0x9f),\
    w(0xa0), w(0xa1), w(0xa2), w(0xa3), w(0xa4), w(0xa5), w(0xa6), w(0xa7),\
    w(0xa8), w(0xa9), w(0xaa), w(0xab), w(0xac), w(0xad), w(0xae), w(0xaf),\
    w(0xb0), w(0xb1), w(0xb2), w(0xb3), w(0xb4), w(0xb5), w(0xb6), w(0xb7),\
    w(0xb8), w(0xb9), w(0xba), w(0xbb), w(0xbc), w(0xbd), w(0xbe), w(0xbf),\
    w(0xc0), w(0xc1), w(0xc2), w(0xc3), w(0xc4), w(0xc5), w(0xc6), w(0xc7),\
    w(0xc8), w(0xc9), w(0xca), w(0xcb), w(0xcc), w(0xcd), w(0xce), w(0xcf),\
    w(0xd0), w(0xd1), w(0xd2), w(0xd3), w(0xd4), w(0xd5), w(0xd6), w(0xd7),\
    w(0xd8), w(0xd9), w(0xda), w(0xdb), w(0xdc), w(0xdd), w(0xde), w(0xdf),\
    w(0xe0), w(0xe1), w(0xe2), w(0xe3), w(0xe4), w(0xe5), w(0xe6), w(0xe7),\
    w(0xe8), w(0xe9), w(0xea), w(0xeb), w(0xec), w(0xed), w(0xee), w(0xef),\
    w(0xf0), w(0xf1), w(0xf2), w(0xf3), w(0xf4), w(0xf5), w(0xf6), w(0xf7),\
    w(0xf8), w(0xf9), w(0xfa), w(0xfb), w(0xfc), w(0xfd), w(0xfe), w(0xff)

#define f1(x)	(x)
#define f2(x)   ((x<<1) ^ (((x>>7) & 1) * WPOLY))
#define f4(x)   ((x<<2) ^ (((x>>6) & 1) * WPOLY) ^ (((x>>6) & 2) * WPOLY))
#define f8(x)   ((x<<3) ^ (((x>>5) & 1) * WPOLY) ^ (((x>>5) & 2) * WPOLY) \
                        ^ (((x>>5) & 4) * WPOLY))
#define f3(x)   (f2(x) ^ x)
#define f9(x)   (f8(x) ^ x)
#define fb(x)   (f8(x) ^ f2(x) ^ x)
#define fd(x)   (f8(x) ^ f4(x) ^ x)
#define fe(x)   (f8(x) ^ f4(x) ^ f2(x))

#define h0(p)   (p)

#define w0(p)   b2w(p, 0, 0, 0)
#define w1(p)   b2w(0, p, 0, 0)
#define w2(p)   b2w(0, 0, p, 0)
#define w3(p)   b2w(0, 0, 0, p)

#define u0(p)   b2w(f2(p), p    , p    , f3(p))
#define u1(p)   b2w(f3(p), f2(p), p    , p    )
#define u2(p)   b2w(p    , f3(p), f2(p), p    )
#define u3(p)   b2w(p    , p    , f3(p), f2(p))

#define v0(p)   b2w(fe(p), f9(p), fd(p), fb(p))
#define v1(p)   b2w(fb(p), fe(p), f9(p), fd(p))
#define v2(p)   b2w(fd(p), fb(p), fe(p), f9(p))
#define v3(p)   b2w(f9(p), fd(p), fb(p), fe(p))

#define no_table(x, box, vf, rf, c) b2w( \
	box[ w2b(vf(x, 0, c), rf(0, c))], \
    box[ w2b(vf(x, 1, c), rf(1, c))], \
    box[ w2b(vf(x, 2, c), rf(2, c))], \
    box[ w2b(vf(x, 3, c), rf(3, c))])

#define one_table(x, op, tab, vf, rf, c)		\
 (     tab[ w2b(vf(x, 0, c), rf(0, c))]			\
  ^ op(tab[ w2b(vf(x, 1, c), rf(1, c))] ,1)		\
  ^ op(tab[ w2b(vf(x, 2, c), rf(2, c))] ,2)		\
  ^ op(tab[ w2b(vf(x, 3, c), rf(3, c))] ,3))

#define four_tables(x, tab, vf, rf, c) \
 (  tab[0][ w2b(vf(x,0,c), rf(0,c)) ] \
  ^ tab[1][ w2b(vf(x,1,c), rf(1,c)) ] \
  ^ tab[2][ w2b(vf(x,2,c), rf(2,c)) ] \
  ^ tab[3][ w2b(vf(x,3,c), rf(3,c)) ] )

#define vf1(x,r,c)  (x)
#define rf1(r,c)    (r)
#define rf2(r,c)    ((r-c)&3)


#if		(FL_TABLE == 4)
#define ls_box(x,c)     four_tables(x, fl_tab, vf1, rf2, c)
#elif	(FL_TABLE == 1)
#define ls_box(x,c)     one_table(x, upr, fl_tab, vf1, rf2, c)
#else
#define ls_box(x,c)		no_table(x, s_box, vf1, rf2, c)
#endif

#define	dec_fmver
#ifdef	FM_TABLE
#if		(FM_TABLE == 4)
#define fwd_mcol(x)     four_tables(x, fm_tab, vf1, rf1, 0)
#elif	(FM_TABLE == 1)
#define fwd_mcol(x)     one_table(x, upr, fm_tab, vf1, rf1, 0)
#else
#undef dec_fmver
#define dec_fmver		u32 f1, f2;	// ɕKvȕϐ`	
#define fwd_mcol(x)		(f1 = (x), f2 = FFmulX(f1), f2 ^ upr(f1 ^ f2, 3) ^ upr(f1, 2) ^ upr(f1, 1))
#endif
#endif

#define dec_imver
#if		(IM_TABLE == 4)
#define inv_mcol(x)     four_tables(x, im_tab, vf1, rf1, 0)
#elif	(IM_TABLE == 1)
#define inv_mcol(x)     one_table(x, upr, im_tab, vf1, rf1, 0)
#else
#undef dec_imver
#define dec_imver		u32 f2, f4, f8, f9;	// ɕKvȕϐ`	
#define inv_mcol(x)		\
    (f9 = (x), f2 = FFmulX(f9), f4 = FFmulX(f2), f8 = FFmulX(f4), f9 ^= f8, \
    f2 ^= f4 ^ f8 ^ upr(f2 ^ f9,3) ^ upr(f4 ^ f9,2) ^ upr(f9,1))
#endif

//======================================================================
// variable
static const u32 rcon_tab[29] =
{
    w0(0x01), w0(0x02), w0(0x04), w0(0x08),
    w0(0x10), w0(0x20), w0(0x40), w0(0x80),
    w0(0x1b), w0(0x36), w0(0x6c), w0(0xd8),
    w0(0xab), w0(0x4d), w0(0x9a), w0(0x2f),
    w0(0x5e), w0(0xbc), w0(0x63), w0(0xc6),
    w0(0x97), w0(0x35), w0(0x6a), w0(0xd4),
    w0(0xb3), w0(0x7d), w0(0xfa), w0(0xef),
    w0(0xc5)
};

#ifdef	SBOX_TABLE
static const u32 s_box[256] = { sb_data(h0) };
#endif
#ifdef	ISBOX_TABLE
static const u32 inv_s_box[256] = { isb_data(h0) };
#endif

#if		(FM_TABLE == 4)
static const u32 fm_tab[4][256] = {
	{  mm_data(u0) }, {  mm_data(u1) }, {  mm_data(u2) }, {  mm_data(u3) } 
};
#elif	(FM_TABLE == 1)
static const u32 fm_tab[256] = { mm_data(u0) };
#endif

#if		(FT_TABLE == 4)
static const u32 ft_tab[4][256] = {
	{  sb_data(u0) }, {  sb_data(u1) }, {  sb_data(u2) }, {  sb_data(u3) } 
};
#elif	(FT_TABLE == 1)
static const u32 ft_tab[256] = { sb_data(u0) };
#endif

#if		(FL_TABLE == 4)
static const u32 fl_tab[4][256] = {
	{  sb_data(w0) }, {  sb_data(w1) }, {  sb_data(w2) }, {  sb_data(w3) } 
};
#elif	(FL_TABLE == 1)
static const u32 fl_tab[256] = { sb_data(w0) };
#endif

#if		(IM_TABLE == 4)
static const u32 im_tab[4][256] = {
	{  mm_data(v0) }, {  mm_data(v1) }, {  mm_data(v2) }, {  mm_data(v3) } 
};
#elif	(IM_TABLE == 1)
static const u32 im_tab[256] = { mm_data(v0) };
#endif

#if		(IT_TABLE == 4)
static const u32 it_tab[4][256] = {
	{ isb_data(v0) }, { isb_data(v1) }, { isb_data(v2) }, { isb_data(v3) } 
};
#elif	(IT_TABLE == 1)
static const u32 it_tab[256] = { isb_data(v0) };
#endif

#if		(IL_TABLE == 4)
static const u32 il_tab[4][256] = {
	{ isb_data(w0) }, { isb_data(w1) }, { isb_data(w2) }, { isb_data(w3) } 
};
#elif	(IL_TABLE == 1)
static const u32 il_tab[256] = { isb_data(w0) };
#endif

//======================================================================
// class
/**********************************************************************//**
 *
 * RXgN^
 *
*//***********************************************************************/
template<u32 _BLB, u32 _KYB>
CRijndael<_BLB, _KYB>::CRijndael(void)
{
}

/**********************************************************************//**
 *
 * ̍쐬
 *
 -----------------------------------------------------------------------
 * @param [in]	pKey	= 
 * @param [in]	size	= z
*//***********************************************************************/
template<u32 _BLB, u32 _KYB>
bool CRijndael<_BLB, _KYB>::CreateKeys(const u8* pKey, u32 nSize)
{
	IRIS_ASSERT( pKey != nullptr );
	IRIS_ASSERT( nSize >= KEYSIZE );
	u32 tmp[NK];
	const u8* pk = pKey;
	s32 i, j, idx;
	for( i=0; i < NK; ++i, pk+=4 )
	{
	    m_KeySch[i] = tmp[i] = b2wary(pk);
	}
	s32 l = (NB * (NR+1) - 1) / NK;
	for( i=0, idx=NK; i < l; ++i )
	{
		tmp[0] ^= ls_box(tmp[NK-1],3) ^ rcon_tab[i];
		m_KeySch[idx++] = tmp[0];
		for( j=1; j < NK && idx < KS_LENGTH; ++j )
		{
			tmp[j] ^= tmp[j-1];
			m_KeySch[idx++] = tmp[j];
		}
	}
	return true;
}

#define fwd_var(x,r,c)	\
 ( r == 0 ?				\
	ary(x, (c)%NB)		\
 : r == 1 ?				\
	ary(x, (c+1)%NB)	\
 : r == 2 ?				\
	( NB == 8 ? ary(x, (c+3)%NB)	\
	:			ary(x, (c+2)%NB))	\
 :						\
	( NB >  6 ? ary(x, (c+4)%NB)	\
	:			ary(x, (c+3)%NB))	\
)

#if		(FT_TABLE == 4)
#define	fwd_round(x,k,c)	(k)[c] ^ four_tables(x, ft_tab, fwd_var, rf1, c)
#elif	(FT_TABLE == 1)
#define	fwd_round(x,k,c)	(k)[c] ^ one_table(x, upr, ft_tab, fwd_var, rf1, c)
#else
#define fwd_round(x,k,c)	fwd_mcol(no_table(x, s_box, fwd_var, rf1, c)) ^ (k)[c]
#endif

#if		(FL_TABLE == 4)
#define fwd_lround(x,k,c)	(k)[c] ^ four_tables(x, fl_tab, fwd_var, rf1, c)
#elif	(FL_TABLE == 1)
#define fwd_lround(x,k,c)	(k)[c] ^ one_table(x, ups, fl_tab, fwd_var, rf1, c)
#else
#define fwd_lround(x,k,c)	no_table(x, s_box, fwd_var, rf1, c) ^ (k)[c]
#endif

/**********************************************************************//**
 *
 * Í
 *
 -----------------------------------------------------------------------
 * @param [out]	dst	= o̓obt@[BLOCKSIZE]
 * @param [in]	src	= ̓obt@[BLOCKSIZE]
*//***********************************************************************/
template<u32 _BLB, u32 _KYB>
bool CRijndael<_BLB, _KYB>::EncryptBlock(u8* dst, const u8* src)
{
	IRIS_ASSERT( dst != nullptr );
	IRIS_ASSERT( src != nullptr );
	// [N
	u32 work[2][NB];
	const u32* pk = m_KeySch;
	const u8* ps = src;
	u8* pd = dst;

	s32 idx0 = NR&1;
	s32 idx1 = (1-idx0);
	s32 i,j;
	for( i=0; i < NB; ++i, ps+=4 )
	{
		work[idx0][i] = b2wary(ps) ^ pk[i];
	}

	pk += NB;
	for( i=NR-1; i > 0; --i, pk+=NB )
	{
		for( j=0; j < NB; ++j )
		{
			work[idx1][j] = fwd_round(work[idx0], pk, j);
		}
		idx1 = idx0;
		idx0 = 1-idx0;
	}
	
	for( j=0; j < NB; ++j )
	{
		work[idx1][j] = fwd_lround(work[idx0], pk, j);
	}

	for( i=0; i < NB; ++i)
	{
		for( j=0; j < 4; ++j )
		{
			*(pd++) = IRIS_DWord2Byte(work[0][i], j);
		}
	}
	return true;
}

#define inv_var(x,r,c)	\
 ( r == 0 ?				\
	ary(x, c%NB)		\
 : r == 1 ?				\
	ary(x, (c+NB-1)%NB)	\
 : r == 2 ?				\
	( NB == 8 ? ary(x, (c+NB-3)%NB)		\
	:			ary(x, (c+NB-2)%NB) )	\
 :						\
	( NB >  6 ? ary(x, (c+NB-4)%NB)		\
	:		    ary(x, (c+NB-3)%NB) )	\
)

#define dec_invver
#if		(IT_TABLE == 4)
#define	inv_round(x,k,c)	inv_mcol((k)[c]) ^ four_tables(x, it_tab, inv_var, rf1, c)
#elif	(IT_TABLE == 1)
#define	inv_round(x,k,c)	inv_mcol((k)[c]) ^ one_table(x, upr, it_tab, inv_var, rf1, c)
#else
#undef dec_invver
#define dec_invver	dec_imver
#define	inv_round(x,k,c)	inv_mcol(no_table(x, inv_s_box, inv_var, rf1, c) ^ (k)[c])
#endif

#if		(IL_TABLE == 4)
#define inv_lround(x,k,c)	(k)[c] ^ four_tables(x, il_tab, inv_var, rf1, c)
#elif	(IL_TABLE == 1)
#define inv_lround(x,k,c)	(k)[c] ^ one_table(x, ups, il_tab, inv_var, rf1, c)
#else
#define	inv_lround(x,k,c)	no_table(x, inv_s_box, inv_var, rf1, c) ^ (k)[c]
#endif

/**********************************************************************//**
 *
 * 
 *
 -----------------------------------------------------------------------
 * @param [out]	dst	= o̓obt@[BLOCKSIZE]
 * @param [in]	src	= ̓obt@[BLOCKSIZE]
*//***********************************************************************/
template<u32 _BLB, u32 _KYB>
bool CRijndael<_BLB, _KYB>::DecryptBlock(u8* dst, const u8* src)
{
	IRIS_ASSERT( dst != nullptr );
	IRIS_ASSERT( src != nullptr );
	// [N
	dec_invver	// inv_mcol p̃[N
	u32 work[2][NB];
	const u32* pk = m_KeySch + NB*NR;
	const u8* ps = src;
	u8* pd = dst;

	s32 idx0 = NR&1;
	s32 idx1 = (1-idx0);
	s32 i,j;
	for( i=0; i < NB; ++i, ps+=4 )
	{
		work[idx0][i] = b2wary(ps) ^ pk[i];
	}

	pk -= NB;
	for( i=0; i < NR-1; ++i, pk-=NB )
	{
		for( j=0; j < NB; ++j )
		{
			work[idx1][j] = inv_round(work[idx0], pk, j);
		}
		idx1 = idx0;
		idx0 = 1-idx0;
	}
	
	for( j=0; j < NB; ++j )
	{
		work[idx1][j] = inv_lround(work[idx0], pk, j);
	}

	for( i=0; i < NB; ++i)
	{
		for( j=0; j < 4; ++j )
		{
			*(pd++) = IRIS_DWord2Byte(work[0][i], j);
		}
	}
	return true;
}

// ^̎̉
// 32̔{ȂΉł
#ifdef _IRIS_RIJNDAEL_BLK

#ifdef _IRIS_RIJNDAEL_KEY
// only one
template class CRijndael<_IRIS_RIJNDAEL_BLK, _IRIS_RIJNDAEL_KEY>;

#else
// only block
template class CRijndael<_IRIS_RIJNDAEL_BLK, 128>;
template class CRijndael<_IRIS_RIJNDAEL_BLK, 192>;
template class CRijndael<_IRIS_RIJNDAEL_BLK, 256>;

#endif

#else

#ifdef _IRIS_RIJNDAEL_KEY
// only key
template class CRijndael<128, _IRIS_RIJNDAEL_KEY>;
template class CRijndael<192, _IRIS_RIJNDAEL_KEY>;
template class CRijndael<256, _IRIS_RIJNDAEL_KEY>;

#else

// all
template class CRijndael<128, 128>;
template class CRijndael<128, 192>;
template class CRijndael<128, 256>;
template class CRijndael<192, 128>;
template class CRijndael<192, 192>;
template class CRijndael<192, 256>;
template class CRijndael<256, 128>;
template class CRijndael<256, 192>;
template class CRijndael<256, 256>;

#endif

#endif

}	// end of namespace fnd
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "../../iris_iostream.h"
#include <string.h>
#include "../../iris_using.h"

#define ALL_TEST

//======================================================================
// test
IRIS_UNITTEST(CFndRijndaelUnitTest, FndRijndaelUnitTest)
{
#ifdef _IRIS_RIJNDAEL_BLK
#define BLK	_IRIS_RIJNDAEL_BLK
#undef ALL_TEST
#else
#define BLK	256
#endif
#ifdef _IRIS_RIJNDAEL_KEY
#define KEY	_IRIS_RIJNDAEL_KEY
#undef ALL_TEST
#else
#define KEY	192
#endif

#ifndef ALL_TEST
	while( 1 )
	{
		typedef CRijndael<BLK,KEY> CCipher;
		CCipher rijndael;
		char c[CCipher::BLOCKSIZE*16] = "test";
		char ck[CCipher::KEYSIZE+1] = "0123456789abcdef";
		u8 key[CCipher::KEYSIZE];
		memcpy(key, ck, sizeof(key));

		// ÍL[
		rijndael.CreateKeys(key, sizeof(key));

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
		std::cout << "͂ĂB" << std::endl;
		std::cin >> c;
#endif

		rijndael.Encrypt((u8*)c, (u8*)c, sizeof(c));
		//char m[CCipher::BLOCKSIZE*16+1];
		//memcpy(m, c, sizeof(c));
		//m[CCipher::BLOCKSIZE*16] = '\0';
		//std::cout << m << std::endl;

		rijndael.Decrypt((u8*)c, (u8*)c, sizeof(c));

		std::cout << c << std::endl;
		
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
		s32 flag=0;
		std::cout << "Iꍇ́A0 ͂ĂB" << std::endl;
		std::safe_cin >> flag;
		if( flag == 0 ) break;
#else
		break;
#endif
	}
#else
#define RIJN_TEST(cipher) {								\
	cipher	rijndael;									\
	char c[512] = "test";								\
	char ck[cipher::KEYSIZE+1] = "0123456789abcdef";	\
	u8 key[cipher::KEYSIZE];							\
	memcpy(key, ck, sizeof(key));						\
	strcpy_s(c, sizeof(c), txt);						\
	printf("%s\n", #cipher);							\
	rijndael.CreateKeys(key, sizeof(key));				\
	rijndael.Encrypt((u8*)c, (u8*)c, sizeof(c));		\
	rijndael.Decrypt((u8*)c, (u8*)c, sizeof(c));		\
	std::cout << c << std::endl;						\
	}

	char txt[512] = "FndRijndael";
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
	std::cout << "͂ĂB" << std::endl;
	std::cin >> txt;
#endif

	RIJN_TEST(CRijndael128x128)
	RIJN_TEST(CRijndael128x192)
	RIJN_TEST(CRijndael128x256)
	RIJN_TEST(CRijndael192x128)
	RIJN_TEST(CRijndael192x192)
	RIJN_TEST(CRijndael192x256)
	RIJN_TEST(CRijndael256x128)
	RIJN_TEST(CRijndael256x192)
	RIJN_TEST(CRijndael256x256)

#endif
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

