//======================================================================
//-----------------------------------------------------------------------
/**
 * @file		FndBlockCipherMode.cpp
 * @brief		ubNÍ[hNXt@C
 *
 * @author		t.sirayanagi
 * @version		1.0
 *
 * @par			copyright
 * Copyright (C) 2009-2011 Takazumi Shirayanagi\n
 * The new BSD License is applied to this software.
 * see iris_LICENSE.txt
*/
//-----------------------------------------------------------------------
//======================================================================
#define INCG_IRIS_FndBlockCipherMode_CPP_

//======================================================================
// include
#include "FndBlockCipherMode.h"
#include "../../iris_debug.h"

namespace iris {
namespace fnd
{

//======================================================================
// function
/**********************************************************************//**
 *
 * Í(ECB:Electronic CodeBook)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
*//***********************************************************************/
bool EncryptECB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
{
	IRIS_ASSERT( pCipher != nullptr );
	u32 bs = pCipher->GetBlockSize();
	const u8* ps = src;
	u8* pd = dst;
	IRIS_ASSERT( nSize % bs == 0 );
	for( u32 i=0, n = nSize/bs; i < n; ++i, ps+=bs, pd+=bs )
	{
		pCipher->EncryptBlock(pd, ps);
	}
	return true;
}

/**********************************************************************//**
 *
 * (ECB:Electronic CodeBook)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
*//***********************************************************************/
bool DecryptECB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher)
{
	IRIS_ASSERT( pCipher != nullptr );
	u32 bs = pCipher->GetBlockSize();
	const u8* ps = src;
	u8* pd = dst;
	IRIS_ASSERT( nSize % bs == 0 );
	for( u32 i=0, n = nSize/bs; i < n; ++i, ps+=bs, pd+=bs )
	{
		pCipher->DecryptBlock(pd, ps);
	}
	return true;
}

/**********************************************************************//**
 *
 * Í(CBC : Cipher Block Chain)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
*//***********************************************************************/
bool EncryptCBC(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	u32 bs = pCipher->GetBlockSize();
	const u8* ps = src;
	u8* pd = dst;
	const u8* prev = iv;
	IRIS_ASSERT( nSize % bs == 0 );
	for( u32 i=0, n = nSize/bs; i < n; ++i, ps+=bs, pd+=bs )
	{
		for( u32 j=0; j < bs; ++j)
		{
			pd[j] = (u8)(prev[j] ^ ps[j]);
		}
		pCipher->EncryptBlock(pd, pd);
		prev = pd;
	}
	return true;
}

/**********************************************************************//**
 *
 * (CBC : Cipher Block Chain)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
*//***********************************************************************/
bool DecryptCBC(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	u32 bs = pCipher->GetBlockSize();
	if( nSize < bs ) return false;
	const u8* ps = src + nSize - bs;
	u8* pd = dst + nSize - bs;
	const u8* prev = iv;
	IRIS_ASSERT( nSize % bs == 0 );
	for( u32 i=0, n = nSize/bs; i < n-1; ++i, pd-=bs )
	{
		pCipher->DecryptBlock(pd, ps);
		ps -= bs;
		for( u32 j=0; j < bs; ++j)
		{
			pd[j] = (u8)(pd[j] ^ ps[j]);
		}
	}
	pCipher->DecryptBlock(dst, src);
	for( u32 j=0; j < bs; ++j)
	{
		dst[j] = (u8)(dst[j] ^ prev[j]);
	}
	return true;
}

/**********************************************************************//**
 *
 * Í(CTS : Cipher Text Stealing)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
 * @return	
*//***********************************************************************/
bool EncryptCTS(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	u32 bs = pCipher->GetBlockSize();
	IRIS_ASSERT( nSize >= bs );
	if( nSize == bs ) return EncryptCBC(dst, src, nSize, pCipher, iv);
	const u8* ps = src;
	u8* pd = dst;
	s32 ctslen = (s32)((nSize - 1) % bs + 1);
	s32 cbclen = (s32)(nSize - ctslen - bs);

	// Ō̃ubNcCBCÍ
	EncryptCBC(pd, ps, cbclen + bs, pCipher, iv);

	// Ō̃ubNƂPÕubNXORiȂ͖߂j
	// PÕubN̒[Ō̃ubNɈړ
	u8 work;
	for( s32 i=0; i < ctslen; ++i )	
	{
		work = (pd + cbclen)[i];
		(pd + cbclen)[i] ^= (ps + cbclen + bs)[i];
		(pd + cbclen + bs)[i] = work;
	}

	// Ō̓̓ubNio̓ubŇォQԖځjÍ
	pCipher->EncryptBlock(pd + cbclen, pd + cbclen);
	return true;
}

/**********************************************************************//**
 *
 * (CTS : Cipher Text Stealing)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
 * @return	
*//***********************************************************************/
bool DecryptCTS(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	u32 bs = pCipher->GetBlockSize();
	IRIS_ASSERT( nSize >= bs );
	if( nSize == bs ) return DecryptCBC(dst, src, nSize, pCipher, iv);

	const u8* ps = src;
	u8* pd = dst;
	const u8* prev = iv;
	s32 ctslen = (s32)((nSize - 1) % bs + 1);
	s32 cbclen = (s32)(nSize - ctslen - bs);

	// Ō̕ubN𕜍
	pCipher->DecryptBlock(pd + cbclen, ps + cbclen);

	// Ō̕ubN̕ʂŌ̃ubNɈړ
	// Ō̈ÍubNォQԖڂɈړ
	u8 work;
	for( s32 i=0; i < ctslen; ++i )
	{
		work = (ps + cbclen + bs)[i];
		(pd + cbclen + bs)[i] = (u8)((pd + cbclen)[i] ^ (ps + cbclen + bs)[i]);
		(pd + cbclen)[i] = work;
	}

	// ォQԖڂ̃ubN𕜍
	pCipher->DecryptBlock(pd + cbclen, pd + cbclen);

	// CBC
	if( cbclen != 0 )
	{
		prev = (ps + cbclen - bs);
	}
	for( u32 i=0; i < bs; ++i )	(pd + cbclen)[i] ^= prev[i];

	// c̃ubN𕜍
	return DecryptCBC(pd, ps, (u32)cbclen, pCipher, iv);
}

/**********************************************************************//**
 *
 * Í(CTR : Countr Mode)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	couter	= JE^
 * @param [in]	work	= Ɨpobt@
*//***********************************************************************/
bool EncryptCTR(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, u32 counter, u8* work)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( work != nullptr );
	u32 bs = pCipher->GetBlockSize();
	IRIS_ASSERT( bs >= 4 );
	const u8* ps = src;
	const u8* end = ps + nSize;
	u8* pd = dst;
	u32 n = (nSize+bs-1)/bs;
	u32 i,j=0;
	u8* ic = (u8*)&counter;
	for( i=0; i < bs; )
	{
		for( j=0; j < 4; ++j, ++i )	work[i] = ic[j];
	}
	for( i=0; i < n; ++i )
	{
		// ubN𐶐
		pCipher->EncryptBlock(work, work);
		for( j=0; j < bs && ps < end; ++j, ++pd, ++ps)
		{
			*pd = (u8)(work[j] ^ *ps);
		}
		for( j=0; j < bs; ++j )
		{
			++work[j];
			if( work[j] ) break;
		}
	}
	return true;
}

/**********************************************************************//**
 *
 * (CTR : Countr Mode)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	couter	= JE^
 * @param [in]	work	= Ɨpobt@
*//***********************************************************************/
bool DecryptCTR(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, u32 counter, u8* work)
{
	return EncryptCTR(dst, src, nSize, pCipher, counter, work);
}

/**********************************************************************//**
 *
 * Í(OFB : Output FeedBack)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
 * @param [in]	work	= Ɨpobt@
*//***********************************************************************/
bool EncryptOFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	IRIS_ASSERT( work != nullptr );
	u32 bs = pCipher->GetBlockSize();
	const u8* ps = src;
	u8* pd = dst;
	const u8* prev = iv;
	const u8* end = ps + nSize;
	u32 n = (nSize+bs-1)/bs;
	for( u32 i=0; i < n; ++i )
	{
		// ubN𐶐
		pCipher->EncryptBlock(work, prev);
		for( u32 j=0; j < bs && ps < end; ++j, ++pd, ++ps)
		{
			*pd = (u8)(work[j] ^ *ps);
		}
		prev = work;
	}
	return true;
}

/**********************************************************************//**
 *
 * (OFB : Output FeedBack)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
 * @param [in]	work	= Ɨpobt@
*//***********************************************************************/
bool DecryptOFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work)
{
	return EncryptOFB(dst, src, nSize, pCipher, iv, work);
}

/**********************************************************************//**
 *
 * Í(CFB : Cipher FeedBack)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
 * @param [in]	work	= Ɨpobt@
*//***********************************************************************/
bool EncryptCFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	IRIS_ASSERT( work != nullptr );
	u32 bs = pCipher->GetBlockSize();
	const u8* ps = src;
	u8* pd = dst;
	const u8* prev = iv;
	const u8* end = ps + nSize;
	u32 n = (nSize+bs-1)/bs;
	for( u32 i=0; i < n; ++i )
	{
		// ubN𐶐
		pCipher->EncryptBlock(work, prev);
		for( u32 j=0; j < bs && ps < end; ++j, ++pd, ++ps)
		{
			*pd = (u8)(work[j] ^ *ps);
		}
		prev = pd;
	}
	return true;
}

/**********************************************************************//**
 *
 * (CFB : Cipher FeedBack)
 *
 -----------------------------------------------------------------------
 * @param [out]	dst		= o̓obt@[BLOCKSIZE]
 * @param [in]	src		= ̓obt@[BLOCKSIZE]
 * @param [in]	nSize	= ̓obt@TCY
 * @param [in]	pCipher	= ubNÍNX
 * @param [in]	iv		= xNg[BLOCKSIZE]
 * @param [in]	work	= Ɨpobt@
*//***********************************************************************/
bool DecryptCFB(u8* dst, const u8* src, u32 nSize, IBlockCipher* pCipher, const u8* iv, u8* work)
{
	IRIS_ASSERT( pCipher != nullptr );
	IRIS_ASSERT( iv != nullptr );
	IRIS_ASSERT( work != nullptr );
	u32 bs = pCipher->GetBlockSize();
	if( nSize < bs ) return false;
	const u8* ps = src;
	u8* pd = dst;
	u8 tmp=0;
	const u8* prev = iv;
	const u8* end = ps + nSize;
	u32 n = (nSize+bs-1)/bs;
	for( u32 i=0; i < n; ++i )
	{
		// ubN̐
		pCipher->EncryptBlock(work, prev);
		for( u32 j=0; j < bs && ps < end; ++j, ++pd, ++ps )
		{
			tmp = *ps;
			*pd = (u8)(work[j] ^ *ps);
			work[j] = tmp;
		}
		prev = work;
	}
	return true;
}

}	// end of namespace fnd
}	// end of namespace iris

#if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

//======================================================================
// include
#include "../../unit/UnitCore.h"
#include "FndAES.h"
#include "FndRijndael.h"
#include "FndBlowfish.h"
#include "FndTwofish.h"
#include "../../iris_iostream.h"
#include <string.h>
#include "../../iris_using.h"

//======================================================================
// define
#define TEST_MODE_ECB	'_ECB'
#define TEST_MODE_CBC	'_CBC'
#define TEST_MODE_CTS	'_CTS'
#define TEST_MODE_CTR	'_CTR'
#define TEST_MODE_OFB	'_OFB'
#define TEST_MODE_CFB	'_CFB'

#define CIPHER_MODE_RIJNDAEL	'RIJN'
#define CIPHER_MODE_BLOWFISH	'BWFH'
#define CIPHER_MODE_TWOFISH		'TWFH'
#define CIPHER_MODE_AES			'_AES'

#define TEST_MODE	TEST_MODE_CTR
#define CIPHER_MODE	CIPHER_MODE_RIJNDAEL

//======================================================================
// test
IRIS_UNITTEST(CFndBlockCipherModeUnitTest, FndBlockCipherModeUnitTest)
{
	while( 1 )
	{
#if		(CIPHER_MODE == CIPHER_MODE_RIJNDAEL)
		typedef CDefRijndael		CCipher;
#elif	(CIPHER_MODE == CIPHER_MODE_BLOWFISH)
		typedef CBlowfish			CCipher;
#elif	(CIPHER_MODE == CIPHER_MODE_TWOFISH)
		typedef CDefTwofish			CCipher;
#else
		typedef CAES128				CCipher;
#endif

#if		(TEST_MODE == TEST_MODE_CBC)
		typedef CCipherCBC<CCipher::BLOCKSIZE>	CMode;
#elif	(TEST_MODE == TEST_MODE_CTS)
		typedef CCipherCTS<CCipher::BLOCKSIZE>	CMode;
#elif	(TEST_MODE == TEST_MODE_CTR)
		typedef CCipherCTR<CCipher::BLOCKSIZE>	CMode;
#elif	(TEST_MODE == TEST_MODE_OFB)
		typedef CCipherOFB<CCipher::BLOCKSIZE>	CMode;
#elif	(TEST_MODE == TEST_MODE_CFB)
		typedef CCipherCFB<CCipher::BLOCKSIZE>	CMode;
#else
		typedef CCipherECB	CMode;
#endif

		CCipher cipher;
		CMode	mode;
		char c[CCipher::BLOCKSIZE*16] = "test";
		printf("C %c%c%c%c : M %c%c%c%c\n"
			, IRIS_DWord2Byte(CIPHER_MODE,3)
			, IRIS_DWord2Byte(CIPHER_MODE,2)
			, IRIS_DWord2Byte(CIPHER_MODE,1)
			, IRIS_DWord2Byte(CIPHER_MODE,0)

			, IRIS_DWord2Byte(TEST_MODE,3)
			, IRIS_DWord2Byte(TEST_MODE,2)
			, IRIS_DWord2Byte(TEST_MODE,1)
			, IRIS_DWord2Byte(TEST_MODE,0)
			);

		// 
		u8 key[CCipher::KEYSIZE];
		for( int i=0; i < CCipher::KEYSIZE; ++i )
		{
			key[i] = (u8)('a'+i);
		}
		cipher.CreateKeys((u8*)key, sizeof(key));

#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
		std::cout << "͂ĂB" << std::endl;
		std::cin >> c;
#endif

		u32 iv = 0xFEDCBA98;
		mode.SetInitVector((u8*)&iv, sizeof(iv));
		if( mode.IsStream() )
		{
			u32 len = (u32)strlen(c)+1;
			if( len < cipher.GetBlockSize()*mode.GetMinimumBlockNum() ) len = sizeof(c);
			mode.Encrypt((u8*)c, (u8*)c, len, &cipher);
		}
		else
		{
			mode.Encrypt((u8*)c, (u8*)c, sizeof(c), &cipher);
		}
		//char m[CCipher::BLOCKSIZE*16+1];
		//memcpy(m, c, sizeof(c));
		//m[CCipher::BLOCKSIZE*16] = '\0';
		//std::cout << m << std::endl;

#if		(TEST_MODE == TEST_MODE_CTS)
		mode.Decrypt((u8*)c, (u8*)c, len, &cipher);
#else
		mode.Decrypt((u8*)c, (u8*)c, sizeof(c), &cipher);
#endif

		std::cout << c << std::endl;
		
#ifndef _IRIS_SUPPORT_AUTO_UNITTEST
		s32 flag=0;
		std::cout << "Iꍇ́A0 ͂ĂB" << std::endl;
		std::safe_cin >> flag;
		if( flag == 0 ) break;
#else
		break;
#endif
	}
}

#endif // #if (defined(_IRIS_UNITTEST) || defined(_IRIS_MULTI_UNITTEST))

