﻿/**
 * @file TrayApp.cpp
 *
 */


#define DBG_LEVEL 0
#include <Raym/Log.h>

#include <time.h>
#include <direct.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <fcntl.h>
#include <io.h>
#include <share.h>
#include <winsock2.h>
#include <ws2tcpip.h>
#include <Iphlpapi.h>

#include "ry0/iPTd/TrayApp.h"

using namespace Raym;

namespace ry0
{
namespace iPTd
{

static const char *PLIST_PREFIX = "com.gmail.tim.and.pom";
static const int   DEF_FORCED_SHUTDOWN = 120;          // 強制シャットダウンするまでの時間（分単位）

bool TrayApp::canTerminate()
{
    return true;
}

/*
 * TrayApp::WndProc() からコールされる
 *
 *   スレッドコンテキスト：メインスレッド
 */
void TrayApp::systemWillSuspend()
{
    DebugLog2("TrayApp::systemWillSuspend()");
}

/*
 * TrayApp::WndProc() からコールされる
 *
 *   スレッドコンテキスト：メインスレッド
 */
void TrayApp::systemResumed()
{
    DebugLog2("TrayApp::systemResumed()");
}

/*
 * TrayApp::WndProc() からコールされる
 *
 *   スレッドコンテキスト：メインスレッド
 */
void TrayApp::detectIdle()
{
    DebugLog2("TrayApp::detectIdle()");

    // ここはメインスレッドなのでARPを用意する
    AutoreleasePool *pool = AutoreleasePool::alloc()->init();

    // lock
    RaymLock(this);

    if (_idle_count == 0)
    {
        DebugLog0("detect idle...");
    }

    // アイドルカウンタを更新
    ++_idle_count;

    // 起動中アプリと休止状態抑止アプリのチェック
    bool found = false;

    Array *dont_in_suspend = _props->arrayForKey(KEY_DO_NOT_IN_SHUTDOWN);
    Array *running_apps = Workspace::sharedWorkspace()->runningApplications();
    for (uint i = 0; (i < running_apps->count()) && !found; ++i)
    {
        RunningApplication *ra = (RunningApplication *)running_apps->objectAtIndex(i);

        // 実行中でなければ次へ
        if (!ra->isRunning())
        {
            continue;
        }

        // 実行ファイルのチェック
        String *path = ra->executePath();
        if ((path == NULL) || (path->length() == 0))
        {
            continue;
        }
        DebugLog3("exec path: %s", path->cString());

        // 休止状態抑止アプリリストのチェック
        for (uint j = 0; (j < dont_in_suspend->count()) && !found; ++j)
        {
            found = path->isMatch((String *)dont_in_suspend->objectAtIndex(j));
        }
    }

    // 抑止有効なら KEY_FORCED_SUSPEND_TIME、無効なら KEY_SUSPEND_TIME で取得
    int limit = _props->integerForKey(KEY_FORCED_SHUTDOWN);
    if (found && (_idle_count < limit))
    {
        notification();
    }

    // unlock
    RaymUnlock(this);

    // ARP解放
    pool->release();
}

void TrayApp::detectNonIdle()
{
    DebugLog2("TrayApp::detectNonIdle()");

    // lock
    RaymLock(this);

    if (_idle_count > 0)
    {
        DebugLog0("detect non idle...");
    }
    _idle_count = 0;

    // unlock
    RaymUnlock(this);

    notification();
}

int TrayApp::start()
{
    // ログファイル数設定
    //   設定以前にログ出力しないこと
    Raym::LOG_NUM_MAX = 8;

#ifdef RAYM_MEMORY_CHECK
    DebugLog0("");
    DebugLog0("global_raym_count_ = %d", Raym::global_raym_count_);
#endif

    //
    AutoreleasePool *pool = AutoreleasePool::alloc()->init();

    // メンバ初期化
    _system_path        = NULL;
    _props              = NULL;
    _idle_count         = 0;

    DebugLog0("");
    DebugLog0("------------------------------------------------------------------------");
    DebugLog0("iPTd ver %s (rev.%d)", VERSION, REVISION);
    DebugLog0("initialize...");

    int result = 0;

    // 初期化
    while (true)
    {
        // システムパス設定
        _system_path = String::stringWithFormat("%s", GetExecutePath());
        if (_system_path == NULL)
        {
            DebugLog0("error: GetExecutePath()");
            result = -1;
            break;
        }
        _system_path = _system_path->stringByReplacingOccurrencesOfString("iPTd.exe", "");
        _system_path->retain();
        DebugLog2("_system_path: %s", _system_path->cString());

        // プロパティの読み込み
        _props = Dictionary::alloc()->initWithContentsOfFile(String::stringWithFormat("%s%s.iptd.plist", _system_path->cString(), PLIST_PREFIX));
        if (_props == NULL)
        {
            DebugLog0("property file read ng.");
            result = -1;
            break;
        }

        if (_props->integerForKey(KEY_HTTP_PORT) == 0)
        {
            DebugLog0("property check ng. 00");
            result = -1;
            break;
        }

        // 強制シャットダウンまでの時間
        if ((_props->integerForKey(KEY_FORCED_SHUTDOWN) == 0) || (_props->integerForKey(KEY_FORCED_SHUTDOWN) < DEF_FORCED_SHUTDOWN))
        {
            DebugLog0("property check ng. 01");
            result = -1;
            break;
        }

        // シャットダウン抑止アプリリスト
        if (_props->arrayForKey(KEY_DO_NOT_IN_SHUTDOWN) == NULL)
        {
            DebugLog0("property check ng. 02");
            result = -1;
            break;
        }

        // プロパティの確認
        DebugLog0("  Forced Shutdown    : %d min", _props->integerForKey(KEY_FORCED_SHUTDOWN));
        Array *apps = _props->arrayForKey(KEY_DO_NOT_IN_SHUTDOWN);
        DebugLog0("  Do not in shutdown :");
        for (uint i = 0; i < apps->count(); ++i)
        {
            DebugLog0("    RegExp[%02d]     : %s", i, ((String *)apps->objectAtIndex(i))->cString());
        }

        result = Application::start();
        break;
    }

    // 解放
    RELEASE(_system_path);
    RELEASE(_props);

    pool->release();

    // 終了
    DebugLog0("finished.");

#ifdef RAYM_MEMORY_CHECK
    DebugLog0("global_raym_count_ = %d", Raym::global_raym_count_);
#endif

    return result;
}

TrayApp::TrayApp()
{
}

TrayApp::~TrayApp()
{
}

TrayApp *TrayApp::alloc()
{
    return new TrayApp();
}

void TrayApp::notification()
{
    char http_req[256];
    sprintf_s(http_req, "http://localhost:%d/non_idle.xml", _props->integerForKey(KEY_HTTP_PORT));
    URL *url = URL::URLWithString(http_req);
    URLRequest *req = URLRequest::requestWithURL(url);
    URLResponse *resp = NULL;
    Error *error = NULL;
    Data *data = URLConnection::sendSynchronousRequest(req, &resp, &error);
}

} // iPTd
} // ry0

/*
 * Win32 アプリケーションの初期エントリポイント
 */
int WINAPI wWinMain(_In_ HINSTANCE hInstance,
                   _In_opt_ HINSTANCE hPrevInstance,
                   _In_ LPTSTR    lpCmdLine,
                   _In_ int       nCmdShow)
{
    return ApplicationMain(ry0::iPTd::TrayApp, TEXT("ry0.app.iptd"), hInstance, hPrevInstance, lpCmdLine, nCmdShow);
}

