#ifndef _EX_Buffer_H
#define _EX_Buffer_H

#include "Prefix.h"
#include "EARTH_PT3.h"
#include "EX_Utility.h"
#include "OS_Memory.h"
#include <vector>

namespace EARTH {
namespace EX {
	class Buffer {
	public:
		Buffer(PT::Device *device) : mDevice(device)
		{
			for (uint32 i=0; i<sizeof(mInfo)/sizeof(*mInfo); i++) {
				Info	*info = &mInfo[i];
				info->memory      = NULL;
				info->ptr         = NULL;
				info->size        = 0;
			}
		}

		~Buffer()
		{
			Free();
		}

		status Alloc(uint32 blockSize, uint32 blockCount, bool loop = true)
		{
			status	status;
			
			status = AllocMemory(blockSize, blockCount);
			if (status) return status;

			BuildPageDescriptor(loop);

			status = mDevice->SyncBufferCpu(mInfo[Index_Page].physicals[0].handle);
			if (status) return status;

			return status;
		}

		status Free()
		{
			status	status;
			status = FreeMemory();
			if (status) return status;

			return status;
		}

		status SyncCpu(uint32 blockIndex)
		{
			return mDevice->SyncBufferCpu(mInfo[Index_TS].physicals.at(blockIndex).handle);
		}

		status SyncIo(uint32 blockIndex)
		{
			return mDevice->SyncBufferIo(mInfo[Index_TS].physicals.at(blockIndex).handle);
		}

		void *Ptr(uint32 blockIndex)
		{
			Info	*info = &mInfo[Index_TS];
			uint8	*ptr = static_cast<uint8 *>(info->ptr);
			ptr += info->blockSize * blockIndex;
			return ptr;
		}

		uint64 PageDescriptorAddress()
		{
			_ASSERT(mInfo[Index_Page].physicals[0].bufferInfo);
			uint64	address = mInfo[Index_Page].physicals[0].bufferInfo[0].Address;

			return address;
		}

	protected:
		enum Index {
			Index_TS,		// TS f[^̈
			Index_Page,		// y[WLqq̈
			Index_Count
		};

		struct Physical {
			void							*handle;
			const PT::Device::BufferInfo	*bufferInfo;
			uint32							bufferCount;
		};

		struct Info {
			OS::Memory				*memory;
			void					*ptr;
			uint32					size;
			uint32					blockSize;
			uint32					blockCount;
			std::vector<Physical>	physicals;
		};

		PT::Device		*const mDevice;
		Info			mInfo[Index_Count];

		status AllocMemory(uint32 blockSize, uint32 blockCount)
		{
			status	status = PT::STATUS_OK;

			// TS f[^̈mۂ
			status = AllocMemory(Index_TS, blockSize , blockCount);
			if (status) return status;

			// y[WLqq̈mۂ
			// Lqq̃TCY 20 oCgȂ̂ŁA4096 oCg̗̈ɍ\złLqq̐ 204 łB
			uint32	pageCount = (mInfo[Index_TS].size / OS::Memory::PAGE_SIZE + 203) / 204;
			status = AllocMemory(Index_Page, OS::Memory::PAGE_SIZE * pageCount, 1);
			if (status) return status;

			return status;
		}

		status AllocMemory(Index index, uint32 blockSize, uint32 blockCount)
		{
			status	status = PT::STATUS_OK;

			_ASSERT(index < Index_Count);
			Info	*info = &mInfo[index];

			// mۂ
			uint32	size = blockSize * blockCount;

			_ASSERT(info->memory == NULL);
			info->memory = new OS::Memory(size);
			if (info->memory == NULL) return PT::STATUS_OUT_OF_MEMORY_ERROR;

			uint8	*ptr = static_cast<uint8 *>(info->memory->Ptr());
			info->ptr        = ptr;
			info->size       = size;
			info->blockSize  = blockSize;
			info->blockCount = blockCount;
			if (ptr == NULL) return PT::STATUS_OUT_OF_MEMORY_ERROR;
			
			Physical	block;
			block.handle      = NULL;
			block.bufferInfo  = NULL;
			block.bufferCount = 0;

			// DMA ]ł悤ɃubNɕɌŒ肷
			PT::Device::TransferDirection	direction = (index == Index_TS) ?
				PT::Device::TRANSFER_DIRECTION_WRITE : PT::Device::TRANSFER_DIRECTION_READ;

			uint32	offset = 0;
			for (uint32 i=0; i<blockCount; i++) {
				status = mDevice->LockBuffer(ptr + offset, blockSize, direction, &block.handle);
				if (U::Error(L"Device::LockBuffer()", status)) return status;

				status = mDevice->GetBufferInfo(block.handle, &block.bufferInfo, &block.bufferCount);
				if (U::Error(L"Device::GetBufferInfo()", status)) return status;

				info->physicals.push_back(block);

				offset += blockSize;
			}

			return status;
		}

		status FreeMemory()
		{
			status	status = PT::STATUS_OK;

			for (uint32 i=0; i<sizeof(mInfo)/sizeof(*mInfo); i++) {
				Info	*info = &mInfo[i];
				std::vector<Physical>	&physicals = info->physicals;
				for (uint32 i=0; i<physicals.size(); i++) {
					Physical	*block = &physicals[i];
					if (block->handle) {
						status = mDevice->UnlockBuffer(block->handle);
						if (U::Error(L"Device::UnlockBuffer()", status)) return status;
					}
				}
				physicals.clear();

				if (info->memory) {
					delete_(info->memory);
					info->memory = NULL;
					info->ptr    = NULL;
					info->size   = 0;
				}
			}

			return status;
		}

		// ̃Tvł 4096 oCgɂ 1̋Lqq\zĂ܂B
		// OS::Memory Ŋmۂ郁TCY 4096 ̔{ŁAOS::Memory
		// Ŏ擾ł鉼zAhX 4096 oCgEɂȂĂ̂ŁA
		// FPGA.txt ɋLqĂu4GB ׂy[Ẅ 2ɕv
		// ͍lKvȂȂ܂B
		void BuildPageDescriptor(bool loop)
		{
			void							*pagePtr       = mInfo[Index_Page].ptr;
			std::vector<Physical>			&pagePhysicals = mInfo[Index_Page].physicals;
			const PT::Device::BufferInfo	*pageInfo      = pagePhysicals[0].bufferInfo;

			uint64	pageAddress = pageInfo->Address;
			uint32	pageSize    = pageInfo->Size;
			pageInfo++;

			// [vpɐ擪̕AhXۑĂ
			uint64	loopPageAddress = pageAddress;

			void	*prevPagePtr = NULL;

			std::vector<Physical>	&tsPhysicals = mInfo[Index_TS].physicals;
			for (uint32 i=0; i<tsPhysicals.size(); i++) {
				Physical						*physical = &tsPhysicals[i];
				const PT::Device::BufferInfo	*tsInfo   = physical->bufferInfo;
				uint32							tsCount   = physical->bufferCount;

				for (uint32 j=0; j<tsCount; j++) {
					uint64	tsAddress = tsInfo->Address;
					uint32	tsSize    = tsInfo->Size;
					tsInfo++;

					_ASSERT((tsSize % OS::Memory::PAGE_SIZE) == 0);
					for (uint32 k=0; k<tsSize/OS::Memory::PAGE_SIZE; k++) {
						// c肪 20 oCgǂ`FbN
						if (pageSize < 20) {
							// c蕔XLbv
							pagePtr = static_cast<uint8 *>(pagePtr) + pageSize;

							// AhXƃTCYXV
							pageAddress = pageInfo->Address;
							pageSize    = pageInfo->Size;
							pageInfo++;
						}

						// ŐLqq΁A̋LqqƂȂ
						if (prevPagePtr) {
							LinkPageDescriptor(pageAddress, prevPagePtr);
						}

						// ̋Lqq|C^ۑ
						prevPagePtr = pagePtr;

						// Lqq\z (nextAddress ͌_ł͂킩Ȃ 0 ɂĂ)
						WritePageDescriptor(tsAddress, OS::Memory::PAGE_SIZE, 0, &pagePtr);

						tsAddress += OS::Memory::PAGE_SIZE;

						pageAddress += 20;
						pageSize    -= 20;
					}
				}
			}

			if (prevPagePtr) {
				if (loop) {
					// 擪փ[v
					LinkPageDescriptor(loopPageAddress, prevPagePtr);
				} else {
					//  DMA ]I
					LinkPageDescriptor(1, prevPagePtr);
				}
			}
		}

		void WritePageDescriptor(uint64 address, uint32 size, uint64 nextAddress, void **ptr_)
		{
			_ASSERT(ptr_);
			uint8	*ptr = static_cast<uint8 *>(*ptr_);
			_ASSERT(ptr);

			*reinterpret_cast<uint64 *>(ptr +  0) = address     | 7;	// [2:0] ͖
			*reinterpret_cast<uint32 *>(ptr +  8) = size        | 7;	// [2:0] ͖
			*reinterpret_cast<uint64 *>(ptr + 12) = nextAddress | 2;	// [1  ] ͖

			*ptr_ = ptr + 20;
		}

		void LinkPageDescriptor(uint64 nextAddress, void *ptr_)
		{
			_ASSERT(ptr_);
			uint8	*ptr = static_cast<uint8 *>(ptr_);
			_ASSERT(ptr);

			*reinterpret_cast<uint64 *>(ptr + 12) = nextAddress | 2;	// [1] ͖
		}
	};
}
}

#endif

