/*
* avc_vcr.c - An example of an AV/C Tape Recorder target implementation
*
* Copyright Dan Dennedy <dan@dennedy.org>
* 
* Inspired by virtual_vcr from Bonin Franck <boninf@free.fr>
* 
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/
#include <signal.h>
#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/select.h>

#include <libraw1394/raw1394.h>
#include <libavc1394/avc1394.h>

#include "handler.h"
#include "mpeg2.h"
#include "status.h"
#include "playlist.h"

const char not_compatible[] = "\n"
    "This libraw1394 does not work with your version of Linux. You need a different\n"
    "version that matches your kernel (see kernel help text for the raw1394 option to\n"
    "find out which is the correct version).\n";

const char not_loaded[] = "\n"
    "This probably means that you don't have raw1394 support in the kernel or that\n"
    "you haven't loaded the raw1394 module.\n";

int g_done = 0;

void sigHandler(int sig) {
    g_done = 1;
    unsetPlaying();
    unsetRecording();
    unsetPaused();
}

static int isTs(char *argv) {
    int len;

    len = strlen(argv);
    if(len < 3) {
        return 0;
    } else {
        if(argv[len - 3] == '.' &&
           tolower(argv[len - 2]) == 't' &&
           tolower(argv[len - 1]) == 's')
            return 1;
        else
            return 0;
    }
}


int main(int argc, char **argv) {
    if(argc < 2) {
        fprintf(stderr, "usage> %s playlist\n", argv[0]);
        fprintf(stderr, "usage> %s hoge0.ts hoge1.ts ...\n", argv[0]);
        exit(EXIT_FAILURE);
    }

    int n = 0;
    int i = 1;
    if(strcmp(argv[1], "-d") == 0) {
        setDebug();
        i = 2;
    }
    if(isTs(argv[i]))
        n = initPlaylistFromArg(argv+i-1);
    else {
        fp = fopen(argv[i], "r");
        if(fp == NULL) {
            fprintf(stderr, "Error: Cannot open playlist\n");
            exit(EXIT_FAILURE);
        }
        n = initPlaylistFromFile(fp);
        fclose(fp);
    }

    printf("%d files\n", n);
    //checkPlaylist();
    
    raw1394handle_t handle;
    int result = 0;
    
    handle = raw1394_new_handle_on_port(0);
    
    if(handle == NULL) {
        if(!errno) {
            printf(not_compatible);
        } else {
            perror("couldn't get handle");
            fprintf(stderr, not_loaded);
        }
        exit(EXIT_FAILURE);
    }

    signal(SIGINT, sigHandler);
    signal(SIGTERM, sigHandler);
    
    avc1394_init_target(handle, command_handler);
    printf("Starting AV/C target; press Ctrl+C to quit...\n");

    int fd = raw1394_get_fd(handle);
    struct timeval tv;
    fd_set rfds;
    do {
        FD_ZERO (&rfds);
        FD_SET (fd, &rfds);
        
        tv.tv_sec = 0;
        tv.tv_usec = 200000;
        
        if (select (fd + 1, &rfds, NULL, NULL, &tv) > 0)
            result = raw1394_loop_iterate(handle);
    } while(g_done == 0);

    avc1394_close_target(handle);
    raw1394_destroy_handle(handle);
    printf("Done.\n");
    
    return EXIT_SUCCESS;
}
