#include <time.h>
#include <signal.h>
#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <assert.h>

#include <libraw1394/raw1394.h>
#include <libavc1394/avc1394.h>

#include "mpeg2.h"
#include "handler.h"
#include "status.h"
#include "playlist.h"

#define printf if(isDebug())printf
#define fprintf if(isDebug())fprintf

unsigned char g_signal_mode = 0x08; // HD 1125-60, TODO: get from media
unsigned char g_transport_mode = AVC1394_VCR_CMD_WIND;
unsigned char g_transport_state = AVC1394_VCR_OPERAND_WIND_STOP;

inline unsigned char convertBcd(unsigned char n) {
    return 16*(n/10)+(n%10);
}

/**** subunit handlers ****/
int subunit_control( avc1394_cmd_rsp *cr )
{
//    printf("%s\n", __FUNCTION__);
    switch ( cr->opcode )
    {
    case AVC1394_VCR_CMD_PLAY:
        switch ( cr->operand[0] )
        {
        case AVC1394_VCR_OPERAND_PLAY_FORWARD:
        case AVC1394_VCR_OPERAND_PLAY_SLOWEST_FORWARD:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_FORWARD_6:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_FORWARD_5:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_FORWARD_4:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_FORWARD_3:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_FORWARD_2:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_FORWARD_1:
        case AVC1394_VCR_OPERAND_PLAY_X1_FORWARD:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = AVC1394_VCR_OPERAND_PLAY_FORWARD;
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PLAY FORWARD\n");
            if(!isPlaying()) {
                setPlaying();
                pthread_create(&play_thread, NULL, mpeg2_transmit, NULL);
            }
            if(isPaused())
               unsetPaused();
            break;        
        case AVC1394_VCR_OPERAND_PLAY_FASTEST_FORWARD:
        case AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_1:
        case AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_2:
        case AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_3:
        case AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_4:
        case AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_5:
        case AVC1394_VCR_OPERAND_PLAY_FAST_FORWARD_6:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = AVC1394_VCR_OPERAND_PLAY_FASTEST_FORWARD;
            cr->status = AVC1394_RESP_ACCEPTED;
            setForward();
            printf("PLAY FASTEST FORWARD\n");
            break;
        case AVC1394_VCR_OPERAND_PLAY_REVERSE_PAUSE:
        case AVC1394_VCR_OPERAND_PLAY_FORWARD_PAUSE:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PAUSE PLAY\n");
            if(isPlaying()) {
                if(isPaused()) {
                    printf("unset Pause\n");
                    unsetPaused();
                } else {
                    printf("set Pause\n");
                    setPaused();
                }
            }
            break;
        case AVC1394_VCR_OPERAND_PLAY_REVERSE:
        case AVC1394_VCR_OPERAND_PLAY_SLOWEST_REVERSE:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_REVERSE_6:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_REVERSE_5:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_REVERSE_4:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_REVERSE_3:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_REVERSE_2:
        case AVC1394_VCR_OPERAND_PLAY_SLOW_REVERSE_1:
        case AVC1394_VCR_OPERAND_PLAY_X1_REVERSE:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = AVC1394_VCR_OPERAND_PLAY_REVERSE;
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PLAY REVERSE\n");
            break;
        case AVC1394_VCR_OPERAND_PLAY_FASTEST_REVERSE:
        case AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_1:
        case AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_2:
        case AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_3:
        case AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_4:
        case AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_5:
        case AVC1394_VCR_OPERAND_PLAY_FAST_REVERSE_6:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = AVC1394_VCR_OPERAND_PLAY_FASTEST_REVERSE;
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PLAY FASTEST REVERSE\n");
            break;
        case AVC1394_VCR_OPERAND_PLAY_NEXT_FRAME:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PLAY NEXT FRAME\n");
            break;
        case AVC1394_VCR_OPERAND_PLAY_PREVIOUS_FRAME:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PLAY PREVIOUS FRAME\n");
            break;
        default:
            fprintf( stderr, "play mode 0x%02x non supported\n", cr->operand[0] );
            return 0;
        }
        break;
    case AVC1394_VCR_CMD_RECORD:
        switch ( cr->operand[0] )
        {
        case AVC1394_VCR_OPERAND_RECORD_RECORD:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_RECORD );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            setRecording();
            printf("RECORD\n");
            break;
        case AVC1394_VCR_OPERAND_RECORD_PAUSE:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_RECORD );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("PAUSE RECORD\n");
            break;
        default:
            fprintf( stderr, "record mode 0x%02x non supported\n", cr->operand[0] );
            return 0;
        }
        break;
    case AVC1394_VCR_CMD_WIND:
        switch ( cr->operand[0] )
        {
        case AVC1394_VCR_OPERAND_WIND_STOP:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_WIND );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("STOP\n");
            unsetPlaying();
            unsetRecording();
            unsetPaused();
            break;
        case AVC1394_VCR_OPERAND_WIND_REWIND:
        case AVC1394_VCR_OPERAND_WIND_HIGH_SPEED_REWIND:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_WIND );
            g_transport_state = AVC1394_VCR_OPERAND_WIND_REWIND;
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("REWIND\n");
            break;
        case AVC1394_VCR_OPERAND_WIND_FAST_FORWARD:
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_WIND );
            g_transport_state = cr->operand[0];
            cr->status = AVC1394_RESP_ACCEPTED;
            printf("FAST FORWARD\n");
            break;
        default:
            fprintf( stderr, "wind mode 0x%02x non supported\n", cr->operand[0] );
            return 0;
        }
        break;
    case AVC1394_VCR_CMD_TAPE_RECORDING_FORMAT:
        cr->status = AVC1394_RESP_ACCEPTED;
        cr->operand[0] = 1 << 3;
        cr->operand[7] = 0xff;
        break;
    case AVC1394_VCR_CMD_BACKWARD:
        cr->status = AVC1394_RESP_ACCEPTED;
        cr->operand[0] = 0; //0でいいのか未確認
        for(; cr->operand[1] > 0; cr->operand[1]--)
            setPrevFile();
        printf("Prev File\n");
        break;
    case AVC1394_VCR_CMD_FORWARD:
        cr->status = AVC1394_RESP_ACCEPTED;
        cr->operand[0] = 0; //0でいいのか未確認
        for(; cr->operand[1] > 0; cr->operand[1]--)
            setNextFile();
        printf("Next File\n");
        break;
    case AVC1394_VCR_CMD_INPUT_SIGNAL_MODE:
        cr->status = AVC1394_RESP_ACCEPTED;
        g_signal_mode = cr->operand[0];
        printf("AVC1394_VCR_CMD_INPUT_SIGNAL_MODE\tg_signal_mode = %d\n", g_signal_mode);
    default:
        fprintf( stderr, "subunit control command 0x%02x non supported\n", cr->opcode );
        return 0;
    }

    return 1;
}

int subunit_status( avc1394_cmd_rsp *cr ) {
    //  printf("%s opcode = %02x\n", __FUNCTION__, cr->opcode);
    switch ( cr->opcode )
    {
    case AVC1394_VCR_CMD_OUTPUT_SIGNAL_MODE:
        printf("AVC1394_VCR_CMD_OUTPUT_SIGNAL_MODE\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = g_signal_mode;
        break;
    case AVC1394_VCR_CMD_INPUT_SIGNAL_MODE:
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = g_signal_mode;
        break;
    case AVC1394_VCR_CMD_TRANSPORT_STATE:
        if(isPlaying()) {
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_PLAY );
            if(isPaused())
                g_transport_state = AVC1394_VCR_OPERAND_PLAY_FORWARD_PAUSE;
            else 
                g_transport_state = AVC1394_VCR_OPERAND_PLAY_FORWARD;
        } else if(isRecording()) {
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_RECORD );
        } else {
            g_transport_mode = AVC1394_GET_OPCODE( AVC1394_VCR_RESPONSE_TRANSPORT_STATE_WIND );
            g_transport_state = AVC1394_VCR_OPERAND_WIND_STOP;
        }
        cr->status = AVC1394_RESP_STABLE;
        cr->opcode = g_transport_mode;
        cr->operand[0] = g_transport_state;
        break;
    case AVC1394_VCR_CMD_TIME_CODE:
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = AVC1394_VCR_OPERAND_RECORDING_TIME_STATUS;
        // TODO: extract timecode from media or use time elapsed since start of app
        cr->operand[1] = 1; //frames
        cr->operand[2] = 2; //seconds
        cr->operand[3] = 3; //minutes
        cr->operand[4] = 4; //hours
        break;
    case AVC1394_VCR_CMD_MEDIUM_INFO:
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = AVC1394_VCR_OPERAND_MEDIUM_INFO_VHS;
        cr->operand[1] = 0x50;
        break;
    case AVC1394_VCR_CMD_TAPE_PLAYBACK_FORMAT:
        printf("AVC1394_VCR_CMD_TAPE_PLAYBACK_FORMAT\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = 0x08;
        cr->operand[1] = 0x00;
        break;
    case AVC1394_VCR_CMD_RELATIVE_TIME_COUNTER:
        cr->operand[1] = 0x0;  //frames
        cr->operand[2] = convertBcd(56);
        cr->operand[3] = convertBcd(34); //min
        cr->operand[4] = convertBcd(12); //hour BCD
        cr->status = AVC1394_RESP_STABLE;
        break;
    case AVC1394_VCR_CMD_RECORDING_SPEED:
        printf("AVC1394_VCR_CMD_RECORDING_SPEED\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = 0x6f; //0x6f = standard speed
        break;
    case AVC1394_VCR_CMD_TAPE_RECORDING_FORMAT:
        printf("AVC1394_VCR_CMD_TAPE_RECORDING_FORMAT\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = 1 << 3;
        break;
    default:
        fprintf( stderr, "subunit status command 0x%02x not supported\n", cr->opcode );
        return 0;
    }

    return 1;
}


int subunit_inquiry( avc1394_cmd_rsp *cr ) {
    //printf("%s\n", __FUNCTION__);
    switch ( cr->opcode )
    {
    case AVC1394_VCR_CMD_PLAY:
    case AVC1394_VCR_CMD_RECORD:
    case AVC1394_VCR_CMD_WIND:
    case AVC1394_VCR_CMD_OUTPUT_SIGNAL_MODE:
    case AVC1394_VCR_CMD_INPUT_SIGNAL_MODE:
    case AVC1394_VCR_CMD_TRANSPORT_STATE:
    case AVC1394_VCR_CMD_TIME_CODE:
    case AVC1394_VCR_CMD_MEDIUM_INFO:
        cr->status = AVC1394_RESP_IMPLEMENTED;
        return 1;
    default:
        fprintf( stderr, "subunit inquiry command 0x%02x not supported\n", cr->opcode );
        return 0;
    }
    return 1;
}


/**** Unit handlers ****/
int unit_control( avc1394_cmd_rsp *cr ) {
    //printf("%s\n", __FUNCTION__);
    switch ( cr->opcode )
    {
    case AVC1394_CMD_CONNECT:
        cr->opcode = AVC1394_RESP_ACCEPTED;
        printf("AVC1394_CMD_CONNECT: %d (%02x:%02x) -> %d (%02x:%02x)\n",
               cr->operand[2],
               cr->operand[1] >> 3, cr->operand[1] & 0x7,
               cr->operand[4],
               cr->operand[3] >> 3, cr->operand[3] & 0x7);
        set_oPCR(cr->operand[2]);
        set_iPCR(cr->operand[4]);
        break;
    case AVC1394_CMD_POWER:
        cr->opcode = AVC1394_RESP_ACCEPTED;
        printf("AVC1394_CMD_POWER\n");
        break;
    default:
        fprintf( stderr, "unit control command 0x%02x not supported\n", cr->opcode );
        return 0;
    }

    return 1;
}

int unit_status(avc1394_cmd_rsp *cr) {
    //printf("%s\n", __FUNCTION__);
    cr->operand[1] = 0xff;
    cr->operand[2] = 0xff;
    cr->operand[3] = 0xff;
    cr->operand[4] = 0xff;
    switch ( cr->opcode )
    {
    case AVC1394_CMD_UNIT_INFO:
        printf("AVC1394_CMD_UNIT_INFO\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = AVC1394_OPERAND_UNIT_INFO_EXTENSION_CODE;
        cr->operand[1] = AVC1394_SUBUNIT_TAPE_RECORDER;
        break;
    case AVC1394_CMD_SUBUNIT_INFO: {
        printf("AVC1394_CMD_SUBUNIT_INFO\n");
        //int ext_code = cr->operand[0] & 7;
        int page = ( cr->operand[0] >> 4 ) & 7;
        if ( page == 0 )
        {
            cr->status = AVC1394_RESP_STABLE;
            cr->operand[0] = (page << 4) | AVC1394_OPERAND_UNIT_INFO_EXTENSION_CODE;
            cr->operand[1] = AVC1394_SUBUNIT_TAPE_RECORDER << 3;
        }
        else
        {
            fprintf( stderr, "invalid page %d for subunit\n", page );
            return 0;
        }
        break;
    }
    case AVC1394_CMD_POWER:
        printf("AVC1394_CMD_POWER\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = 0x70; // power on
        break;
    case AVC1394_CMD_CONNECT:
        printf("AVC1394_CMD_CONNECT\n");
        cr->status = AVC1394_RESP_STABLE;
        cr->operand[0] = 0xff;
        break;
    case 0x00:
        printf("AVC1394_CMD_VENDOR_DEPENDENT\n");
        cr->status = AVC1394_RESP_STABLE;
        break;
    default:
        fprintf( stderr, "unit status command 0x%02x not supported\n", cr->opcode );
        return 0;
    }

    return 1;
}


int unit_inquiry( avc1394_cmd_rsp *cr ) {
    //printf("%s\n", __FUNCTION__);
    switch ( cr->opcode )
    {
    case AVC1394_CMD_SUBUNIT_INFO:
    case AVC1394_CMD_UNIT_INFO:
        cr->status = AVC1394_RESP_IMPLEMENTED;
    default:
        fprintf( stderr, "unit inquiry command 0x%02x not supported\n", cr->opcode );
        return 0;
    }

    return 1;
}

/**** primary avc1394 target callback ****/
int command_handler( avc1394_cmd_rsp *cr ) {
    switch ( cr->subunit_type )
    {
    case AVC1394_SUBUNIT_TAPE_RECORDER:
        if ( cr->subunit_id != 0 ) {
            fprintf( stderr, "subunit id 0x%02x not supported\n", cr->subunit_id );
            return 0;
        }
        switch ( cr->status )
        {
        case AVC1394_CTYP_CONTROL:
            return subunit_control( cr );
            break;
        case AVC1394_CTYP_STATUS:
            return subunit_status( cr );
            break;
        case AVC1394_CTYP_GENERAL_INQUIRY:
            return subunit_inquiry( cr );
            break;
        }
        break;
    case AVC1394_SUBUNIT_UNIT:
        switch ( cr->status )
        {
        case AVC1394_CTYP_CONTROL:
            return unit_control( cr );
            break;
        case AVC1394_CTYP_STATUS:
            return unit_status( cr );
            break;
        case AVC1394_CTYP_GENERAL_INQUIRY:
            return unit_inquiry( cr );
            break;
        }
        break;
    default:
        fprintf( stderr, "subunit type 0x%02x not supported\n", cr->subunit_type );
        return 0;
    }

    return 1;
}
