#include "htlib2.h"
#include <fcntl.h>
#include "Socket.h"
#include "Thread.h"
#include "Time.h"

BOOL
M(Socket,Open)(HTLIB* o,int domain, int type, int protocol, HTLIB_ERROR* err)
{
	LOGDBG("before open socket: %d", o->soc);
	if ((o->soc = socket(domain, type, protocol))==-1) {
		o->system_errno = errno;
		*err = HTLIB_E_SYSTEM;
		LOGERR("socket failed err=%d", errno);
		return FALSE;
	}
	LOGINFO("open socket %d", o->soc);
	return TRUE;
}
void
M(Socket,Close)(HTLIB* o)
{
	if (o->soc != -1) {
		LOGINFO("close(%d)", o->soc);
		close(o->soc);
		o->soc = -1;
	}
}

BOOL
M(Socket,GetAddrInfo)(HTLIB* o,
					  const char* host, const char* service,
					  const struct addrinfo* hints,
					  struct addrinfo** info, HTLIB_ERROR* err)
{
	if ((o->system_errno=getaddrinfo(host, service, hints, info))!=0) {
		LOGERR("getaddrinfo returns: %s", gai_strerror(o->system_errno));
		*err = HTLIB_E_SYSTEM;
		return FALSE;
	}
	return TRUE;
}
void
M(Socket,FreeAddrInfo)(HTLIB* o, struct addrinfo* info)
{
	freeaddrinfo(info);
}

BOOL
M(Socket,SetSockOpt)(HTLIB* o, int level, int optname,
					 const void* optval, socklen_t len, HTLIB_ERROR* err)
{
	if (setsockopt(o->soc, level, optname, optval, len)==-1) {
		o->system_errno = errno;
		*err = HTLIB_E_SYSTEM;
		LOGERR("setsockopt failed errno=%d", errno);
		return FALSE;
	}
	return TRUE;
}
BOOL
M(Socket,SetNonBlock)(HTLIB* o, BOOL dontblock, HTLIB_ERROR* err)
{
	int flags = fcntl(o->soc, F_GETFL, 0);
	if (flags == -1) {
		*err = HTLIB_E_SYSTEM;
		o->system_errno = errno;
		LOGERR("fcntl failed errno=%d", errno);
		return FALSE;
	}
	if (dontblock) {
		flags |= O_NONBLOCK;
	} else {
		flags &= ~O_NONBLOCK;
	}
	if (fcntl(o->soc, F_SETFL, flags | O_NONBLOCK)==-1) {
		*err = HTLIB_E_SYSTEM;
		o->system_errno = errno;
		LOGERR("fcntl failed errno=%d", errno);
		return FALSE;
	}
	return TRUE;
}

BOOL
M(Socket,Connect)(HTLIB* o,
				  const struct sockaddr* addr, socklen_t len,
				  HTLIB_ERROR* err)
{
	LOGNOTICE("Socket_Connect(family=%d, s_addr=%x, sin_port=%hu) len=%d",
			addr->sa_family,
			ntohl(((struct sockaddr_in*)addr)->sin_addr.s_addr),
			ntohs(((struct sockaddr_in*)addr)->sin_port),
			len);
	
	if (connect(o->soc, addr, len)==-1) {
		o->system_errno = errno;
		*err = HTLIB_E_SYSTEM;
		LOGERR("connect failed errno=%d", o->system_errno);
		return FALSE;
	}
	return TRUE;
}

BOOL
M(Socket,Write)(HTLIB* o, const char* data, int len, HTLIB_ERROR* err)
{
	if (len<0) {
		len = strlen(data);
	}
	LOGDBG("%.*s", len, data);
	while (len>0) {
		int l = len;
		if (l>1000) {
			l = 1000;
		}
		if ((l=send(o->soc, data, l, MSG_NOSIGNAL))==-1) {
			o->system_errno = errno;
			*err = HTLIB_E_SYSTEM;
			LOGERR("SYSTEM errno=%d", o->system_errno);
			return FALSE;
		}
		len -= l;
		data += l;
	}
	return TRUE;
}

int
M(Socket,Read)(HTLIB* o, char* buffer, int buffer_len, HTLIB_ERROR* err)
{
	int len;
	if ((len=recv(o->soc, buffer, buffer_len, 0))==-1) {
		o->system_errno = errno;
		*err = HTLIB_E_SYSTEM;
		LOGERR("recv failed errno=%d", o->system_errno);
		return -1;
	}
	LOGDBG("recv returns %d", len);
	return len;
}

#define TIMEOUT_INTERVAL 100

BOOL
M(Socket,Wait)(HTLIB* o,
			   int timeout_millis,
			   HTLIB_WAIT w,
			   BOOL (*checkCanceled)(HTLIB* o, HTLIB_ERROR* err),
			   HTLIB_ERROR* err)
{
	struct timeval tv;
	fd_set readfs;
	fd_set writefs;

	if (checkCanceled(o, err)==FALSE) {
		return FALSE;
	}

	if (timeout_millis < 0) {
		if (FLAG_ISSET(FLAG_REPORTED_NO_SELECT, o->_flags)==FALSE) {
			FLAG_SET(FLAG_REPORTED_NO_SELECT, o->_flags);
			LOGINFO("no select (-1), so may cause blocking");
		}
		return TRUE;
	}

	int elapsed = G(GetElapsedMillis)(o->_called);
	
	do {
		fd_set* rp = NULL;
		fd_set* wp = NULL;


		int t = TIMEOUT_INTERVAL;
		int rest = timeout_millis-elapsed;
		if (rest<TIMEOUT_INTERVAL) {
			t = rest;
			if (t<0) {
				t = 0;
			}
		}

		tv.tv_sec = t / 1000;
		tv.tv_usec = (t%1000) * 1000;
	
		FD_ZERO(&readfs);
		FD_ZERO(&writefs);
		switch (w) {
		case HTLIB_W_READ:
			FD_SET(o->soc, &readfs);
			rp = &readfs;
			break;

		case HTLIB_W_WRITE:
			FD_SET(o->soc, &writefs);
			wp = &writefs;
			break;

		default:
			*err = HTLIB_E_INVALID_ARG;
			return FALSE;
		}

		switch (select(o->soc+1, rp, wp, NULL, &tv)) {
		case 0:
			if (checkCanceled(o, err)==FALSE) {
				return FALSE;
			}
			break;

		case -1:
			o->system_errno = errno;
			*err = HTLIB_E_SYSTEM;
			LOGERR("SYSTEM errno=%d", o->system_errno);
			return FALSE;

		default:
			if ((rp!=NULL && FD_ISSET(o->soc, rp))||
				(wp!=NULL && FD_ISSET(o->soc, wp))) {
				return TRUE;
			}
			*err = HTLIB_E_INVALID_STATE;
			return FALSE;
		}

	} while ((elapsed = G(GetElapsedMillis)(o->_called))< timeout_millis);

	*err = HTLIB_E_TIMEOUT;
	return FALSE;
}

