package com.interpress_project.modernshare.client.controller.model;

import org.apache.commons.io.FilenameUtils;

import com.interpress_project.modernshare.client.events.LocalViewEvent;
import com.interpress_project.modernshare.client.events.LocalViewEventSource;
import com.interpress_project.modernshare.client.events.ViewEventType;
import com.interpress_project.modernshare.ipcommon.SystemBase;

/**
 * @stereotype Singleton factory
 */
public class LocalModelManager {
	private final SystemBase sb = SystemBase.getInstance();
	private final LocalViewEventSource localViewEventSource = LocalViewEventSource.getInstance();
	private final String filesep = sb.getFileSeparator();

	/**
	 * @stereotype uses
	 */
	/*#com.interpress_project.modernshare.client.controller.model.XmlDAOManager Dependency20*/
	private LocalRootLineItem lineItem = null;

	/**
	 * @supplierCardinality 0...1
	 */
	private XmlDAOManager daomgr;

	/**
	 * Holds singleton instance
	 */
	private static LocalModelManager instance;

	/**
	 * LocalItemManager
	 */
	protected LocalModelManager() {
		daomgr = new XmlDAOManager();
		lineItem = new LocalRootLineItem();
	}

	/**
	 * Returns the singleton instance.
	 @return	the singleton instance
	 */
	static public LocalModelManager getInstance() {
		if (instance == null) {
			instance = new LocalModelManager();
		}
		return instance;
	}

	/**
	 * load
	 * Build hashtable for LocalRoots.
	 */
	public void load() {
		daomgr.load();
		daomgr.needXMLSave(false); // Need remote store.

		LocalRoot[] localRoots = daomgr.getLocalRoots();
		for (int i = 0; i < localRoots.length; i++) {
			lineItem.addRoot(localRoots[i]);
		}
	}

	/**
	 * addRoot
	 * @param path
	 * @param name
	 * @param status
	 * @param bPc2host
	 */
	public void addRoot(String path, String name, RootStatus status, boolean bPc2host) {
		LocalRoot root = new LocalRoot(path, name, status, bPc2host);
		lineItem.addRoot(root);
		daomgr.setLocalRoots(lineItem.getRoots());
		daomgr.save();
		daomgr.needXMLSave(true); // Need remote store.

		localViewEventSource.fireEvent(new LocalViewEvent(this, ViewEventType.REFRESH));
	}

	/**
	 * canRegisterMore
	 * ő̓o^m[h5܂
	 * @return
	 */
	public boolean canRegisterMore() {
		return lineItem.canRegisterMore();
	}

	/**
	 * removeRoot
	 * @param path
	 */
	public void removeRoot(String path) {
		lineItem.removeRoot(path);
		daomgr.setLocalRoots(lineItem.getRoots());
		daomgr.save();
		daomgr.needXMLSave(true); // Need remote store.		

		localViewEventSource.fireEvent(new LocalViewEvent(this, ViewEventType.REFRESH));
	}

	/**
	 * removeRootByTagname
	 * @param tagname
	 */
	public void removeRootByTagname(String tagname) {
		lineItem.removeRootByTagname(tagname);
		daomgr.save();
		daomgr.needXMLSave(true); // Need remote store.		

		localViewEventSource.fireEvent(new LocalViewEvent(this, ViewEventType.REFRESH));
	}

	/**
	 * setStatus
	 * @param path
	 * @param status
	 */
	public void setStatus(String path, RootStatus status) {
		LocalRoot root = lineItem.getRoot(path);
		if (root == null) {
			return;
		}
		root.setStatus(status);
		daomgr.setLocalRoots(lineItem.getRoots());
		daomgr.save();
		daomgr.needXMLSave(true); // Need remote store.		

		localViewEventSource.fireEvent(new LocalViewEvent(this, ViewEventType.REFRESH));
	}

	/**
	 * isPathOverwrapped
	 * @param path
	 * @return
	 */
	public boolean isPathOverwrapped(String path) {
		String a1, a2;
		String newpath = FilenameUtils.getFullPath(path + filesep);

		LocalRoot[] root = lineItem.getRoots();
		for (int i = 0; i < root.length; i++) {
			String existpath = FilenameUtils.getFullPath(root[i].getAbsolutePath() + filesep);

			if (newpath.length() > existpath.length()) {
				a1 = newpath;
				a2 = existpath;
			}
			else {
				a1 = existpath;
				a2 = newpath;
			}

			if (a1.indexOf(a2) != -1) {
				return true;
			}
		}
		return false;
	}

	/**
	 * isExistTagnameInRocal
	 * @param tagName
	 * @return
	 */
	public boolean isExistTagnameInRocal(String tagName) {
		return lineItem.isExistTagnameInRocal(tagName);
	}

	/**
	 * getRootLineItem
	 * @return
	 */
	public LocalRootLineItem getRootLineItem() {
		return lineItem;
	}

	/**
	 * hasXMLLoaded
	 * @return
	 */
	public boolean hasXMLLoaded() {
		return daomgr.hasXMLLoaded();
	}

	/**
	 * hasXMLModified
	 * @return
	 */
	public boolean needXMLSave() {
		return daomgr.needXMLSave();
	}

	/**
	 * setXMLModified
	 * @param b
	 */
	public void needXMLSave(boolean b) {
		daomgr.needXMLSave(b);
	}

	/**
	 * dispose
	 */
	public void dispose() {
		if (daomgr != null) {
			daomgr.setLocalRoots(lineItem.getRoots());
			daomgr.save();
		}
		lineItem.dispose();
	}
}
