﻿Option Explicit On
Option Strict On

Imports System.Windows.Media.Animation

''' <summary>ページスライド・編集用コントロール。</summary>
''' <remarks>
''' 表示しているページの変更と、名前を編集するコントロール。
''' </remarks>
Public Class SlidePageControl

    ''' <summary>格納できるページタイトル最大数。</summary>
    Private PAGE_NAME_MAX As Integer = 8

#Region "events"

    ''' <summary>ページ名変更イベント。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="selectIndex">選択されたページのインデックス。</param>
    ''' <param name="newName">新しい名前。</param>
    Public Event ChangePageName(sender As Object, selectIndex As Integer, newName As String)

    ''' <summary>ページスライドが開始されたことを通知するイベント。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event SlideAnimeRun(sender As Object)

    ''' <summary>ページスライドが終了されたことを通知するイベント。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="slideCount">スライドした数。</param>
    Public Event SlideAnimeFinish(sender As Object, slideCount As Integer)

    ''' <summary>ページ追加イベント。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト</param>
    Public Event NotifyAddPage(sender As Object, e As RoutedEventArgs)

    ''' <summary>ページ削除イベント。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト</param>
    Public Event NotifyRemovePage(sender As Object, e As RoutedEventArgs)

#End Region

    ' 表示しているページ名コントロールリスト
    Private mPageNames As List(Of PageNameControl)

    ' ページ名を押下したときの時間を記憶
    Private mDownTime As Date

    ' スライドするページ数を記憶
    Private mSlideCount As Integer

    ''' <summary>コンストラクタ。</summary>
    ''' <remarks>インスタンスを初期化する。</remarks>
    Public Sub New()
        InitializeComponent()

        ' ページ名コントロールのリストを作成
        Me.mPageNames = New List(Of PageNameControl)
        Me.mPageNames.Add(Me.CreatePageName(1))
        For i As Integer = 0 To PAGE_NAME_MAX - 1
            Me.mPageNames.Add(Me.CreatePageName(0.9))
        Next

        ' スタックコントロールに追加
        For Each pageCtrl As PageNameControl In Me.mPageNames
            Me.pageNameStack.Children.Add(pageCtrl)
        Next
    End Sub

#Region "新しいページ名コントロールを作成する"

    ' 新しいページ名コントロールを作成する
    Private Function CreatePageName(opacity As Double) As PageNameControl
        Dim res As New PageNameControl()

        ' 設定変更
        res.HorizontalAlignment = Windows.HorizontalAlignment.Left
        res.VerticalAlignment = Windows.VerticalAlignment.Center
        res.Margin = New Thickness(8, 0, 0, 0)
        res.MinWidth = 40
        res.FontSize = 13
        res.Opacity = opacity
        If opacity < 1 Then
            res.Foreground = Brushes.DarkGray
        Else
            res.Foreground = Brushes.White
        End If

        ' イベント設定
        AddHandler res.MouseDown, AddressOf pageName_MouseDown
        AddHandler res.MouseUp, AddressOf pageName_MouseUp

        Return res
    End Function

#End Region

    ''' <summary>コントロールに名前リストを追加する。</summary>
    ''' <param name="titles">名前リスト。</param>
    ''' <remarks>
    ''' コントロールに名前リストを追加する。
    ''' </remarks>
    Public Sub SetNames(titles As List(Of String))
        For i As Integer = 0 To titles.Count - 1
            Me.mPageNames(i).PageName = titles(i)
        Next

        For i As Integer = titles.Count To mPageNames.Count - 1
            Me.mPageNames(i).PageName = String.Empty
        Next
    End Sub

    ''' <summary>ページ名コントロールをマウス押下したときの処理。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub pageName_MouseDown(sender As Object, e As MouseButtonEventArgs)
        Me.mDownTime = Date.Now
    End Sub

    ''' <summary>ページ名コントロールをマウスリリースしたときの処理。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub pageName_MouseUp(sender As Object, e As MouseButtonEventArgs)
        ' クリックされたコントロールのインデックスを取得
        Dim pageCtrl As PageNameControl = DirectCast(sender, PageNameControl)
        Me.mSlideCount = TargetPageNameIndex(pageCtrl)
        If Me.mSlideCount < 0 Then Return

        ' 操作に対する処理を実行
        ' 1. 名前編集処理
        ' 2. ページスライド処理
        If Date.Now.Subtract(Me.mDownTime).TotalSeconds > 0.5 Then
            Me.EditPageName(pageCtrl) ' 1
        Else
            Me.SlidePageNameRun(sender) ' 2
        End If
    End Sub

#Region "マウスリリースに対する処理の実装"

    ' 引数で与えられたコントロールのインデックスを取得する
    Private Function TargetPageNameIndex(sender As PageNameControl) As Integer
        Dim res As Integer = -1

        ' 引数のオブジェクトと一致するコントロールのインデックスを取得する
        For i As Integer = 0 To Me.mPageNames.Count - 1
            If Me.mPageNames(i) Is sender Then
                res = i
                Exit For
            End If
        Next

        ' ページに名前が設定されていたらインデックスを返す
        If Me.mPageNames(res).PageName <> String.Empty Then
            Return res
        Else
            Return -1
        End If
    End Function

    ' ページ名編集コントロールの表示
    Private Sub EditPageName(target As PageNameControl)
        ' ポップアップを開く
        With Me.pageNameInput
            .PlacementTarget = target
            .Placement = Primitives.PlacementMode.Relative
            .Width = 110
            .Height = target.ActualHeight + 2
            .IsOpen = True
        End With

        ' テキストにフォーカスを与える
        With Me.pageNameInputText
            .Text = target.PageName
            .SelectAll()
            .Focus()
        End With
    End Sub

    ''' <summary>スライドアニメ開始処理。</summary>
    ''' <param name="sender">スライド対象コントロール。</param>
    Private Sub SlidePageNameRun(sender As Object)
        If mSlideCount > 0 Then
            ' スライド開始を通知
            RaiseEvent SlideAnimeRun(Me)

            ' アニメーション開始
            Dim pos As Point = Me.PointFromScreen(DirectCast(sender, PageNameControl).PointToScreen(New Point(0, 0)))
            With DirectCast(Me.Resources("SlideAnime"), Storyboard)
                Dim frames As ThicknessAnimationUsingKeyFrames = CType(.Children(0), ThicknessAnimationUsingKeyFrames)
                frames.KeyFrames(1).Value = New Thickness(-pos.X + 8, 0, 0, 0)
                .Begin()
            End With
        End If
    End Sub

    ''' <summary>スライドアニメ終了イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub SlidePageNameFinish(sender As Object, e As EventArgs)
        RaiseEvent SlideAnimeFinish(Me, Me.mSlideCount)
    End Sub

#End Region

    ''' <summary>ページ編集中のキー押下イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub pageNameInputText_PreviewKeyDown(sender As Object, e As KeyEventArgs) _
                                                    Handles pageNameInputText.PreviewKeyDown
        ' Enterキーを押下したら、編集終了
        If e.Key = Key.Enter Then
            Me.pageNameInput.IsOpen = False
        End If
    End Sub

    ''' <summary>ページ名編集ポップアップ終了イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub pageNameInput_Closed(sender As Object, e As EventArgs) Handles pageNameInput.Closed
        If Me.pageNameInputText.Text.Trim() <> String.Empty Then
            RaiseEvent ChangePageName(Me, Me.mSlideCount, Me.pageNameInputText.Text)
        End If
    End Sub

    ''' <summary>ページ追加イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub AddButton_Click(sender As Object, e As RoutedEventArgs)
        RaiseEvent NotifyAddPage(Me, e)
        e.Handled = True
    End Sub

    ''' <summary>ページ削除イベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub RemoveButton_Click(sender As Object, e As RoutedEventArgs)
        RaiseEvent NotifyRemovePage(Me, e)
        e.Handled = True
    End Sub

    ''' <summary>名前編集コントロールの表示を取得する。</summary>
    ''' <value>真偽値。</value>
    Public ReadOnly Property IsEditOpen() As Boolean
        Get
            Return Me.pageNameInput.IsOpen
        End Get
    End Property

End Class
