﻿Option Explicit On
Option Strict On

''' <summary>アイコンセルを格納したパネルコントロール。</summary>
''' <remarks>
''' 要素（ショートカット／エグゼ）のアイコンを格納した表を表現する。
''' </remarks>
Public NotInheritable Class IconsControl

    ''' <summary>コントロールの上部パディング。</summary>
    Private Const TOP_PADDING As Integer = 68

    ''' <summary>アイコンの半分のサイズ。</summary>
    Private Const ICON_HALF As Integer = 24

    ''' <summary>アイコンの列最大値。</summary>
    Private Const CELL_CLM_MAX As Integer = 5

    ''' <summary>アイコンの行最大値。</summary>
    Private Const CELL_ROW_MAX As Integer = 4

#Region "events"

    ''' <summary>コントロールに要素（ショートカット／エグゼ）をドロップされたことを通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="dropPath">ドロップされたファイルパス。</param>
    ''' <param name="dropPos">ドロップされた位置。</param>
    Public Event DropOfFile(sender As Object, dropPath As String, dropPos As Point)

    ''' <summary>コントロールのアイコンが移動されたことを通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="fromCell">移動開始セル。</param>
    ''' <param name="toCell">移動終了セル。</param>
    Public Event DropOfIcon(sender As Object, fromCell As IconCell, toCell As IconCell)

    ''' <summary>マウスがアイコンにホバー中であることを通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event NotifyHovering(sender As Object, hoverIconCell As IconCell)

    ''' <summary>コントロールよりマウスが離脱したことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <remarks>イベントオブジェクト。</remarks>
    Public Event NotifyMouseLeave(sender As Object)

    ''' <summary>アイコンが左クリックされたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event Clicked(sender As Object, hoverIconCell As IconCell)

    ''' <summary>アイコンが右クリックされたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event ClickedByRight(sender As Object, hoverIconCell As IconCell)

    ''' <summary>アイコンが削除されたことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    Public Event Removed(sender As Object, hoverIconCell As IconCell)

#End Region

    ' アイコン格納コントロールリスト
    Private mIcons As List(Of IconCell)

    ' ドラッグ処理中のファイルパス
    Private mDragFilePath As String

    ' ドラッグファイルのゴースト
    Private mDragFileGhost As DragAdorner

    ' ドラッグアイコンのゴースト
    Private mDragIconGhost As DragAdorner

    ' 移動開始セルを記憶する
    Private mFromCell As IconCell

    ''' <summary>コンストラクタ。</summary>
    ''' <remarks>
    ''' インスタンスの初期化を行う。
    ''' </remarks>
    Public Sub New()
        InitializeComponent()

        ' アイコン格納領域を動的に生成する
        Me.mIcons = New List(Of IconCell)
        For r As Integer = 0 To CELL_ROW_MAX - 1
            For c As Integer = 0 To CELL_CLM_MAX - 1
                Dim cell As IconCell = Me.CreateCell(r, c)
                Me.mIcons.Add(cell)
                Me.uniGrid.Children.Add(cell)
            Next
        Next
    End Sub

#Region "アイコン格納コントロールとイベントハンドラ"

    ' 新しいアイコン格納コントロールを生成する。
    Private Function CreateCell(row As Integer, column As Integer) As IconCell
        Dim res As New IconCell(row, column)
        AddHandler res.NotifyHovering, AddressOf IconCell_NotifyHovering
        AddHandler res.Clicked, AddressOf IconCell_Clicked
        AddHandler res.ClickedByRight, AddressOf IconCell_ClickedByRight
        AddHandler res.Removed, AddressOf IconCell_Removed
        AddHandler res.DragStart, AddressOf IconCell_DragStart
        Return res
    End Function

    ' ホバー中イベントハンドラ
    Private Sub IconCell_NotifyHovering(sender As Object)
        RaiseEvent NotifyHovering(Me, DirectCast(sender, IconCell))
    End Sub

    ' 左クリックイベントハンドラ
    Private Sub IconCell_Clicked(sender As Object)
        RaiseEvent Clicked(Me, DirectCast(sender, IconCell))
    End Sub

    ' 右クリックイベントハンドラ
    Private Sub IconCell_ClickedByRight(sender As Object)
        RaiseEvent ClickedByRight(Me, DirectCast(sender, IconCell))
    End Sub

    ' アイコン削除イベントハンドラ
    Private Sub IconCell_Removed(sender As Object)
        RaiseEvent Removed(Me, DirectCast(sender, IconCell))
    End Sub

#End Region

    ''' <summary>アイコン格納コントロールの画像情報を削除する。</summary>
    Public Sub ClearIconImage()
        For Each icon As IconCell In Me.mIcons
            icon.IconImage = Nothing
        Next
    End Sub

#Region "マウス座標変換"

    ' マウス座標をコントロールの座標に変換
    Private Function ConvertMousePosition() As Point
        With Me.PointToScreen(DragAdorner.GetNowPosition(Me))
            Return New Point(.X, .Y - TOP_PADDING)
        End With
    End Function

    ' マウス座標を半透明ゴースト用座標に変換
    Private Function GhostIconPosition() As Point
        With Me.PointFromScreen(Me.PointToScreen(DragAdorner.GetNowPosition(Me)))
            Return New Point(.X, .Y - TOP_PADDING)
        End With
    End Function

#End Region

#Region "ファイルドラッグ処理"

    ' ドラッグイベントハンドラ
    '   ショートカットまたはエグゼがドラッグされてきたとき、ゴーストを表示する。
    Private Sub UserControl_DragEnter(sender As Object, e As DragEventArgs)
        ' ドラッグ情報を取得する
        Dim file = e.Data.GetData(DataFormats.FileDrop)
        e.Effects = DragDropEffects.None

        ' ドラッグされたファイルを調査し、対象ファイルならゴーストを表示する
        If file IsNot Nothing Then
            Dim files() As String = DirectCast(file, String())
            If Text.RegularExpressions.Regex.IsMatch(files(0), "(lnk|exe)$") Then
                e.Effects = DragDropEffects.Move
                DragGhostEffect(files(0))
            End If
        Else
            Dim obj = e.Data.GetData(GetType(IconCell))
            If obj IsNot Nothing Then
                e.Effects = DragDropEffects.Move
            End If
        End If
        e.Handled = True
    End Sub

    ' ドラッグファイルのゴーストを表示する
    Private Sub DragGhostEffect(ByVal path As String)
        Me.IsHovering = True

        ' エフェクト用のゴーストを生成する
        If Me.mDragFileGhost Is Nothing Then
            Me.mDragFileGhost = New DragAdorner(Me, SellApis.GetIcon(path, SellApis.SHIL_EXTRALARGE))
        Else
            Me.GhostLayer.Remove(Me.mDragFileGhost)
            If Me.mDragFilePath <> path Then
                Me.mDragFileGhost = New DragAdorner(Me, SellApis.GetIcon(path, SellApis.SHIL_EXTRALARGE))
            End If
        End If

        ' ゴーストをマウス位置に表示する
        Me.GhostLayer.Add(Me.mDragFileGhost)
        Dim widPos As Point = GhostIconPosition()
        Me.mDragFileGhost.Left = widPos.X - ICON_HALF
        Me.mDragFileGhost.Top = widPos.Y - ICON_HALF
        Me.mDragFilePath = path
    End Sub

    ' ドラッグ中イベントハンドラ
    Private Sub UserControl_DragOver(sender As Object, e As DragEventArgs)
        If Me.mDragFileGhost IsNot Nothing Then
            Dim widPos As Point = GhostIconPosition()
            Me.mDragFileGhost.Left = widPos.X - ICON_HALF
            Me.mDragFileGhost.Top = widPos.Y - ICON_HALF
        End If
    End Sub

#End Region

#Region "アイコンドラッグ処理"

    ' ドラッグ開始イベントハンドラ
    Private Sub IconCell_DragStart(sender As Object)
        ' クリックされた IconCell を記憶
        Dim clickCell As IconCell = DirectCast(sender, IconCell)
        Me.mFromCell = clickCell

        ' ドラッグ用のゴーストを作成
        Dim newPos As Point = GhostIconPosition()
        Me.mDragIconGhost = New DragAdorner(Me, clickCell.IconImage)
        Me.mDragIconGhost.Left = newPos.X - ICON_HALF
        Me.mDragIconGhost.Top = newPos.Y - ICON_HALF

        ' ドラッグ・ドロップを実行
        Me.GhostLayer.Add(Me.mDragIconGhost)
        DragDrop.DoDragDrop(clickCell, clickCell, DragDropEffects.All)
        Me.HideGhostEffect()
    End Sub

    ' ドラッグによるゴースト移動処理
    Private Sub UserControl_QueryContinueDrag(sender As Object, e As QueryContinueDragEventArgs)
        If Me.mDragIconGhost IsNot Nothing Then
            Dim clickpos As Point = GhostIconPosition()
            Me.mDragIconGhost.Left = clickpos.X - ICON_HALF
            Me.mDragIconGhost.Top = clickpos.Y - ICON_HALF
        End If
    End Sub

#End Region

    ''' <summary>ドロップイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub UserControl_Drop(sender As Object, e As DragEventArgs)
        ' ファイルドラック通知
        If Me.mDragFileGhost IsNot Nothing Then
            RaiseEvent DropOfFile(Me, Me.mDragFilePath, e.GetPosition(Me))
        End If

        ' アイコンドラッグ通知
        If Me.mDragIconGhost IsNot Nothing Then
            Dim clickpos As Point = GhostIconPosition()
            Dim clickCel As IconCell = Me(clickpos)
            If clickCel IsNot Nothing Then
                RaiseEvent DropOfIcon(Me, Me.mFromCell, clickCel)
            End If
        End If

        Me.HideGhostEffect()
    End Sub

    ''' <summary>ドラッグ処理が終了するイベントハンドラ。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub UserControl_DragLeave(sender As Object, e As DragEventArgs)
        Dim widPos As Point = GhostIconPosition()
        If widPos.X <= 0 OrElse widPos.X >= Me.ActualWidth OrElse
           widPos.Y <= 0 OrElse widPos.Y >= Me.ActualHeight Then
            Me.HideGhostEffect()
        End If
    End Sub

    ''' <summary>マウスがコントロールより離脱したことを外部に通知する。</summary>
    ''' <param name="sender">イベント発行元。</param>
    ''' <param name="e">イベントオブジェクト。</param>
    Private Sub UserControl_MouseLeave(sender As Object, e As MouseEventArgs)
        RaiseEvent NotifyMouseLeave(Me)
    End Sub

    ''' <summary>ドラッグゴーストを消去する。</summary>
    Private Sub HideGhostEffect()
        ' ファイルドラッグを消去
        If Me.mDragFileGhost IsNot Nothing Then
            Me.GhostLayer.Remove(Me.mDragFileGhost)
            Me.mDragFileGhost = Nothing
        End If
        Me.mDragFilePath = String.Empty

        ' アイコンドラッグを消去
        If Me.mDragIconGhost IsNot Nothing Then
            Me.GhostLayer.Remove(Me.mDragIconGhost)
            Me.mDragIconGhost = Nothing
        End If

        ' ホバーモードを解除
        Me.IsHovering = False
    End Sub

#Region "properties"

    ''' <summary>半透明ゴーストを表示するレイヤー。</summary>
    ''' <value>AdornerLayer型。</value>
    Private ReadOnly Property GhostLayer() As AdornerLayer
        Get
            Return AdornerLayer.GetAdornerLayer(Me)
        End Get
    End Property

    ''' <summary>指定された列・行のアイコン格納コントロールを取得する。</summary>
    ''' <param name="row">行位置。</param>
    ''' <param name="clm">列位置。</param>
    Default Public ReadOnly Property Item(row As Integer, clm As Integer) As IconCell
        Get
            Return Me.mIcons(row * CELL_CLM_MAX + clm)
        End Get
    End Property

    ''' <summary>指定したマウス位置のコントロールを取得する。</summary>
    ''' <param name="mousePoint">マウス位置。</param>
    Default Public ReadOnly Property Item(mousePoint As Point) As IconCell
        Get
            For Each icon As IconCell In Me.mIcons
                Dim pos As Point = Me.PointFromScreen(icon.PointToScreen(New Point(0, 0)))
                If pos.X < mousePoint.X AndAlso pos.X + 64 > mousePoint.X AndAlso
                   pos.Y < mousePoint.Y AndAlso pos.Y + 64 > mousePoint.Y Then
                    Return icon
                End If
            Next
            Return Nothing
        End Get
    End Property

    ''' <summary>コントロールのホバーモードを変更する。</summary>
    ''' <value>真偽値。</value>
    Public Property IsHovering() As Boolean
        Get
            Return Me.mIcons(0).Hovering
        End Get
        Set(value As Boolean)
            For Each icon As IconCell In Me.mIcons
                icon.Hovering = value
            Next
        End Set
    End Property

    ''' <summary>アプリケーションがドラッグ中ならば真を返す。</summary>
    ''' <value>真偽値。</value>
    Public ReadOnly Property IsDragging() As Boolean
        Get
            Return Me.mDragFileGhost IsNot Nothing OrElse Me.mDragIconGhost IsNot Nothing
        End Get
    End Property

#End Region

End Class
