﻿Option Strict On
Option Explicit On

''' <summary>アイコンコレクションクラス。</summary>
''' <remarks>
''' アイコン情報の管理を行う。
''' </remarks>
Public NotInheritable Class IconCollection

    ' カレントページインデックス
    Private mCurIndex As Integer

    ' ページタイトルリスト
    Private mPageTitles As List(Of String)

    ' ページごとのアイコンリスト
    Private mIcons As List(Of List(Of IconInfo))

    ''' <summary>コンストラクタ。</summary>
    Public Sub New()
        Me.mPageTitles = New List(Of String)
        Me.mIcons = New List(Of List(Of IconInfo))

        Me.mCurIndex = 0
        Me.mPageTitles.Add("新規")
        Me.mIcons.Add(New List(Of IconInfo))
    End Sub

    ''' <summary>コンストラクタ。</summary>
    ''' <param name="root">ルート要素。</param>
    ''' <param name="pointPage">初期表示ページ。</param>
    Public Sub New(root As XElement, pointPage As Integer)
        Me.mPageTitles = New List(Of String)
        Me.mIcons = New List(Of List(Of IconInfo))

        ' 表示するカレントページのインデックスを取得
        If pointPage >= 0 Then
            Me.mCurIndex = pointPage
        Else
            Me.mCurIndex = Integer.Parse(root.Element("pages").Attribute("current").Value)
        End If

        ' 格納しているアイコン情報を取得する
        ' 1. ページのタイトルを取得する
        ' 2. アイコンリストを取得する
        Dim allPages = From c In root.Elements.<page> Select c
        For Each page In allPages
            Me.mPageTitles.Add(page.Attribute("title").Value)   ' 1

            Dim lnks As New List(Of IconInfo)() ' 2
            For Each lnk In page.<link>
                Dim c As Integer = Integer.Parse(lnk.Attribute("clm").Value)
                Dim r As Integer = Integer.Parse(lnk.Attribute("row").Value)
                Dim n As String = lnk.Attribute("name").Value
                lnks.Add(New IconInfo(c, r, n, lnk.Value))
            Next
            Me.mIcons.Add(lnks)
        Next
    End Sub

    ''' <summary>アイコン情報を XMLに書き込む。</summary>
    ''' <param name="root">ルート要素。</param>
    Public Sub Save(root As XElement)
        Dim pages As New XElement("pages", New XAttribute("current", Me.mCurIndex))

        For i As Integer = 0 To Me.mPageTitles.Count - 1
            Dim page As New XElement("page", New XAttribute("title", Me.mPageTitles(i)))

            For Each lnkInfo As IconInfo In Me.mIcons(i)
                Dim lnk As New XElement("link",
                    New XAttribute("row", lnkInfo.Row),
                    New XAttribute("clm", lnkInfo.Column),
                    New XAttribute("name", lnkInfo.Name)
                )
                lnk.Value = lnkInfo.Path
                page.Add(lnk)
            Next
            pages.Add(page)
        Next
        root.Add(pages)
    End Sub

#Region "ページ操作処理"

    ''' <summary>カレントを指している位置をインクリメントする。</summary>
    ''' <param name="count">インクリメントする数。</param>
    Public Sub IncrementPage(count As Integer)
        Me.mCurIndex += count
        If Me.mCurIndex >= Me.mPageTitles.Count Then
            Me.mCurIndex -= Me.mPageTitles.Count
        End If
    End Sub

    ''' <summary>カレントを指している位置を変更する。</summary>
    ''' <param name="index">変更する数。</param>
    Public Sub ChangePage(index As Integer)
        Me.mCurIndex = index
        If Me.mCurIndex >= Me.mPageTitles.Count Then
            Me.mCurIndex -= Me.mPageTitles.Count
        End If
    End Sub

    ''' <summary>カレントにページを追加する。</summary>
    Public Sub InsertPage()
        Me.mPageTitles.Insert(Me.mCurIndex, "新規")
        Me.mIcons.Insert(Me.mCurIndex, New List(Of IconInfo))
    End Sub

    ''' <summary>カレントのページを削除する。</summary>
    Public Sub RemovePage()
        If Me.mPageTitles.Count > 1 Then
            Me.mPageTitles.RemoveAt(Me.mCurIndex)
            Me.mIcons.RemoveAt(Me.mCurIndex)
        End If
    End Sub

    ''' <summary>指定位置のページのタイトルを更新する。</summary>
    ''' <param name="newTitle">新しいページタイトル。</param>
    ''' <param name="tarIndex">変更対象のページの参照位置。</param>
    Public Sub UpdatePageTitle(newTitle As String, tarIndex As Integer)
        Dim newIdx As Integer = Me.mCurIndex + tarIndex
        If newIdx >= Me.mPageTitles.Count Then
            newIdx -= Me.mPageTitles.Count
        End If
        Me.mPageTitles(newIdx) = newTitle
    End Sub

#End Region

#Region "properties"

    ''' <summary>ページのタイトルリストを取得する。</summary>
    ''' <value>List(Of String)型。</value>
    Public ReadOnly Property PageTitles() As List(Of String)
        Get
            Dim res As New List(Of String)
            Dim ptr As Integer = Me.mCurIndex

            ' カレントからのリストを作成する
            Do While res.Count < Me.mPageTitles.Count
                res.Add(Me.mPageTitles(ptr))
                ptr += 1
                If ptr >= Me.mPageTitles.Count Then ptr = 0
            Loop
            Return res
        End Get
    End Property

    ''' <summary>カレントページのインデックスを取得する。</summary>
    ''' <value>整数値。</value>
    Public ReadOnly Property Current() As Integer
        Get
            Return Me.mCurIndex
        End Get
    End Property

    ''' <summary>カレントのアイコン情報リストを取得する。</summary>
    ''' <value>List(Of IconInfo)型。</value>
    Public ReadOnly Property CurrentIcons() As List(Of IconInfo)
        Get
            Return Me.mIcons(Me.mCurIndex)
        End Get
    End Property

    ''' <summary>全てのアイコン情報のリストを取得する。</summary>
    ''' <value>List(Of IconInfo)型。</value>
    Public ReadOnly Property AllIconInfo() As List(Of IconInfo)
        Get
            Dim res As New List(Of IconInfo)
            For Each lst As List(Of IconInfo) In Me.mIcons
                res.AddRange(lst)
            Next
            Return res
        End Get
    End Property

#End Region

End Class
