/*
 * Copyright [yyyy] [name of copyright owner]
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */


// Anonymous function start
//
(function( window, undefined )
{

// reference
var Config			= window.h5glib.Config;
var Debug			= window.h5glib.Debug;
var Command			= window.h5glib.Command;
var Task			= window.h5glib.Task;
var SoundTask		= window.h5glib.SoundTask;
var ReadyTask		= window.h5glib.ReadyTask;
var ImageAnimator	= window.h5glib.ImageAnimator;
var Message			= window.h5glib.Message;
var MessageHandler	= window.h5glib.MessageHandler;
var MessageManager	= window.h5glib.MessageManager;
var SceneStatus		= window.h5glib.SceneStatus;
var Scene			= window.h5glib.Scene;

/**
 * Res
 */
var Res =
{
	String :
	{
		HTML_USAGE	: "<p>" +
					  "アイコンは全部ダミーです。" +
					  "</p>",
		TWEET_NAME	: "shuzo"
	},
	Color :
	{
		BACKGROUND		: "white",
		HLINE			: "lightsteelblue",
		FONT			: "black"
	},
	Font :
	{
		TWEET_NAME		: "bold 14px 'Times New Roman'",
		TWEET_DATE		: "normal 10px 'Times New Roman'",
		TWEET_TEXT		: "bold 12px 'ＭＳ Ｐゴシック'",
		LARGE			: "bold 18px 'ＭＳ Ｐゴシック'",
		SMALL			: "bold 14px 'ＭＳ Ｐゴシック'"
	}
};

/**
 * StageTask
 */
var StageTask = function( scene )
{
	this.command	= scene.command;	// input
};
StageTask.prototype = new Task();

(function( pt )
{
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;

		context.save();

		context.fillStyle	= Res.Color.BACKGROUND;
		context.fillRect( 0, 0, canvas.width, canvas.height );

		var image;
		// header
		image = resource.image.header.data;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			0,				// dx
			0,				// dy
			image.width,	// dw
			image.height	// dh
		);
		// footer
		image = resource.image.footer.data;
		context.drawImage(
			image,
			0,				// sx
			0,				// sy
			image.width,	// sw
			image.height,	// sh
			0,				// dx
			canvas.height - image.height,	// dy
			image.width,	// dw
			image.height	// dh
		);

		context.restore();
	};
})( StageTask.prototype );

/**
 * TweetListTask
 */
var TweetListTask = function( scene )
{
	this.command		= scene.command;
	this.selector		= null;
	this.tweets			= [];
	this.lastTime		= 0;
};
TweetListTask.prototype = new Task();

(function( pt )
{
	pt.INTERVAL		= 3000;
	pt.TWEETS_MAX	= 5;
	pt.TOP_X		= 0;
	pt.TOP_Y		= 40;
	pt.HEADER_HEIGHT= 40;
	pt.FOOTER_HEIGHT= 40;
	pt.TWEET_HEIGHT	= 60;

	/**
	 * 
	 */
	pt.setData = function( scene, sentences )
	{
		this.selector	= new SentenceSelector( sentences );
		for ( var i = 0; i < this.TWEETS_MAX; i++ )
		{
			this.tweets[i] = new TweetTask( scene );
			this.tweets[i].init( this.selector );
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		var nowTime = (new Date()).getTime();
		if ( this.lastTime == 0 )
		{
			this.lastTime	= nowTime;
		}
		// update tweet list
		if ( ( nowTime - this.lastTime ) >  this.INTERVAL )
		{
			// add new tweet
			this.tweets.shift();
			var tweet = new TweetTask( scene );
			tweet.init( this.selector );
			this.tweets.push( tweet );

			this.lastTime	= nowTime;
			upd = true;
		}

		// update children
		for ( var i = 0; i < this.tweets.length; i++ )
		{
			var tweet = this.tweets[i];
			if ( tweet.update( scene ) )
			{
				upd = true;
			}
		}
		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		for ( var i = 0; i < this.tweets.length; i++ )
		{
			var tweet = this.tweets[i];
			tweet.trans.x = 0;
			tweet.trans.y = this.TOP_Y + ( i * this.TWEET_HEIGHT );
			tweet.draw( scene );
		}
	};
})( TweetListTask.prototype );

/**
 * TweetTask
 */
var TweetTask = function( scene )
{
	this.command		= scene.command;
	this.trans			= { x : 0, y : 0 };
	this.sentences		= [];
	this.date			= new Date();
};
TweetTask.prototype = new Task();

(function( pt )
{
	pt.TWEET_HEIGHT	= 60;
	pt.FONT_HEIGHT	= 18;
	pt.LINECHAR_MAX	= 16;

	/**
	 * 
	 */
	pt.init = function( selector )
	{
		var str = selector.getSentence();
		if ( str.length <= this.LINECHAR_MAX )
		{
			this.sentences[0] = str;
		}
		else
		{
			this.sentences[0] = str.substring( 0, this.LINECHAR_MAX );
			this.sentences[1] = str.substring( this.LINECHAR_MAX );
		}
	};
	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;

		return upd;
	};
	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		var context		= scene.context;
		var canvas		= context.canvas;
		var resource	= scene.data.resource;

		context.save();

		// draw icon
		var image = resource.image.icon.data;
		context.drawImage(
			image,
			0,					// sx
			0,					// sy
			image.width,		// sw
			image.height,		// sh
			this.trans.x + 4,	// dx
			this.trans.y + 1,	// dy
			image.width,		// dw
			image.height		// dh
		);

		// draw name
		context.font		= Res.Font.TWEET_NAME;
		context.fillStyle	= Res.Color.FONT;
		context.fillText( Res.String.TWEET_NAME, this.trans.x + 60, this.trans.y + 14 );

		// draw date
		context.font		= Res.Font.TWEET_DATE;
		context.fillStyle	= Res.Color.FONT;
		context.fillText( this.date.getMonth() + "/" + this.date.getDate() + " " + this.date.getHours() + ":" + this.date.getMinutes(),
							canvas.width - 60, this.trans.y + 14 );

		// draw text
		context.font		= Res.Font.TWEET_TEXT;
		context.fillStyle	= Res.Color.FONT;
		for ( var i = 0; i < this.sentences.length; i++ )
		{
			context.fillText( this.sentences[i], this.trans.x + 58, this.trans.y + this.FONT_HEIGHT * ( i + 2 ) );
		}

		// draw horizontal line
		context.strokeStyle = Res.Color.HLINE;
		var y = this.trans.y + this.TWEET_HEIGHT - 1;
		context.beginPath();
		context.moveTo( 4, y );
		context.lineTo( canvas.width - 4, y );
		context.stroke();

		context.restore();
	};
})( TweetTask.prototype );

/**
 * SentenceSelector
 */
var SentenceSelector = function( sentences )
{
	this.sentences	= sentences;
	this.array		= [];
};

(function( pt )
{
	/**
	 * 
	 */
	pt.getSentence = function()
	{
		// copy string array
		if ( this.array.length <= 0 )
		{
			for ( var i = 0; i < this.sentences.length; i++ )
			{
				this.array.push( this.sentences[i] );
			}
		}
		// update current block
		var index		= Math.floor( Math.random() * this.array.length );
		var sentence	= this.array[ index ];
		this.array.splice( index, 1 );

		return sentence;
	};
})( SentenceSelector.prototype );

/**
 * DebugTask
 */
var DebugTask = function( scene )
{
	this.command	= scene.command;	// input
	this.info		= null;
	this.lag		= 0;
};
DebugTask.prototype = new Task();

(function( pt )
{
	pt.LAG_VAL	= 10;

	/**
	 * 
	 */
	pt.update = function( scene )
	{
		var upd = false;
		// lag
		if ( this.lag )
		{
			this.lag--;
		}
		else
		{
			if ( this.command.tbl.debug )
			{
				this.info	= "scene.tweetListTask.status=" + scene.tweetListTask.status;
				this.lag	= this.LAG_VAL;
				upd = true;
			}
		}
		return upd;
	};

	/**
	 * 
	 */
	pt.draw = function( scene )
	{
		// output debug info
		if ( this.info )
		{
			Debug.print( this.info );
			this.info = null;
		}
	};
})( DebugTask.prototype );

/**
 * InputCommand
 */
var InputCommand = function()
{
	this.tbl.escape	= 0;
	this.tbl.debug	= 0;
};
InputCommand.prototype = new Command();

(function( pt )
{
	/**
	 * 
	 */
	pt.update = function() {};
	/**
	 * 
	 */
	pt.handleSysEvent = function( event )
	{
		var type = event.type.toLowerCase();
		if ( type.substring( 0, 3 ) == "key" )
		{
			var value = 0;
			if      ( type == "keydown" ) { value = 1; }
			else if ( type == "keyup"   ) { value = 0; }

			switch ( event.keyCode )
			{
				case 27: this.tbl.escape= value; break;	// ESC
				case 81: this.tbl.debug	= value; break;	// Q
			}
		}
	};
})( InputCommand.prototype );

/**
 * ShuzotterScene
 */
var ShuzotterScene = function( app, name )
{
	this.app			= app;
	this.name			= name;
	this.data			= null;
	this.command		= new InputCommand();

	// create task
	this.stageTask		= new StageTask( this );
	this.tweetListTask	= new TweetListTask( this );
	this.soundTask		= new SoundTask( this );
	this.debugTask		= new DebugTask( this );
	// create list
	this.stageTask.append( this.tweetListTask );
	this.stageTask.append( this.soundTask );
	this.stageTask.append( this.debugTask );
	// head of task list
	this.child		= this.stageTask;
	this.setStatus( SceneStatus.READY );

	// message handler
	//this.msgManager		= new MessageManager();
};
ShuzotterScene.prototype = new Scene();

(function( pt )
{
	/**
	 * 
	 */
	pt.init = function()
	{
		// window.onload のタイミング
	};
	/**
	 * 
	 */
	pt.show = function()
	{
		this.setUsage( Res.String.HTML_USAGE );

		this.command.clear();
		this.holdContext();
		this.draw( this );
	};
	/**
	 * 
	 */
	pt.setData = function( data )
	{
		this.data		= data;
		this.tweetListTask.setData( this, data.sentences );
	};
	/**
	 * 
	 */
	pt.getData = function()
	{
		return this.data;
	};
	/**
	 * 
	 */
	pt.loadData = function( data )
	{
		try
		{
			// load resource
			this.app.loadResource( this.name, data.resource );
			// set data
			this.setData( data );
			// set status
			var self = this;
			window.setTimeout( function() { self.setStatus( SceneStatus.RUNNING ); self.show(); }, Config.loadInterval );
		}
		catch ( e )
		{
			this.app.kill();
			Debug.alertError( e );
		}
	};
})( ShuzotterScene.prototype );


// Expose
if ( !window.h5glib ) { window.h5glib = {}; }
window.h5glib.ShuzotterScene	= ShuzotterScene;

// Anonymous function end
//
})( window );
