import java.util.*;
import java.io.*;

public class StatusBrowserModel {
	public static final int SORT_INDEX_OFFENCE   = 0;
	public static final int SORT_INDEX_DEFENSE   = 1;
	public static final int SORT_INDEX_HITPOINT  = 2;
	public static final int SORT_INDEX_SP_ATTACK = 3;
	public static final int SORT_INDEX_NONE      = 4;
	
	private List<Rider> riders;
	
	StatusBrowserModel() {setUp();}
	
	public void setUp() {
		riders = new ArrayList<Rider>();
		try {
			File csv = new File("data.csv");
			BufferedReader reader = new BufferedReader(new FileReader(csv));
			String line;
			while((line = reader.readLine()) != null) {
				if((line.length() >= 2) && (line.charAt(0) == '/' && line.charAt(1) == '/')) continue; // comment line
		        StringTokenizer st = new StringTokenizer(line, ",");
		        riders.add(new Rider(getStringValue(st), 
		        					 getIntValue(st), 
		        					 getStringValue(st),
		        					 getInt16Value(st),
		        					 getIntValue(st),
		        					 getIntValue(st),
		        					 getIntValue(st),
		        					 getIntValue(st),
		        					 getIntValue(st),
		        					 new int[]{getIntValue(st), getIntValue(st), getIntValue(st), getIntValue(st)},
		        					 new RiderSkill(getIntValue(st), getIntValue(st), getIntValue(st), getIntValue(st), getIntValue(st), getInt16Value(st)),
		        					 getIntValue(st)));
			}
			reader.close();
		} catch(FileNotFoundException e) {
			e.printStackTrace();
		} catch(IOException e) {
			e.printStackTrace();
		}
	}

	public Rider search(String series, int number)
	{
		Rider rider = null;
		
		Iterator<Rider> i = riders.iterator();
		while(i.hasNext()) {
			Rider r = i.next();
			if(r.getSeries().equals(series) && (r.getNumber() == number)) {
				rider = r; break;
			}
		}
		
		return rider;
	}
	
	public DeckStatus getStatus(Rider vanguard, Rider rearguard, Rider breaker)
	{
		return new DeckStatus(vanguard, rearguard, breaker);
	}

	public List<DeckStatus> getStatusList(Rider rider, int riderPos, Filter filter, int sort)
	{
		List<DeckStatus> list = new ArrayList<DeckStatus>();
		Iterator<Rider> i = riders.iterator();
		while(i.hasNext()) {
			Rider partner = i.next();
			if(rider.getName().equals(partner.getName())) continue;
			
			Iterator<Rider> j = riders.iterator();
			while(j.hasNext()) {
				Rider partner2 = j.next();
				if(rider.getName().equals(partner2.getName()) || partner.getName().equals(partner2.getName())) continue;

				DeckStatus status;
				switch(riderPos){
				case Rider.POS_VANGUARD : status = getStatus(rider, partner, partner2); break;
				case Rider.POS_SUPPORTER: status = getStatus(partner, rider, partner2); break;
				default :/*POS_BREAKER */ status = getStatus(partner, partner2, rider); break;
				}
				if(!filter.isFiltered(status)) {
					list.add(status);
				}
				//if(filter.isFullEntry(list)) break;
			}
		}
		if(sort != SORT_INDEX_NONE) Collections.sort(list, new StatusComparator(sort));
		return list;
	}
	
	public ArrayList<String> getSeriesList()
	{
		ArrayList<String> seriesList = new ArrayList<String>();
		Iterator<Rider> i = riders.iterator();
		while(i.hasNext()) {
			Rider rider = i.next();
			if(seriesList.contains(rider.getSeries())) continue;
			seriesList.add(rider.getSeries());
		}		
		return seriesList;
	}
	
	public ArrayList<String> getNumberList(String series)
	{
		ArrayList<String> numList = new ArrayList<String>();
		Iterator<Rider> i = riders.iterator();
		while(i.hasNext()) {
			Rider rider = i.next();
			if(rider.getSeries().equals(series)) numList.add(String.valueOf(rider.getNumber()));
		}		
		return numList;		
	}

	// Utility method for data convert
	private String getStringValue(StringTokenizer st) {return st.nextToken().trim();}
	private int    getIntValue(StringTokenizer st)    {return Integer.parseInt(st.nextToken().trim());}
	private int    getInt16Value(StringTokenizer st)  {return Integer.parseInt(st.nextToken().trim(), 16);}
}

class DeckStatus extends Rider
{
	private static final int CHEMISTRY_UNIT = 50;
	private Rider vanguard;
	private Rider supporter;
	private Rider breaker;
	
	DeckStatus(Rider vanguard, Rider supporter, Rider breaker)
	{
		super("status", 0, 0, 0, 0, 0, 0, new int[CHEMISTRY_NUM], new RiderSkill(0, 0, 0, 0, 0, 0), 0);
		this.vanguard  = vanguard;
		this.supporter = supporter;
		this.breaker   = breaker;
		calcChemistry(vanguard, supporter, breaker);
		offense  = vanguard.getOffense() + chemistry[CHEMISTRY_INDEX_OFFENSE] * CHEMISTRY_UNIT;
		defense  = vanguard.getDefense() + chemistry[CHEMISTRY_INDEX_DEFENSE] * CHEMISTRY_UNIT;
		hitpoint = vanguard.getHitpoint() + supporter.getHitpoint() + breaker.getHitpoint() + chemistry[CHEMISTRY_INDEX_HITPOINT] * CHEMISTRY_UNIT;
		specialAttack    = vanguard.getSpecialAttack() + chemistry[CHEMISTRY_INDEX_SP_ATTACK] * CHEMISTRY_UNIT;
		rearGuardTactics = supporter.getRearGuardTactics();
		reflectSkills(vanguard, supporter, breaker);
	}
	
	private void calcChemistry(Rider vgd, Rider spt, Rider brk)
	{
	    int chemi1st[] = new int[CHEMISTRY_NUM];
	    int chemi2nd[] = new int[CHEMISTRY_NUM];
	    
	    calcChemistrySub(vgd, spt, chemi1st);
	    calcChemistrySub(vgd, brk, chemi2nd);
	    
	    for(int i = 0; i < CHEMISTRY_NUM; i++)
	    	this.chemistry[i] = chemi1st[i] + chemi2nd[i];
	}
	
	private void calcChemistrySub(Rider vanguard, Rider rearguard, int chemi[])
	{
		for(int i = 0; i < CHEMISTRY_NUM; i++)
			chemi[i] = vanguard.getChemistry(i) + rearguard.getChemistry(i);
		
		int first  = CHEMISTRY_INDEX_OFFENSE;
		int second = first + 1;
		
		for(int i = first + 1; i < CHEMISTRY_NUM; i++) {
			if(chemi[first] < chemi[i]) {
				second = first; first = i;
			} else if(chemi[second] < chemi[i]) {
				second = i;
			}
		}
		chemi[first]++;
		
		for(int i = 0; i < CHEMISTRY_NUM; i++) {
			if(i != first && i != second) chemi[i] = 0;
		}
	}
	
	private void reflectSkills(Rider vanguard, Rider supporter, Rider breaker)
	{
		if(vanguard.isVanguardSkillValid(supporter, breaker)) {
			offense  += vanguard.getOffenseSkill();
			defense  += vanguard.getDefenseSkill();
			hitpoint += vanguard. getHitpointSkill();
			specialAttack += vanguard.getSpecialAttackSkill();
		}
		
		if(supporter.isRearguardSkillValid(vanguard, breaker)) {
			offense  += supporter.getOffenseSkill();
			defense  += supporter.getDefenseSkill();
			hitpoint += supporter. getHitpointSkill();
			specialAttack += supporter.getSpecialAttackSkill();
		}
		
		if(breaker.isRearguardSkillValid(vanguard, supporter)) {
			offense  += breaker.getOffenseSkill();
			defense  += breaker.getDefenseSkill();
			hitpoint += breaker. getHitpointSkill();
			specialAttack += breaker.getSpecialAttackSkill();
		}
	}
	
	public String toString() {
		String str = new String("");
		str += "[GC(" + vanguard.getSeries() + "-" + vanguard.getNumber() + "),";
		str += "REGC(" + supporter.getSeries() + "-" + supporter.getNumber() + ")\t=>\t";
		str += "REQLF" + offense + ", ";
		str += "{EM F" + defense + ", ";
		str += "^CN F" + hitpoint + ", ";
		str += "qbTc F " + specialAttack;
		return str;
	}
	
	public Rider getVanguard()  {return vanguard; }
	public Rider getSupporter() {return supporter;}
	public Rider getBreaker()   {return breaker;  }
}

class StatusComparator implements Comparator<DeckStatus>
{
	private int compIndex;
	
	StatusComparator(int index) {
		compIndex = index;
	}
	public int compare(DeckStatus obj1, DeckStatus obj2) {
		int val1, val2;
		switch(compIndex) {
		case StatusBrowserModel.SORT_INDEX_OFFENCE:
			val1 = obj1.getOffense(); val2 = obj2.getOffense();
			break;
		case StatusBrowserModel.SORT_INDEX_DEFENSE:
			val1 = obj1.getDefense(); val2 = obj2.getDefense();
			break;
		case StatusBrowserModel.SORT_INDEX_HITPOINT:
			val1 = obj1.getHitpoint(); val2 = obj2.getHitpoint();
			break;
		case StatusBrowserModel.SORT_INDEX_SP_ATTACK:
			val1 = obj1.getSpecialAttack(); val2 = obj2.getSpecialAttack();
			break;
		default:
			return 0; 
		}
		if(val1 > val2)      return -1;
		else if(val2 > val1) return  1;
		else                 return  0;
	}
}

class Filter
{
	private int lim_o, lim_d, lim_h, lim_s, lim_e;
	Filter(int lim_o, int lim_d, int lim_h, int lim_s, int lim_e) {
		this.lim_o = lim_o; this.lim_d = lim_d; this.lim_h = lim_h; this.lim_s = lim_s; this.lim_e = lim_e;
	}
	
	public boolean isFiltered(DeckStatus status) {
		return (lim_o > status.getOffense())  || 
			   (lim_d > status.getDefense())  || 
			   (lim_h > status.getHitpoint()) || 
			   (lim_s > status.getSpecialAttack());
	}
	
	public boolean isFullEntry(List<DeckStatus> list) {
		return lim_e <= list.size();
	}
}