/*
 * Copyright (C) 2008-2009 GLAD!! (ITO Yoshiichi)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.glad.calendar.joda.chrono;

import java.util.Locale;

import jp.sourceforge.glad.calendar.era.JapaneseEra;

import org.joda.time.DateTimeField;
import org.joda.time.DateTimeFieldType;
import org.joda.time.DurationField;
import org.joda.time.field.BaseDateTimeField;

/**
 * 
 * @author GLAD!!
 */
class JapaneseYearDateTimeField extends BaseDateTimeField {

    static final int FIRST_YEAR_OF_ERA = 1;

    static final int MAX_SHORT_TEXT_LENGTH = 2;

    final JapaneseChronology iChronology;

    /**
     * Restricted constructor.
     */
    JapaneseYearDateTimeField(JapaneseChronology chronology) {
        super(DateTimeFieldType.yearOfEra());
        iChronology = chronology;
    }

    DateTimeField getYearField() {
        return iChronology.year();
    }

    public boolean isLenient() {
        return true;
    }

    // Main access API
    //------------------------------------------------------------------------

    public int get(long instant) {
        return iChronology.getYearOfEra(instant);
    }

    @Override
    public String getAsText(int yearOfEra, Locale locale) {
        if (yearOfEra == FIRST_YEAR_OF_ERA) {
            return getFirstYearText(locale);
        }
        return super.getAsText(yearOfEra, locale);
    }

    @Override
    public String getAsShortText(int yearOfEra, Locale locale) {
        String text = String.valueOf(yearOfEra);
        int length = text.length();
        if (length > MAX_SHORT_TEXT_LENGTH) {
            text = text.substring(length - MAX_SHORT_TEXT_LENGTH);
        }
        return text;
    }

    protected String getFirstYearText(Locale locale) {
        return JapaneseEra.getFirstYearText(locale);
    }

    public long set(long instant, int value) {
        return iChronology.setYearOfEra(instant, value);
    }

    @Override
    protected int convertText(String text, Locale locale) {
        if (getFirstYearText(locale).equals(text)) {
            return FIRST_YEAR_OF_ERA;
        }
        return super.convertText(text, locale);
    }

    // Extra information API
    //------------------------------------------------------------------------

    public DurationField getDurationField() {
        return getYearField().getDurationField();
    }

    public DurationField getRangeDurationField() {
        return getYearField().getRangeDurationField();
    }

    public int getMinimumValue() {
        return FIRST_YEAR_OF_ERA;
    }

    public int getMaximumValue() {
        return getYearField().getMaximumValue();
    }

    @Override
    public int getMaximumTextLength(Locale locale) {
        return getYearField().getMaximumTextLength(locale);
    }

    @Override
    public int getMaximumShortTextLength(Locale locale) {
        return MAX_SHORT_TEXT_LENGTH;
    }

    // Calculation API
    //------------------------------------------------------------------------

    public long roundFloor(long instant) {
        return getYearField().roundFloor(instant);
    }

    @Override
    public long roundCeiling(long instant) {
        return getYearField().roundCeiling(instant);
    }

}
