/*
 * Copyright (C) 2008-2009 GLAD!! (ITO Yoshiichi)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.glad.calendar;

import java.util.Calendar;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

/**
 * 時点を表すインタフェースです。
 * 
 * @author GLAD!!
 */
public interface Instant extends Comparable<Instant> {

    /**
     * 基準時点からの通算ミリ秒を返します。
     * 
     * @param 通算ミリ秒
     */
    long getTimeInMillis();

    /**
     * タイムゾーンを返します。
     * 
     * @return タイムゾーン
     */
    TimeZone getTimeZone();

    /**
     * Calendar に変換します。
     * 
     * @return Calendar
     */
    Calendar toCalendar();

    /**
     * Calendar に変換します。
     * 
     * @param locale ロケール
     * @return Calendar
     */
    Calendar toCalendar(Locale locale);

    /**
     * GregorianCalendar に変換します。
     * 
     * @return GregorianCalendar
     */
    GregorianCalendar toGregorianCalendar();

    /**
     * java.util.Date に変換します。
     * 
     * @return java.util.Date
     */
    java.util.Date toDate();

    /**
     * オブジェクトの文字列表現を返します。
     * 
     * @return 文字列
     */
    String toString();

    /**
     * ハッシュコードを返します。
     * 
     * @return ハッシュコード
     */
    int hashCode();

    /**
     * 他のオブジェクトと等しいか判定します。
     * 
     * @param other 他のオブジェクト
     * @return 等しければ true
     */
    boolean equals(Object other);

    /**
     * 他の時点と比較します。
     * 
     * @param other 他の時点
     * @return this < other ならば負の整数、
     *         this = other ならば0、
     *         this > other ならば正の整数
     */
    int compareTo(Instant other);

    /**
     * 他の時点より前か判定します。
     * 
     * @param other 他の時点
     * @return 前ならば true
     */
    boolean isBefore(Instant other);

    /**
     * 他の時点と同時か判定します。
     * 
     * @param other 他の時点
     * @return 同時ならば true
     */
    boolean isEqual(Instant other);

    /**
     * 他の時点より後か判定します。
     * 
     * @param other 他の時点
     * @return 後ならば true
     */
    boolean isAfter(Instant other);

}
