/*
 * Copyright (C) 2008-2009 GLAD!! (ITO Yoshiichi)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sourceforge.glad.calendar.joda.chrono;

import java.lang.ref.Reference;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.WeakHashMap;

import jp.sourceforge.glad.calendar.era.JapaneseEra;
import jp.sourceforge.glad.calendar.era.JapaneseEras;

import org.joda.time.DateTimeFieldType;
import org.joda.time.IllegalFieldValueException;

/**
 * 
 * 
 * @author GLAD!!
 */
class JapaneseLocaleSymbols {

    static final JapaneseEras cEras = JapaneseEras.getInstance();

    static final Map<Locale, JapaneseLocaleSymbols> cCache =
            new WeakHashMap<Locale, JapaneseLocaleSymbols>();

    public static JapaneseLocaleSymbols forLocale(Locale locale) {
        synchronized (cCache) {
            JapaneseLocaleSymbols symbols = cCache.get(locale);
            if (symbols == null) {
                symbols = new JapaneseLocaleSymbols(locale);
                cCache.put(locale, symbols);
            }
            return symbols;
        }
    }

    final Reference<Locale> iLocale;

    final int iMaxEraLength;
    final int iMaxShortEraLength;

    final Map<String, Integer> iParseEra = new HashMap<String, Integer>();

    public JapaneseLocaleSymbols(Locale locale) {
        iLocale = new WeakReference<Locale>(locale);
        int maxShortEraLength = 0;
        int maxLongEraLength = 0;
        for (JapaneseEra era : cEras.getJapaneseEras()) {
            Integer value = era.getId();
            iParseEra.put(String.valueOf(value), value);
            String shortEraText = era.getShortName(locale);
            iParseEra.put(shortEraText, value);
            if (maxShortEraLength < shortEraText.length()) {
                maxShortEraLength = shortEraText.length();
            }
            String mediumEraText = era.getMediumName(locale);
            iParseEra.put(mediumEraText, value);
            String longEraText = era.getLongName(locale);
            iParseEra.put(longEraText, value);
            if (maxLongEraLength < longEraText.length()) {
                maxLongEraLength = longEraText.length();
            }
        }
        iMaxEraLength = maxLongEraLength;
        iMaxShortEraLength = maxShortEraLength;
    }

    public String eraToText(int era) {
        return cEras.getJapaneseEra(era).getLongName(iLocale.get());
    }

    public String eraToShortText(int era) {
        return cEras.getJapaneseEra(era).getShortName(iLocale.get());
    }

    public int textToEra(String text) {
        Integer era = iParseEra.get(text);
        if (era != null) {
            return era;
        }
        throw new IllegalFieldValueException(DateTimeFieldType.era(), text);
    }

    public int getMaxEraLength() {
        return iMaxEraLength;
    }

    public int getMaxShortEraLength() {
        return iMaxShortEraLength;
    }

}
