#include "../gintenlib/plane/angle.hpp"

// boost の単体テストフレームワーク
#include <boost/test/minimal.hpp>

namespace plane = gintenlib::plane;

template<typename T>
T absolute( T x )
{
  return x >= 0 ? x : -x;
}

#include <boost/math/special_functions/hypot.hpp>
template<typename Real>
bool nearly_equal( Real a, Real b, Real eps = 1e-5 )
{
  using boost::math::hypot;
  using std::fabs;
  
  return a == b ? true : absolute( a - b ) / hypot( a, b ) <= eps;
}
template<typename Real>
bool nearly_equal( const plane::basic_angle<Real>& a, const plane::basic_angle<Real>& b, Real eps = 1e-5 )
{
  return nearly_equal( a.theta, b.theta, eps );
}

typedef plane::angle angle_t;

int test_main( int, char** )
{
  // neary_equal のチェック
  BOOST_CHECK( !nearly_equal( 0.0, 1.0 ) );
  
  // 構築（引数無しなら 0 、引数はラジアン単位）
  angle_t a, b( M_PI );
  
  BOOST_CHECK( a.theta == 0 );
  BOOST_CHECK( b.theta == M_PI );
  
  // to_deg のテスト
  BOOST_CHECK( nearly_equal( a.to_deg(), 0.0 ) );
  BOOST_CHECK( nearly_equal( to_deg(b), 180.0 ) );  // こう書くことも出来る
  // to_rad のテスト
  BOOST_CHECK( a.theta == a.to_rad() );
  BOOST_CHECK( b.theta == to_rad(b) );
  
  // 勝手な値に変更、ついでに radian 関数と degree 関数もチェック
  a = plane::radian( 1.0 ); // 1rad ≒ 57.3°
  b = plane::degree( 60.0 ); // つまり a < b
  
  // == のテスト（ついでに rad 関数や deg 関数もテスト）
  BOOST_CHECK( a == a );
  BOOST_CHECK( b == b );
  BOOST_CHECK( a == plane::rad( a.to_rad() ) ); // これはあたりまえ
  BOOST_CHECK( plane::deg( 60.0 ) == b ); // 上見れば当たり前
  // != のテスト
  BOOST_CHECK( a != b );
  // そのほか
  BOOST_CHECK( a <  b );
  BOOST_CHECK( a <= b );
  BOOST_CHECK( b >  a );
  BOOST_CHECK( b >= a );
  
  // 加減乗除（当たり前の事実）
  // なんか + 演算子と += 演算子の挙動が違うらしいので nearly 比較する
  BOOST_CHECK( nearly_equal( (a+b).theta, a.theta + b.theta ) );
  BOOST_CHECK( nearly_equal( (a-b).theta, a.theta - b.theta ) );
  BOOST_CHECK( nearly_equal( (-a).theta , -(a.theta)        ) );
  BOOST_CHECK( nearly_equal( (a*2).theta, a.theta * 2       ) );
  BOOST_CHECK( nearly_equal( (a/2).theta, a.theta / 2       ) );
  BOOST_CHECK( nearly_equal(  a/b       , a.theta / b.theta ) ); // 二つの角の比は実数
  BOOST_CHECK( nearly_equal( (a%b).theta, std::fmod( a.theta, b.theta ) ) );  // 剰余算は fmod
  
  // normalize
  b = a + plane::rad( 2*M_PI );
  a.normalize();  // a を正規化（この場合は特に意味無いけど）
  b.normalize();  // b も正規化。a と b は 2π だけ違うので同じ値になるはず
  BOOST_CHECK( -M_PI <= b.theta && b.theta <= M_PI ); // normalize した値はこの条件を満たす
  BOOST_CHECK( nearly_equal( a.theta, b.theta ) );
  // こんどは -2π だけずらして再度ちぇっく
  b = a - plane::rad( 2*M_PI );
  // normalize した値のみ必要なら、こうも書ける（変数には影響しない）
  BOOST_CHECK( nearly_equal( normalized(a).theta, normalized(b).theta ) );
  
  // この辺は unique でも全く同じ（結果の値の範囲だけが違う）
  b = a + plane::rad( 2*M_PI );
  a.unique();
  b.unique();
  BOOST_CHECK( 0 <= b.theta && b.theta < M_PI * 2 ); // 違うのはこの条件だけ
  BOOST_CHECK( nearly_equal( a.theta, b.theta ) );
  b = a - plane::rad( 2*M_PI );
  BOOST_CHECK( nearly_equal( uniqued(a).theta, uniqued(b).theta ) );
  
  // 境界での振舞い
  // normalize の場合、値が ±M_PI の時に、同じ角度にも関わらず等しくない場合がある
  a = plane::rad( -M_PI );
  b = plane::rad( +M_PI );
  BOOST_CHECK( !nearly_equal( normalized(a).theta, normalized(b).theta ) );
  // unique なら、その点は安心（ただし 2π 付近で、ごく僅かな差が大きな違いになりえることはある）
  BOOST_CHECK(  nearly_equal( uniqued(a).theta,    uniqued(b).theta    ) );
  
  // サインコサインタンジェント
  using namespace std;
  a = plane::deg(45.0);
  b = plane::deg(60.0);
  BOOST_CHECK( nearly_equal( a.sin(), sin( a.theta ) ) );
  BOOST_CHECK( nearly_equal( a.cos(), cos( a.theta ) ) );
  BOOST_CHECK( nearly_equal( a.tan(), tan( a.theta ) ) );
  
  // こう書いてもいい（そしてそっちのほうが自然）
  BOOST_CHECK( nearly_equal( sin(b), sin( b.theta ) ) );
  BOOST_CHECK( nearly_equal( cos(b), cos( b.theta ) ) );
  BOOST_CHECK( nearly_equal( tan(b), tan( b.theta ) ) );
  
  // 逆三角関数
  double x = 0.5, y = 1.0;
  BOOST_CHECK( nearly_equal( plane::rad( asin(x) ), angle_t::asin(x) ) );
  BOOST_CHECK( nearly_equal( plane::rad( acos(x) ), angle_t::acos(x) ) );
  BOOST_CHECK( nearly_equal( plane::rad( atan(x) ), angle_t::atan(x) ) );
  BOOST_CHECK( nearly_equal( plane::rad( atan2(y,x) ), angle_t::atan2(y,x) ) );
  
  return 0;
}
