#ifndef GINTENLIB_INCLUDED_TYPED_SAVER_HPP_
#define GINTENLIB_INCLUDED_TYPED_SAVER_HPP_

/*

      <gintenlib/typed_saver.hpp>

  typed_saver ： デストラクタでの値の復帰（テンプレート版）

  宣言：
    template< typename T >
    class typed_saver : boost::noncopyable
    {
     public:
      // t の値を記録して
      explicit typed_saver( T& t );
      // デストラクタで元に戻す
      ~typed_saver();
      
      // 明示的に値を戻したい場合に呼ぶ
      // これを呼んでも、デストラクタでは再度値が巻き戻される
      void invoke();
      // これを呼ぶと、デストラクタで値が戻されることはなくなる
      void release();
      // invoke して release する
      void reset();
      
    };
    
  機能：
    コンストラクタで受け取った変数の値を記録し、デストラクタで書き戻します。
    value_saver と違い変数の型を指定する必要がありますが、その分高速に動作します。
    
    詳しくは value_saver の項を確認して下さい。
  
  補足事項：
    typed_saver, value_saver を利用できるオブジェクトの条件は、
    ・CopyConstructible
    ・Assignable
    の２点です（version1.0.0で変更）。
    また、例外を投げない swap() 関数呼び出しが存在すれば、例外安全です。

*/

#include <algorithm>

#include <boost/noncopyable.hpp>
#include <boost/utility/addressof.hpp>

namespace gintenlib
{
  // コンストラクタの時の値に、デストラクタで強制復帰させるオブジェクト
  template< typename T >
  struct typed_saver
    : private boost::noncopyable
  {
    // t の値を記録し、デストラクタで巻き戻す
    explicit typed_saver( T& t )
      : target_( boost::addressof(t) ), saver_(t) {}
    
    // 復帰
    ~typed_saver() { reset(); }
    
    // 明示的な復帰
    // この呼出が行われたとしても、デストラクタでは再度 値の復帰が行われる
    void invoke()
    {
      if( target_ )
      {
        *target_ = saver_;
      }
    }
    
    // 復帰処理の抑制
    void release()
    {
      target_ = 0;
    }
    // 明示的に復帰してから release する
    void reset()
    {
      if( target_ )
      {
        using std::swap;
        swap( *target_, saver_ );
        target_ = 0;
      }
    }

   private:
    T* target_;
    T saver_;

  };  // class typed_saver<T>

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_TYPED_SAVER_HPP_
