#ifndef GINTENLIB_INCLUDED_POINTER_FACADE_HPP_
#define GINTENLIB_INCLUDED_POINTER_FACADE_HPP_

/*

      <gintenlib/pointer_facade.hpp>

  pointer_facade ： 簡易スマートポインタ製作

  宣言：
    template< class Derived, typename T, typename Category = Derived >
    struct pointer_facade
    {
      typedef T  element_type;
      typedef T  value_type;
      typedef T* pointer;
      typedef T& reference;
      
      // 参照外しとポインタ取得
      reference operator*() const;
      pointer operator->() const;
      friend pointer get_pointer( const Derived& target );
      
      // bool 比較
      operator unspecified_bool_type() const;
      bool operator!() const;
    
    };
    
    // テンプレート版相互比較
    // カテゴリが同じものなら相互比較ＯＫ
    template<typename T, typename U, typename D1, typename D2, typename C>
    inline bool operator==
      ( const pointer_facade<D1, T, C>& lhs, const pointer_facade<D2, U, C>& rhs );
      
    template<typename T, typename U, typename D1, typename D2, typename C>
    inline bool operator!=
      ( const pointer_facade<D1, T, C>& lhs, const pointer_facade<D2, U, C>& rhs );
    
    template<typename T, typename U, typename D1, typename D2, typename C>
    inline bool operator<
      ( const pointer_facade<D1, T, C>& lhs, const pointer_facade<D2, U, C>& rhs );
  
  機能：
    get() さえ実装すれば、それを元にスマートポインタとして必要なものを生成するクラスです。
    実装してくれるものは以下の通り：
    　・element_type, value_type, pointer, reference の型定義
    　・operator*, operator->, get_pointer() によるポインタ取得
    　・bool値との比較、スマートポインタ同士の比較
    　・異なる型を示すスマートポインタ同士の比較（同じカテゴリに属するもののみ）
    
    使い方は bool_comparable と同じ、「テンプレートに自分自身の型名を入れて継承」。
    第二引数はポインタを格納するオブジェクトの型を示します。
    省略可能な第三引数はポインタのカテゴリ、第三引数が同じスマートポインタ同士は相互比較できます。
    省略した場合、他のスマートポインタとは比較が出来ません（でも通常は比較できた方が良いですよ？）。
    さらに詳しくは、下の欄や、銀天ライブラリで定義されたスマートポインタの実装をご覧ください。
    
  使い方：
    // 単純なポインタへのラッパ
    
    // スマートポインタを作る場合、まずカテゴリクラスを定義する。
    // 同じ種類のスマートポインタで、同じカテゴリを共有することで、
    // スマートポインタ同士の相互比較が（同じ種類のスマートポインタに限り）可能になる。
    struct trivial_ptr_category {}; // カテゴリクラスはタグなので適当に
    
    // 本体 
    template<typename T>
    class trivial_ptr :
      public gintenlib::pointer_facade< trivial_ptr<T>, T, trivial_ptr_category > // こう使う
    {
     public:
      trivial_ptr() : p(0) {}
      explicit trivial_ptr( T* ptr ) : p( ptr ) {}
      
      // 解放処理とか行わない
      // ~trivial_ptr() throw () {}
      
      // get() さえあればOK
      T* get() const { return p; }
      
      // reset とか swap もあると良いけど、基本はこれで終了！
      
     private:
      T* p;
      
    };
    
    int main()
    {
      trivial_ptr<int> p1( new int() );
      
      // 色々と自動定義してくれるので
      // こんな感じに書ける
      if( p1 )  // OK
      {
        cout << *p1 << endl; // OK
      }
      
      trivial_ptr<vector<int> > p2( new vector<int>() );
      cout << p2->size() << endl; // OK
      
      trivial_ptr<void> p0; // void へのポインタも適切に振り分けてくれる
      assert( p0 != p1 ); // もちろん比較できる
      
      // todo: delete
    }
  
  注意事項：
    ポインタに必要な機能はあらかた用意してくれる便利クラスですが、
    static_pointer_cast やら swap やら reset やらは定義してくれません。忘れないように。
    なお、 friend 関数の get_pointer や operator== なんかは、gintenlib 名前空間内に定義されます。
    通常は ADL によって意識する必要はありませんが、古いコンパイラでは引っかかるかも。

*/

#include "bool_comparable.hpp"

#include <cassert>
#include <functional> // for std::less
#include <boost/type_traits/add_reference.hpp>

namespace gintenlib
{
  namespace pointer_facade_ // ADL 回避
  {
    // スマートポインタ生成ファクトリ
    template< typename Derived, typename T, typename Category = Derived >
    struct pointer_facade
      : bool_comparable< pointer_facade< Derived, T, Category > >
    {
      // スマートポインタとして必要な typedef
      typedef T  element_type;
      typedef T  value_type;
      typedef T* pointer;
      // reference は void 型への対応のため特殊
      typedef typename boost::add_reference<T>::type reference;
      
      // 参照外しとポインタ取得
      reference operator*() const
      {
        using namespace std;
        assert( get_() != 0 );
        return *get_();
      }
      pointer operator->() const
      {
        using namespace std;
        assert( get_() != 0 );
        return get_();
      }
      // for boost::mem_fn
      friend pointer get_pointer( const Derived& target )
      {
        return target.get();
      }
      
      // bool との比較
      bool operator!() const
      {
        return !get_();
      }
      // operator safe_bool は bool_comparable の管轄
      
      // 相互比較は外部で定義する
      
     private:
      // 派生クラスの get() を呼び出す
      pointer get_() const
      {
        return static_cast<const Derived*>(this)->get();
      }
      
     protected:
       pointer_facade(){}
      ~pointer_facade(){}
    
    };  // pointer_facade<Derived, T, Category>
    
    // 相互比較
    // カテゴリが同じものなら相互比較ＯＫ
    template<typename T, typename U, typename D1, typename D2, typename C>
    inline bool operator==( const pointer_facade<D1, T, C>& lhs, const pointer_facade<D2, U, C>& rhs )
    {
      return static_cast<const D1&>(lhs).get() == static_cast<const D2&>(rhs).get();
    }
    template<typename T, typename U, typename D1, typename D2, typename C>
    inline bool operator!=( const pointer_facade<D1, T, C>& lhs, const pointer_facade<D2, U, C>& rhs )
    {
      return static_cast<const D1&>(lhs).get() != static_cast<const D2&>(rhs).get();
    }
    
    // < は、ポインタ型が同じ場合とそうでない場合で分ける
    template<typename T, typename D, typename C>
    inline bool operator< ( const pointer_facade<D, T, C>& lhs, const pointer_facade<D, T, C>& rhs )
    {
      // 同じポインタの比較は std::less で比較する
      return std::less<T*>()( static_cast<const D&>(lhs).get(), static_cast<const D&>(rhs).get() );
    }
    template<typename T, typename U, typename D1, typename D2, typename C>
    inline bool operator< ( const pointer_facade<D1, T, C>& lhs, const pointer_facade<D2, U, C>& rhs )
    {
      // 異なるポインタ同士の比較は、仕方ないので < で比較する
      return static_cast<const D1&>(lhs).get() < static_cast<const D2&>(rhs).get();
    }
  
  } // namespace pointer_facade_
  
  // 名前空間に引き入れる
  using namespace pointer_facade_;

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_POINTER_FACADE_HPP_
