#ifndef GINTENLIB_PLANE_INCLUDED_RECT_HPP_
#define GINTENLIB_PLANE_INCLUDED_RECT_HPP_

/*

      <gintenlib/plane/rect.hpp>

  rect ： 二次元空間上の矩形

  宣言：
    // rect : 矩形
    template<typename Real>
    struct basic_rect
    {
      typedef Real real_type;
      typedef basic_point<Real> pos_type;
      typedef basic_vector<Real> vec_type;
      
      real_type x1, y1, x2, y2;
      
      basic_rect();
      basic_rect( real_type x1_, real_type y1_, real_type x2_, real_type y2_ );
        
      template<typename T>
      basic_rect( const basic_rect<T>& src );
      
      // operator[]
      // 頂点の座標を返します。
      pos_type operator[]( std::size_t index ) const;
      
      pos_type at( std::size_t index ) const;
      static std::size_t size()
      {
        return 4;
      }
      
      // 位置のシフト
      // この矩形自身をシフトさせます。
      basic_rect& shift( const vec_type& d );
      // friend 版はコピーしてからシフトします。
      friend basic_rect shift( const basic_rect& target, const vec_type& d );
      
      // 回転
      // この矩形自身を回転させたいが、そうすると矩形でなくなってしまうので、
      // やむなく四角形としてコピーしてから回転させます。
      // Angle 版は角度からサインコサインを求めて回転、
      // s, c 版はサインコサイン値から直接回転させます。
      template<typename Angle>
      basic_quadrangle<Real> rotate( const Angle& theta ) const;
      basic_quadrangle<Real> rotate( real_type s, real_type c ) const;
      template<typename Angle>
      friend basic_quadrangle<Real> rotate( const basic_rect& x, const Angle& theta );
      basic_quadrangle<Real> rotate( const basic_rect& x, real_type s, real_type c );
      
    };
    typedef basic_rect<double> rect;

  機能：
    二次元空間上の、位置の確定した矩形を表すクラス。
    回転させると四角形に変化する以外は普通かと。

*/

#include <cstddef>
#include <cassert>
#include <stdexcept>

#include "fwd.hpp"
#include "vector.hpp"
#include "point.hpp"
#include "quadrangle.hpp"

namespace gintenlib
{
 namespace plane
 {
  // rect : 矩形
  template<typename Real>
  struct basic_rect
  {
    typedef Real real_type;
    typedef basic_point<Real> pos_type;
    typedef basic_vector<Real> vec_type;
    
    real_type x1, y1, x2, y2;
    
    basic_rect()
      : x1(), y1(), x2(), y2() {}
    
    basic_rect( real_type x1_, real_type y1_, real_type x2_, real_type y2_ )
      : x1(x1_), y1(y1_), x2(x2_), y2(y2_) {}
      
    template<typename T>
    basic_rect( const basic_rect<T>& src )
      : x1(src.x1), y1(src.y1), x2(src.x2), y2(src.y2) {}
    
    // operator[]
    pos_type operator[]( std::size_t index ) const
    {
      using namespace std;
      
      switch( index )
      {
       case 0:
        return pos_type( x1, y1 );
       
       case 1:
        return pos_type( x2, y1 );
      
       case 2:
        return pos_type( x2, y2 );
      
       case 3:
        return pos_type( x1, y2 );
      
       default:
        assert( !"should not get here." );
      
      }
    }
    pos_type at( std::size_t index ) const
    {
      if( index > size() )
      {
        throw std::out_of_range();
      }
      
      return operator[](index);
    }
    static std::size_t size()
    {
      return 4;
    }
    
    // 位置のシフト
    basic_rect& shift( const vec_type& d )
    {
      x1 += d.x;
      y1 += d.y;
      x2 += d.x;
      y2 += d.y;
      
      return *this;
    }
    friend basic_rect shift( const basic_rect& target, const vec_type& d )
    {
      basic_rect temp = target;
      return temp.shift(d);
    }
    
    // 回転
    template<typename Angle>
    basic_quadrangle<Real> rotate( const Angle& theta ) const
    {
      using std::sin; using std::cos;
      return rotate( sin(theta), cos(theta) );
    }
    basic_quadrangle<Real> rotate( real_type s, real_type c ) const
    {
      pos_type center( (x1 + x2) / 2, (y1 + y2) / 2 );
      real_type w = (x2 - x1) / 2, h = (x2 - x1) / 2;
      real_type sw = s * w, sh = s * h, cw = c * w, ch = c * h;
      
      vec_type p1( -cw - sh, -ch + sw ), p2( cw - sh, -ch - sw );
      
      return basic_quadrangle<Real>( center, p1, p2, -p1, -p2 );
    }
    template<typename Angle>
    friend basic_quadrangle<Real> rotate( const basic_rect& x, const Angle& theta )
    {
      return x.rotate(theta);
    }
    basic_quadrangle<Real> rotate( const basic_rect& x, real_type s, real_type c )
    {
      return x.rotate( s, c );
    }
    
  };
  typedef basic_rect<double> rect;
  
 }  // namespace plane
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_PLANE_INCLUDED_RECT_HPP_
