#ifndef GINTENLIB_PLANE_INCLUDED_QUADRANGLE_HPP_
#define GINTENLIB_PLANE_INCLUDED_QUADRANGLE_HPP_

/*

      <gintenlib/plane/quadrangle.hpp>

  quadrangle ： 二次元空間上の四角形

  宣言：
    // quadrangle : 四角形
    template<typename Real>
    struct basic_quadrangle
    {
      typedef Real real_type;
      typedef basic_point<Real> pos_type;
      typedef basic_vector<Real> vec_type;
      
      // 中心座標
      pos_type pos;
      // 各頂点の中心座標からの位置ベクトル
      vec_type a1, a2, a3, a4;
      
      basic_quadrangle();
      
      // 原点を中心とした四角形を製作する
      basic_quadrangle( const vec_type& a1_, const vec_type& a2_,
         const vec_type& a3_, const vec_type& a4_ );
      
      // 位置ベクトル４つから四角形を製作する。
      // 中心は４点の平均、すなわち重心
      basic_quadrangle( const pos_type& a1_, const pos_type& a2_,
         const pos_type& a3_, const pos_type& a4_ );
      
      // 全要素を指定して構築
      basic_quadrangle( const pos_type& pos_, const vec_type& a1_, const vec_type& a2_,
         const vec_type& a3_, const vec_type& a4_ );
      
      // 全要素を位置ベクトルで指定して構築
      basic_quadrangle( const pos_type& pos_, const pos_type& a1_, const pos_type& a2_,
         const pos_type& a3_, const pos_type& a4_ );
        
      template<typename T>
      basic_quadrangle( const basic_quadrangle<T>& src );
      
      
      // 各頂点の位置ベクトルを算出
      pos_type vertex1() const { return pos + a1; }
      pos_type vertex2() const { return pos + a2; }
      pos_type vertex3() const { return pos + a3; }
      pos_type vertex4() const { return pos + a4; }
      
      // operator[]
      // インテックスに対応する頂点座標を算出
      pos_type operator[]( std::size_t index ) const;
      pos_type at( std::size_t index ) const;
      static std::size_t size()
      {
        return 4;
      }
      
      // 位置のシフト
      // メンバ版は自身を変更、 friend 版は結果を出力。
      basic_quadrangle& shift( const vec_type& d );
      friend basic_quadrangle shift( const basic_quadrangle& target, const vec_type& d );
      
      // 頂点の位置を変えず、中心を変更
      // friend 版はありません。
      basic_quadrangle& move_center( const pos_type& center );
      basic_quadrangle& move_center( const vec_type& d );
      // 引数なしだと重心に持っていく
      basic_quadrangle& move_center();
      
      // 回転
      // rect, vector とかとまったく同じ。
      // 中心座標をわざわざ個別に設定しているのは、このときの回転軸を決めるため。
      template<typename Angle>
      basic_quadrangle& rotate( const Angle& theta );
      basic_quadrangle& rotate( real_type s, real_type c );
      template<typename Angle>
      friend basic_quadrangle rotate( const basic_quadrangle& x, const Angle& theta );
      friend basic_quadrangle rotate( const basic_quadrangle& x, real_type s, real_type c );
      
    };
    typedef basic_quadrangle<double> quadrangle;

  機能：
    二次元空間上の、頂点の位置と中心の確定したを四角形を表すクラス。
    平行移動、回転などが行えます。つーか名前長いよ。

*/

#include "fwd.hpp"
#include "vector.hpp"
#include "point.hpp"

namespace gintenlib
{
 namespace plane
 {
  
  // quadrangle : 四角形
  template<typename Real>
  struct basic_quadrangle
  {
    typedef Real real_type;
    typedef basic_point<Real> pos_type;
    typedef basic_vector<Real> vec_type;
    
    // 中心座標
    pos_type pos;
    // 各頂点の中心座標からの位置ベクトル
    vec_type a1, a2, a3, a4;
    
    basic_quadrangle()
      : pos(), a1(), a2(), a3(), a4() {}
    
    // 原点を中心とした四角形を製作する
    basic_quadrangle( const vec_type& a1_, const vec_type& a2_,
       const vec_type& a3_, const vec_type& a4_ )
      : pos(), a1(a1_), a2(a2_), a3(a3_), a4(a4_) {}
    
    // 位置ベクトル４つから四角形を製作する。
    // 中心は４点の平均、すなわち重心
    basic_quadrangle( const pos_type& a1_, const pos_type& a2_,
       const pos_type& a3_, const pos_type& a4_ )
    {
      pos = midpoint( midpoint( a1_, a2_ ), midpoint( a3_, a4_ ) );
      
      a1 = a1_ - pos;
      a2 = a2_ - pos;
      a3 = a3_ - pos;
      a4 = a4_ - pos;
    }
    
    // 全要素を指定して構築
    basic_quadrangle( const pos_type& pos_, const vec_type& a1_, const vec_type& a2_,
       const vec_type& a3_, const vec_type& a4_ )
      : pos(pos_), a1(a1_), a2(a2_), a3(a3_), a4(a4_) {}
    
    // 全要素を位置ベクトルで指定して構築
    basic_quadrangle( const pos_type& pos_, const pos_type& a1_, const pos_type& a2_,
       const pos_type& a3_, const pos_type& a4_ )
      : pos(pos_), a1( a1_ - pos ), a2( a2_ - pos ), a3( a3_ - pos ), a4( a4_ - pos ) {}
      
    template<typename T>
    basic_quadrangle( const basic_quadrangle<T>& src )
      : pos(src.pos), a1(src.a1), a2(src.a2), a3(src.a3), a4(src.a4) {}
    
    
    // 各頂点の位置ベクトルを算出
    pos_type vertex1() const { return pos + a1; }
    pos_type vertex2() const { return pos + a2; }
    pos_type vertex3() const { return pos + a3; }
    pos_type vertex4() const { return pos + a4; }
    
    // operator[]
    pos_type operator[]( std::size_t index ) const
    {
      using namespace std;
      
      switch( index )
      {
       case 0:
        return vertex1();
       
       case 1:
        return vertex2();
      
       case 2:
        return vertex3();
      
       case 3:
        return vertex4();
      
       default:
        assert( !"should not get here." );
      
      }
    }
    pos_type at( std::size_t index ) const
    {
      if( index > size() )
      {
        throw std::out_of_range();
      }
      
      return operator[](index);
    }
    static std::size_t size()
    {
      return 4;
    }
    
    // 位置のシフト
    basic_quadrangle& shift( const vec_type& d )
    {
      pos += d;
      
      return *this;
    }
    friend basic_quadrangle shift( const basic_quadrangle& target, const vec_type& d )
    {
      basic_quadrangle temp = target;
      return temp.shift(d);
    }
    
    // 頂点の位置を変えず、中心を変更
    basic_quadrangle& move_center( const pos_type& center )
    {
      vec_type d = pos - center;
      
      pos = center;
      a1 += d;
      a2 += d;
      a3 += d;
      a4 += d;
      
      return *this;
    }
    basic_quadrangle& move_center( const vec_type& d )
    {
      pos += d;
      a1  -= d;
      a2  -= d;
      a3  -= d;
      a4  -= d;
      
      return *this;
    }
    // 引数なしだと重心に持っていく
    basic_quadrangle& move_center()
    {
      basic_quadrangle temp( vertex1(), vertex2(), vertex3(), vertex4() );
      *this = temp;
      return *this;
    }
    
    // 回転
    template<typename Angle>
    basic_quadrangle& rotate( const Angle& theta )
    {
      using std::sin; using std::cos;
      return rotate( sin(theta), cos(theta) );
    }
    basic_quadrangle& rotate( real_type s, real_type c )
    {
      a1.rotate( s, c );
      a2.rotate( s, c );
      a3.rotate( s, c );
      a4.rotate( s, c );
      
      return *this;
    }
    template<typename Angle>
    friend basic_quadrangle rotate( const basic_quadrangle& x, const Angle& theta )
    {
      basic_quadrangle temp = x;
      return temp.rotate(theta);
    }
    friend basic_quadrangle rotate( const basic_quadrangle& x, real_type s, real_type c )
    {
      basic_quadrangle temp = x;
      return temp.rotate( s, c );
    }
    
  };
  typedef basic_quadrangle<double> quadrangle;
  
 }  // namespace plane
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_PLANE_INCLUDED_QUADRANGLE_HPP_
