#ifndef GINTENLIB_PLANE_INCLUDED_POINT_HPP_
#define GINTENLIB_PLANE_INCLUDED_POINT_HPP_

/*

      <gintenlib/plane/point.hpp>

  point ： 二次元位置ベクトル

  機能：
    二次元の位置ベクトルを表すクラス。
    位置ベクトルなので加算できません。実数倍も出来ません。
    でも相対ベクトル（vector）を加算したりすることは出来ます。
    ちなみに差を取ることは可能で、そうすると相対ベクトルに変化します。
    位置ベクトルと相対ベクトルを混同するとバグの元になるので、それを防ぐためだけのクラス。

*/

#include "fwd.hpp"
#include "vector.hpp"

#include <boost/operators.hpp>

namespace gintenlib
{
 namespace plane
 {
  // point : 位置ベクトル
  template<typename Real>
  struct basic_point : private boost::additive< basic_point<Real>, basic_vector<Real> >
    , private boost::equality_comparable< basic_point<Real> >
  {
    typedef Real real_type;
    real_type x, y;
    
    basic_point()
      : x(), y() {}
    
    basic_point( real_type x_, real_type y_ )
      : x(x_), y(y_) {}
      
    template<typename T>
    basic_point( const basic_point<T>& src )
      : x(src.x), y(src.y) {}
    
    template<typename T>
    explicit basic_point( const basic_vector<T>& src )
      : x(src.first), y(src.second) {}
    
    // operator overloads
    basic_point<Real> operator+=( const basic_vector<Real>& rhs )
    {
      x += rhs.x; y += rhs.y;
      return *this;
    }
    basic_point<Real> operator-=( const basic_vector<Real>& rhs )
    {
      x -= rhs.x; y -= rhs.y;
      return *this;
    }
    // 後、位置ベクトル同士の差分
    basic_vector<Real> operator-( const basic_point& rhs ) const
    {
      return basic_vector<Real>( x - rhs.x, y - rhs.y );
    }
    // 等値比較
    bool operator==( const basic_point& rhs ) const
    {
      return ( x == rhs.x ) && ( y == rhs.y );
    }
    
    // vector に変換
    basic_vector<Real> to_vector() const
    {
      return basic_vector<Real>( x, y );
    }
    
  };
  typedef basic_point<double> point;
  
  // 中点を求める
  template<typename T>
  basic_point<T> midpoint( const basic_point<T>& p1, const basic_point<T>& p2 )
  {
    return basic_point<T>( (p1.x+p2.x)/2, (p1.y+p2.y)/2 );
  }
  
 }  // namespace plane
}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_PLANE_INCLUDED_POINT_HPP_
