#ifndef GINTENLIB_OPTIONS_INCLUDED_OPTIONS_HPP_
#define GINTENLIB_OPTIONS_INCLUDED_OPTIONS_HPP_

/*
      <gintenlib/options/options.hpp>

  options ： getopt 的プログラムオプション解析

  TODO: 例外安全性の強化

  宣言：
    struct options
    {
      // 指定された文字列でオプション解析オブジェクトを作る
      // 文字列は基本的に getopt 互換、冒頭の : と - の動作のみ違う
      explicit options( int argc, char* argv[], const std::string& optstr );
      
      // 実際にオプションを解析する
      typedef int result_type;
      result_type operator()();
      
      // 次に処理する引数のインデックス
      int optind() const;
      // 次に処理する引数のインデックスを設定する
      void set_optind( int new_index = 1 );
      
      // 引数
      const std::string& optarg() const;
      
      // エラー時に読んでいた文字
      int optopt() const;
      
    };  // struct options

  機能：
    コンストラクタで与えられたフォーマットを元に、プログラムオプションを解析します。
    フォーマットは基本的にＧＮＵの getopt 互換ですが、若干の違いがあり、
    まず未知のオプションが与えられた場合、'?'を返さず例外を投げる点。
    これは ':' をフォーマットの頭に置くことで、例外を投げず '?' を返すように変更できます。
    このとき、引数が足りない場合は ':' を返します（ここは getopt の動作と同じ）。
    またオプション文字列の頭に '-' を指定することは出来ません（ＧＮＵの仕様が良く分からなかったので）。
    また環境変数 POSIXLY_CORRECT が定義されていても動作はなにも変わりません。
    '+' で始まる文字列を渡すことで対処してください。
    
    具体的なフォーマットの与え方は以下の通り：
      単独の文字は、引数を取らないオプションを示す。
      文字の後に ':' が続いている場合、そのオプションは引数を取る。
      文字の後に二つの ':' が続いている場合、そのオプションは引数を取りうる。
    例：
      "ab:c::d" と渡されたら、a と d は引数なし、 b は引数あり、 c は引数はあってもなくてもいい

  使用例：
    // usage: コマンド名 オプション文字列 解析させたい引数
    // こう起動することにより、 options のほぼ全機能をテストできる例です
    #include <gintenlib/options.hpp>
    #include <gintenlib/cast.hpp>
    
    #include <iostream>
    using namespace std;
    
    int main( int argc, char* argv[] )
    {
      // 暗黙キャスト
      using gintenlib::cast;
      
      try
      {
        if( argc < 2 ){ throw gintenlib::option_error("missing 'OPTSTR'"); }
        
        // argv[1] は特別な引数
        gintenlib::options opt( argc, argv, argv[1] );
        // なので解析させない
        opt.set_optind( 2 );
        
        for(;;)
        {
          // 解析させる。 ch にはオプションの文字が入る
          int ch = opt();
          if( ch == -1 ){ break; }  // -1: 解析終わり
          
          switch(ch)
          {
           case '?':
            // ':' で始まるオプション文字列への対応
            cout << "unknown option '" << cast<char>( opt.optopt() ) << "' is given. continue...\n";
            break;
            
           case ':':
            // 同じく ':' で始まるオプション文字列への対応
            cout << "option '" << cast<char>( opt.optopt() ) << "'requires argument. continue...\n";
            break;
            
           default:
            // オプション一般
            cout << "option '" << cast<char>( ch ) << "' is given.\n";
            
            // 引数は optarg() で獲得
            // 引数が無い場合は string() が入る
            if( !opt.optarg().empty() )
            {
              cout << "  argument: " << opt.optarg() << endl;
            }
            break;
            
          }
        }
        
        // 解析終了。余ったオプションは argv[opt.optind()] から argv[argc-1] に集められる
        // デフォルトでは、オプション引数が非オプション引数の後にある場合、 argv の順序を入れ替える
        // この動作を避けたい場合は、コンストラクタの引数を '+' で始めればよい
        cout << "\nextra options are:\n";
        for( int i = opt.optind(); i < argc; ++i )
        {
          cout << argv[i] << endl;
        }
      }
      catch( gintenlib::option_error& e )
      {
        // オプション解析に失敗して例外が投げられた場合はこちらに飛ぶ
        cerr << e.what() << endl;
        cerr << "usage: " << argv[0] << " OPTSTR [OPTIONS]\n";
        
        return 1;
      }
      catch( std::exception& e )
      {
        // そのほかの例外
        cerr << e.what() << endl;
        return 1;
      }
    } // main( argc, argv )


  補足：
    ・主に <unistd.h> がインクルードできない処理系向けのライブラリです。
    ・あと、もろもろの事情で <unistd.h> をインクルードしたくない場合とかにも。
    ・グローバル変数を使わない、失敗時は例外を投げてくれる、というのもメリットです。
    ・上記のような理由が無いなら、素直に getopt を使ったほうがよいと思います。
    ・代替手段としては boost の progtam_options ってのもあるけど、リンクしなきゃいけないのが面倒。
    ・あと、いろいろオーバースペック過ぎる気がして気軽には使いにくいです。個人的に。
    ・まー大した処理をしないなら、これを使っとけ。ヘッダのインクルードのみで済むしね。
    ・テンプレート？ なにそれ美味しいの？

*/

#include "exceptions.hpp"

#include <map>
#include <string>
#include <cassert>
#include <algorithm>
#include "../tribool.hpp"
#include "../to_string.hpp"
#include "../cast.hpp"

namespace gintenlib
{
  // オプション解析機
  struct options
  {
    // optstr で示されるフォーマットのオプションを解析する
    explicit options( int argc_, char* argv_[], const std::string& optstr )
      : argc(argc_), argv(argv_), index(1), index_max(argc), next_char(0), opt('?'),
        posixly_correct(false), throws_on_error(true)
    {
      using namespace std;
      char prev = 0;

      for( string::const_iterator ite = optstr.begin(), end = optstr.end();
        ite != end; ++ite )
      {
        char ch = *ite;

        if( ch == ':' )
        {
          // コロン記号は直前に解析した文字によって動作を切り替える
          if( prev == 0 )
          {
            // 文頭のコロンは「例外を投げない」指定ということにする
            throws_on_error = false;
          }
          else
          {
            // 直前の文字を検索する
            map_type::iterator pos = m.find( prev );
            // 見つからないことは有り得ない
            assert( pos != m.end() );
            
            tribool& has_arg = (*pos).second;
            if( !has_arg )
            {
              // 一回目の : の時は、引数必須
              has_arg = true;
            }
            else
            {
              // 二回目以降は、引数任意
              has_arg = indeterminate;
            }
          }
        }
        else if( ch == '+' )
        {
          if( prev != 0 )
          {
            // + 指定は冒頭にのみ置ける
            throw invalid_option_string( optstr );
          }
          // この指定があった場合、引数の順番は入れ替えない
          posixly_correct = true;
        }
        else if( ch == '-' )
        {
          // - 記号はGNU拡張に存在するが、よく分からないので常に不正ということにする（要修正）
          throw invalid_option_string( optstr );
        }
        else if( ch == '?' )
        {
          // ? 記号はオプション文字としては使えないので不正
          throw invalid_option_string( optstr );
        }
        else
        {
          // 他一般
          // ':' '+' '-' '\0' 以外ならどんな文字でもいい
          if( m.count( ch ) != 0 )
          {
            // 同じオプション文字を二回定義しちゃダメ
            throw invalid_option_string( optstr );
          }
          // デフォルトでは引数なし
          m[ch] = false;
          // : 指定されたときのために、文字を保存する
          prev = ch;
        }
        
      }  // for
    }    // constructor

    // 実際の解析はこちらで
    typedef int result_type;
    result_type operator()()
    {
      using namespace std;  // for assert

      // まずインデックスのチェック
      if( index_max <= index ){ return -1; }
      // 引数文字列の初期化
      arg = string();

      // 次に処理する文字が指定されて無い場合（普通はそう）は
      // argv[index] に進む
      if( !next_char )
      {
        next_char = argv[index];
        assert( next_char != 0 );

        // オプション文字列だから、'-' ではじまるはず
        char ch = *next_char;
        if( ch == '-' )
        {
          // OK。その次に興味がある
          ++next_char;
        }
        else
        {
          // オプション文字列ではなかった

          // とりあえずポインタはリセット
          next_char = 0;

          if( posixly_correct )
          {
            // POSIX 的には、一回失敗するとそれで終わり
            return -1;
          }
          else
          {
            // GNU 拡張だと、引数の順序を入れ替えてマッチングを続ける
            for( int i = index + 1; i < index_max; ++i )
            {
              char* p = argv[i];
              if( *p == '-' )
              {
                rotate( &argv[index], &argv[i], &argv[argc] );
                index_max -= i - index;
                return (*this)();
              }
            }

            // 見つからなかった
            // 並び順を復元する
            rotate( &argv[index], &argv[index_max], &argv[argc] );
            index_max = index;

            return -1;
          }
        }
      }

      // この時点で、next_char はオプション文字を指している筈
      char ch = *next_char;

      if( ch == 0 )
      {
        // - のみの引数が渡された
        // これは非オプション文字列として扱う
        next_char = 0;
        return -1;
      }
      else if( ch == '-' )
      {
        // '--' は長いオプションか、オプションの終了を表す
        ++next_char;
        if( *next_char != 0 )
        {
          // 長いオプションには非対応
          throw unknown_option( "--" + to_str(next_char) );
        }
        
        // 長いオプションではないのでオプションの終了を表す
        // 終了処理をして
        terminate_analysis_();
        // おしまい
        return -1;
      }

      // ch をマップから検索する
      map_type::const_iterator ite = m.find( ch );
      if( ite == m.end() )
      {
        // 存在しないようなら、そのオプションを optopt に格納し
        opt = ch;
        // 一応、文字を次に進めて
        advance_next_char_();

        // 例外を投げる
        if( throws_on_error )
        {
          throw unknown_option( "-" + to_str( cast<char>(ch) ) );
        }

        // 投げない場合は '?' を返す
        return '?';
      }

      // 引数はある？
      tribool has_arg = (*ite).second;
      if( !has_arg )
      {
        // 引数のない場合
        // とりあえず次の文字に進める
        advance_next_char_();

        // その文字を返す
        return ch;
      }

      // この時点で、引数は「ある」か「任意」かのどちらか
      ++next_char;
      if( *next_char != 0 )
      {
        // 続きがあるなら、そいつが引数文字列
        arg = string( next_char );

        advance_index_();
        return ch;
      }

      // 引数必須か否か
      if( has_arg )
      {
        // 必須なら、次の argv を引数とみなす
        advance_index_();

        if( index_max <= index )
        {
          return missing_argument_found_( ch );
        }

        // 次のインデックスは、まるまる引数と見なす
        arg = argv[index];
        // ただし argv[index] が "--" のときは例外
        if( arg == "--" )
        {
          arg = string();
          terminate_analysis_();
          return missing_argument_found_( ch );
        }

        // さらに解析行を進める
        advance_index_();
        return ch;
      }
      else
      {
        // 引数が省略可能で、オプション文字と同時に引数が渡されていないなら
        // 引数を空文字列にして ch を返す
        advance_index_();
        return ch;
      }

      assert( !"should not get here." );
      return '?';
    }

    // 次に処理する引数のインデックス
    int optind() const { return index; }
    // 次に処理する引数のインデックスを設定する
    void set_optind( int new_index = 1 )
    {
      if( new_index < 1 ){ new_index = 1; }

      index = new_index; next_char = 0; index_max = argc;
      arg = std::string(); opt = '?';
    }

    // 引数
    const std::string& optarg() const { return arg; }

    // エラー時に読んでいた文字
    int optopt() const { return opt; }

   private:
    typedef std::map< char, tribool > map_type;
    map_type m;
    int argc;
    char** argv;
    int index, index_max;
    const char* next_char;
    std::string arg;
    int opt;
    bool posixly_correct, throws_on_error;

    // 内部関数
    // 解析する行を一つ進める
    void advance_index_()
    {
      ++index;
      next_char = 0;
    }
    // 文字を一文字進める
    void advance_next_char_()
    {
      // とりあえず次の文字に進める
      ++next_char;

      if( *next_char == 0 )
      {
        // 次の文字が存在しないなら、解析のインデックスを進める
        ++index;
        next_char = 0;
      }
    }
    // 引数が足りない
    int missing_argument_found_( int ch )
    {
      // 引数不足
      opt = ch;

      // 例外を投げる
      if( throws_on_error )
      {
        throw option_requires_argument( "-" + to_str( cast<char>(ch) ) );
      }

      return ':';
    }
    // '--' に遭遇した
    void terminate_analysis_()
    {
      // インデックスを進めて
      advance_index_();
      
      // 並び替えの途中なら並び順を復元する
      if( argc != index_max )
      {
        std::rotate( &argv[index], &argv[index_max], &argv[argc] );
      }
      
      // これ以上は解析しないよ
      index_max = index - 1;
    }
    
  };  // struct options

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_OPTIONS_INCLUDED_OPTIONS_HPP_
