#ifndef GINTENLIB_OPTIONS_INCLUDED_EXCEPTIONS_HPP_
#define GINTENLIB_OPTIONS_INCLUDED_EXCEPTIONS_HPP_

/*
      <gintenlib/options/exceptions.hpp>

  オプション解析ライブラリ用例外クラス

  宣言：
    // 例外基本クラス
    struct option_error : std::runtime_error;
    // オプション指定文字列がおかしい
    struct invalid_option_string : option_error;
    // 指定されないオプションが渡された
    struct unknown_option : option_error;
    // 引数が必要なのに渡されなかった
    struct option_requires_argument : option_error;

  機能：
    例外用クラスです。以上。

*/

#include <stdexcept>

namespace gintenlib
{
  // 例外基本クラス
  struct option_error : std::runtime_error
  {
    explicit option_error( const std::string& what_ )
      : std::runtime_error( what_ ) {}
  };
  // オプション指定文字列がおかしい
  struct invalid_option_string : option_error
  {
    explicit invalid_option_string( const std::string& optstr )
      : option_error( "invalid option format string: " + optstr ) {}
  };
  // 指定されないオプションが渡された
  struct unknown_option : option_error
  {
    explicit unknown_option( const std::string& opt )
      : option_error( "unknown option: " + opt ) {}
  };
  // 引数が必要なのに渡されなかった
  struct option_requires_argument : option_error
  {
    explicit option_requires_argument( const std::string& opt )
      : option_error( "option requres an argument: " + opt ) {}
  };

}   // namespace gintenlib

#endif  // #ifndef GINTENLIB_OPTIONS_INCLUDED_EXCEPTIONS_HPP_
