#ifndef GINTENLIB_INCLUDED_LIST_FORMAT_HPP_
#define GINTENLIB_INCLUDED_LIST_FORMAT_HPP_

/*

      <gintenlib/list_format.hpp>

  list_format ： コンテナのお手軽ストリーム出力

  宣言：
    template< typename Range >
    unspecified list_format( const Range& src );

    template< typename InIte >
    unspecified list_format( const InIte& begin, const InIte& end );

    template< typename Range, typename Pre, typename Delim, typename Post  >
    unspecified list_format( const Range& src, const Pre& pre, const Delim& delim, const Post& post );

    template< typename InIte, typename Pre, typename Delim, typename Post  >
    unspecified list_format( const InIte& begin, const InIte& end, const Pre& pre, const Delim& delim, const Post& post );

  機能：
    任意のコンテナや配列（ boost 使用版では任意の range ）を、ストリームに流せるよう加工します。
    デフォルトでは ( 1, 2, 3, 4 ) のように出力されますが、引数を指定することでスタイルは変更できます。

  使用例：
    // v は任意のコンテナ。( 1, 2, 3 ) のように表示されます
    cout << gintenlib::list_format(v) << endl;
    // 引数を指定すれば、[ 1 | 2 | 3 ] のような表示にもできます
    cout << gintenlib::list_format( v, "[ ", " | ", " ]" ) << endl;
    // また、イテレータを使って出力対象を指定することも出来ます。
    // この場合は逆順の表示になります
    cout << gintenlib::list_format( v.rbegin(), v.rend() ) << endl;

  補足：
    ・基本的に list_format 関数は、「戻り値をストリームに流す」という使い方のみを想定しています。
    ・ただし、list_format 関数の戻り値に boost::lexical_cast を適応することは可能です。
    ・また、戻り値を gintenlib::output_element オブジェクトに格納することも出来ます。
    ・上記以外の目的で使った場合、list_format 関数の動作は不定となります。

*/

// タイプ数削減のための using 宣言
#include "call_traits.hpp"

#include <boost/range.hpp>

namespace gintenlib
{
  // 実際にストリームに流すクラス
  template< typename InIte, typename Pre = const char*, 
    typename Delim = const char*, typename Post = const char* >
  struct list_format_impl_
  {
    typedef typename call_traits<InIte>::param_type inite_param_t;
    typedef typename call_traits<Pre>::param_type pre_param_t;
    typedef typename call_traits<Delim>::param_type delim_param_t;
    typedef typename call_traits<Post>::param_type post_param_t;
    
    // constructor
    list_format_impl_( inite_param_t begin, inite_param_t end,
      pre_param_t pre, delim_param_t delim, post_param_t post )
     : begin_(begin), end_(end), pre_(pre), delim_(delim), post_(post) {}

    typename call_traits<InIte>::value_type begin_, end_;
    typename call_traits<Pre>::value_type pre_;
    typename call_traits<Delim>::value_type delim_;
    typename call_traits<Post>::value_type post_;

    // 出力の実装
    // テンプレートにより std::ostream に限らず出力できます
    template<typename Stream>
    friend Stream& operator<< ( Stream& os, const list_format_impl_& src )
    {
      // 空のリストを表示させようとした場合、何も表示されない
      if( src.begin_ == src.end_ ){ return os; }
  
      // 表示本体
      os << src.pre_;
  
      InIte ite = src.begin_;
      os << *ite;
      for( ++ite; ite != src.end_; ++ite )
      {
        os << src.delim_ << *ite;
      }
  
      os << src.post_;
  
      return os;
  
    }   // operator<< ( Stream&, const list_format_impl_& )
    
  };  // struct list_format_impl_


  // 関数本体

  // コンテナ版

  // 一般形。任意の pre, delim, post に関して
  // os << pre << 要素１ << delim << 要素２ << ... << delim << 要素ｎ << end;
  // というイメージで出力される。
  template< typename Range, typename Pre, typename Delim, typename Post >
  inline const list_format_impl_< typename boost::range_iterator<const Range>::type, Pre, Delim, Post >
  list_format
    ( const Range& src, const Pre& pre, const Delim& delim, const Post& post )
  {
    using namespace boost;
    
    return list_format_impl_< typename range_iterator<const Range>::type, Pre, Delim, Post >
          ( begin(src), end(src), pre, delim, post );
  }

  // 省略版。( 要素１, 要素２, ... , 要素ｎ ) という感じに出力
  template< typename Range >
    inline const list_format_impl_< typename boost::range_iterator<const Range>::type >
  list_format( const Range& src )
  {
    using namespace boost;
    
    return list_format_impl_< typename range_iterator<const Range>::type >
          ( begin(src), end(src), "( ", ", ", " )" );
  }


  // イテレータ版

  // 一般形
  template< typename InIte, typename Pre, typename Delim, typename Post >
    inline const list_format_impl_<InIte, Pre, Delim, Post>
  list_format
    ( const InIte& begin, const InIte& end, const Pre& pre, const Delim& delim, const Post& post )
  {
    return list_format_impl_<InIte, Pre, Delim, Post>( begin, end, pre, delim, post );
  }

  // 省略版
  template< typename InIte >
    inline const list_format_impl_<InIte>
  list_format( const InIte& begin, const InIte& end )
  {
    return list_format_impl_<InIte>( begin, end, "( ", ", ", " )" );
  }

}   // namespace gintenlib


#endif  // #ifndef GINTENLIB_INCLUDED_LIST_FORMAT_HPP_
