#ifndef GINTENLIB_INCLUDED_CLONABLE_PTR_HPP_
#define GINTENLIB_INCLUDED_CLONABLE_PTR_HPP_

/*

      <gintenlib/clonable_ptr.hpp>

  clonable_ptr ： 深いコピーが可能な参照カウント式スマートポインタ

  宣言：
    template<typename T>
    struct clonable_ptr
    {
      typedef T element_type;
      typedef T value_type;
      typedef T* pointer;
      typedef typename boost::add_reference<T>::type reference;
      
      // デフォルトコンストラクタ
      clonable_ptr();
      
      // ポインタから作るコンストラクタ
      template<typename U>
      explicit clonable_ptr( U* p );
      
      // copy constructor
      clonable_ptr( const clonable_ptr& src );
      template<typename U>
      clonable_ptr( const clonable_ptr<U>& src );
      
      // デストラクタ
      ~clonable_ptr() {}
      
      // nothrow swap
      void swap( clonable_ptr& other );
      friend void swap( clonable_ptr& one, clonable_ptr& another );
      
      // operator=
      clonable_ptr& operator=( const clonable_ptr& rhs );
      template<typename U>
      clonable_ptr& operator=( const clonable_ptr<U>& rhs );
      
      // reset
      void reset();
      template<typename U>
      void reset( U* p );
      
      
      // スマートポインタ関連のあれこれ
      T* get() const;
      friend T* get_pointer( const clonable_ptr& );
      T* operator->() const;
      T& operator*() const;
      
      operator unspecified_bool_type() const;
      bool operator!() const;
      
      
      // 参照カウントの数を返す
      int use_count() const;
      // 参照するオブジェクトが、このポインタのみから参照されている場合 true
      bool unique() const;
      
      // オブジェクトの複製を作り出して返す
      clonable_ptr<typename boost::remove_const<T>::type> clone() const;
      // unique でなければ参照するオブジェクトを複製して unique にする
      void to_unique();
      
      // shared_ptr への変換
      shared_ptr<T> to_shared() const;
      friend shared_ptr<T> to_shared( const clonable_ptr& target );
      
    
    };
    
    // 比較
    template<typename T1, typename T2>
    inline bool operator==( const clonable_ptr<T1>&, const clonable_ptr<T2>& );
    template<typename T1, typename T2>
    inline bool operator!=( const clonable_ptr<T1>&, const clonable_ptr<T2>& );
    template<typename T1, typename T2>
    inline bool operator<( const clonable_ptr<T1>&, const clonable_ptr<T2>& );
    
    // pointer casts
    template<typename T1, typename T2>
    friend clonable_ptr<T1> static_pointer_cast( const clonable_ptr<T2>& src );
    template<typename T1, typename T2>
    friend clonable_ptr<T1> dynamic_pointer_cast( const clonable_ptr<T2>& src );
    template<typename T1, typename T2>
    friend clonable_ptr<T1> const_pointer_cast( const clonable_ptr<T2>& src );

  機能：
    対象の「深いコピー」を生成できる参照カウント式スマートポインタ。
    主に対象の型を正確に記憶し複製する程度の機能を持ちます。
    特に便利なのが to_unique メンバ関数で、このポインタの存在理由の大半がこれに集約されます。
    コピーにコストのかかるオブジェクトを扱うとき、普段は浅いコピーでコピーコストを削減、
    オブジェクトに変更を加えるときはその前に to_unique を呼び出す、なんて使い方が一般的かと。
    もちろん、「対象のオブジェクトの正確なコピーを作れる」という点に着目した使い方もアリです。
    高機能ですが、コピー不能なオブジェクトはそもそも格納できないので、そこだけは注意が必要。
    また、仮想関数とテンプレートを使った実装なので、コンパイル後のサイズは大きくなります。
    便利だけど頼りすぎず、他の上手い実装があればそちらを優先してやってください。

*/

#include "pointer_facade.hpp"
#include "reference_counter.hpp"
#include "shared_ptr.hpp" // for to_shared

#include <boost/intrusive_ptr.hpp>
#include <boost/type_traits/remove_const.hpp>
#include <utility>  // for std::pair

namespace gintenlib
{
  namespace detail_
  {
    // カウントされるオブジェクト
    struct clonable_counted_base
      : reference_counter<clonable_counted_base>
    {
      // virtual destructor
      virtual ~clonable_counted_base(){}
      
      // 本体、複製を作る
      typedef std::pair<clonable_counted_base*, void*> cloned_type;
      virtual cloned_type clone() const = 0;
      
    };  // clonable_counted_base
    
    // make_clonable 用 impl
    template< typename T >
    struct clonable_counted_impl_obj
      : clonable_counted_base
    {
      // 制作
      explicit clonable_counted_impl_obj( const T& src )
        : obj(src) {}
      
      // 解放
      // virtual ~clonable_counted_impl_obj() {}
      
      // 複製制作
      typedef typename boost::remove_const<T>::type T_;
      typedef clonable_counted_impl_obj<T_> impl_type;
      
      using clonable_counted_base::cloned_type;
      virtual cloned_type clone() const
      {
        impl_type* pt = new impl_type(obj);
        return std::make_pair( pt, pt->get() );
      }
      
      // 保持するオブジェクトのアドレスを取得
      T* get() { return &obj; }
      
     private:
      T obj;
     
    };  // clonable_counted_impl_obj
    
    // ポインタで渡された場合に対応
    template< typename T >
    struct clonable_counted_impl_p
      : clonable_counted_base
    {
      explicit clonable_counted_impl_p( T* p_ )
        : p( p_ ) {}
      
      // 解放処理
      virtual ~clonable_counted_impl_p()
      {
        boost::checked_delete(p);
      }
      
      // 複製制作
      using clonable_counted_base::cloned_type;
      typedef typename boost::remove_const<T>::type T_;
      typedef clonable_counted_impl_obj<T_> impl_type;
      virtual cloned_type clone() const
      {
        impl_type* pt = new impl_type(*p);
        return std::make_pair( pt, pt->get() );
      }
      
      // 保持するポインタのアドレスを取得
      T* get() const { return p; }
      
     private:
      T* p;
    
    };  // clonable_counted_impl_p
    
    
    // cloning_ptr 用の参照カウンタ
    struct clonable_count
    {
      // デフォルト初期化
      clonable_count() {}
      
      // T 型のオブジェクト p_ に関連付けられたカウンタを作る
      template<typename T>
      explicit clonable_count( T* p_ )
        : p( p_ ? new clonable_counted_impl_p<T>(p_) : 0 ) {}
      
      // clonable_counted_base* からの構築
      clonable_count( clonable_counted_base* p_ )
        : p( p_ ) {}
      
      // デストラクタは自動定義された奴でおｋ
      // ~clonable_count(){}
      
      // no throw swap
      void swap( clonable_count& other )
      {
        p.swap( other.p );
      }
      friend void swap( clonable_count& one, clonable_count& another )
      {
        one.swap( another );
      }
      
      // 複製制作
      typedef std::pair<clonable_count, void*> cloned_type;
      // clonable_count は clonable_counted_base* から暗黙変換できる
      cloned_type clone() const
      {
        return p ? cloned_type( p->clone() ) : cloned_type( 0, 0 );
      }
      
      // 内部のカウントを使用する
      int use_count() const { return p ? p->use_count() : 0; }
      // ユニークか否か
      bool unique() const { return use_count() == 1; }
      
     private:
      boost::intrusive_ptr<clonable_counted_base> p;
    
    };  // clonable_count
    
    class clonable_ptr_tag {};
    
  } // namespace detail_
  
  
  
  template<typename T>
  struct clonable_ptr
    : pointer_facade< clonable_ptr<T>, T, detail_::clonable_ptr_tag >
  {
   private:
    typedef clonable_ptr this_type;
    typedef detail_::clonable_count clonable_count;
    
   public:
    // デフォルトコンストラクタ
    clonable_ptr() : p(), count() {}
    
    // ポインタから作るコンストラクタ
    template<typename U>
    explicit clonable_ptr( U* p_ )
      : p( p_ ), count( p_ ) {}
    
    // copy constructor では clone しない
    clonable_ptr( const clonable_ptr& src )
      : p( src.p ), count( src.count ) {}
    template<typename U>
    clonable_ptr( const clonable_ptr<U>& src )
      : p( src.p ), count( src.count ) {}
    
    // デストラクタの処理は clonable_count に任せる
    // ~clonable_ptr() {}
    
    // swap
    void swap( this_type& other )
    {
      using std::swap;
      swap( p, other.p );
      swap( count, other.count );
    }
    friend void swap( this_type& one, this_type& another )
    {
      one.swap( another );
    }
    
    // 代入演算
    this_type& operator=( const this_type& rhs )
    {
      this_type(rhs).swap( *this );
      return *this;
    }
    template<typename U>
    this_type& operator=( const clonable_ptr<U>& rhs )
    {
      this_type(rhs).swap( *this );
      return *this;
    }
    
    // reset
    void reset()
    { 
      this_type().swap( *this );
    }
    template<typename U>
    void reset( U* p_ )
    { 
      this_type(p_).swap( *this );
    }
    
    // get() さえ定義すれば後は pointer_facade により自動
    T* get() const { return p; }
    
    
    // 本体
    
    // 複製制作
    typedef typename boost::remove_const<T>::type T_;
    typedef clonable_ptr<T_> cloned_type;
    cloned_type clone() const
    {
      return cloned_type( count.clone() );
    }
    // 参照カウント計算
    int use_count() const { return count.use_count(); }
    bool unique() const { return count.unique(); }
    // unique じゃなければ複製して unique に
    void to_unique()
    {
      if( p != 0 && !unique() )
      {
        clone().swap(*this);
      }
    }
    
    // to_shared
    struct to_shared_helper
    {
      to_shared_helper( const clonable_count& count_ )
        : count( count_ ) {}
      
      // デストラクタで適宜やってくれるので
      // operator() では何も行わない
      // と思いきや、 weak_ptr を使った場合はデストラクタ呼び出しまで
      // 時間がかかるので、きっちり operator() で削除しなきゃダメ
      void operator()( T* )
      {
        // count を破棄する
        clonable_count().swap( count );
      }
      
     private:
      clonable_count count;
      
    };
    // shared_ptr に変換
    shared_ptr<T> to_shared() const
    {
      return p ? shared_ptr<T>( p, to_shared_helper(count) ) : shared_ptr<T>();
    }
    friend shared_ptr<T> to_shared( const this_type& target )
    {
      return target.to_shared();
    }
    
   private:
    T* p;
    clonable_count count;
    
    template<typename U> friend class clonable_ptr;
    
    // 裏技コンストラクタ
    clonable_ptr( const clonable_count& count_, T* p_ )
      : p(p_), count(count_) {}
    explicit clonable_ptr( const clonable_count::cloned_type& pair )
      : p( static_cast<T*>( pair.second ) ), count( pair.first ) {}
    
    // pointer casts
    template<typename T1, typename T2>
    friend clonable_ptr<T1> static_pointer_cast( const clonable_ptr<T2>& src );
    template<typename T1, typename T2>
    friend clonable_ptr<T1> dynamic_pointer_cast( const clonable_ptr<T2>& src );
    template<typename T1, typename T2>
    friend clonable_ptr<T1> const_pointer_cast( const clonable_ptr<T2>& src );
    
  };  // clonable_ptr<T>
  
  // xxx_pointer_cast
  template<typename T, typename S>
  inline clonable_ptr<T> static_pointer_cast( const clonable_ptr<S>& src )
  {
    return clonable_ptr<T>( src.count, static_cast<T*>(src.p) );
  }
  template<typename T, typename S>
  inline clonable_ptr<T> dynamic_pointer_cast( const clonable_ptr<S>& src )
  {
    T* p = dynamic_cast<T*>(src.p);
    return p ? clonable_ptr<T>( src.count, p ) : clonable_ptr<T>();
  }
  template<typename T, typename S>
  inline clonable_ptr<T> const_pointer_cast( const clonable_ptr<S>& src )
  {
    return clonable_ptr<T>( src.count, const_cast<T*>(src.p) );
  }

} // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_CLONABLE_PTR_HPP_
