#ifndef GINTENLIB_INCLUDED_BOOL_COMPARABLE_HPP_
#define GINTENLIB_INCLUDED_BOOL_COMPARABLE_HPP_

/*
  bool_comparable : safe_boolイディオム提供
  
  宣言：
    template<typename Derived>
    class bool_comparable
    {
     public:
      // safe_bool 本体
      operator unspecified_bool_type() const;
      // 派生したクラスの側で operator! が定義されていない場合、
      // 派生したクラスの boolean_test() メンバ関数があればその否定を返す。
      // operator!() も boolean_test() もなければ、エラー
      bool operator!() const;
    
      // bool との比較。 int のときは反応しないようになってます
      friend bool operator==( const bool_comparable& lhs, bool rhs );
      friend bool operator==( bool lhs, const bool_comparable& rhs );
      friend bool operator!=( const bool_comparable& lhs, bool rhs );
      friend bool operator!=( bool lhs, const bool_comparable& rhs );
    
     protected:
      bool_comparable() {}
    
    };  // class bool_comparable<Derived>
    
        
    // 他、呼び出されるとコンパイルエラーになる operator== と operator!= が宣言されています

  機能：
    boost::operators みたいな感じに、簡単に safe_bool イディオムを実現します。
    使用法は、bool_comparable のテンプレート引数に製作するクラスの名前を入れて public 継承。
    そして製作するクラスに、
      bool operator!() const;
    関数か、あるいは
      bool boolean_test() const;
    関数のどちらかを定義すればＯＫです（両方ある場合は operator!() が優先されます）。
  
  使用法 : 
    // 例として boost::optional 的な何かを作りましょう
    #include <boost/scoped_ptr.hpp>
    
    template<typename T>
    class my_optional
      : public gintenlib::bool_comparable< my_optional<T> > // このように使う
    {
      // 本体部分はなんら気にせず書いてよい
     public:
      typedef  T  value_type;
      
      typedef       T&       reference;
      typedef const T& const_reference;
      
      my_optional() : p() {}
      my_optional( const T& x ) : p( new T(x) ) {}
      
      reference        operator*()       { return *p; }
      const_reference  operator*() const { return *p; }
      
      // operator!() を確実に実装することを忘れなければ。
      bool operator!() const
      {
        return !p;
      }
      // これにより、bool比較も出来るようになる
      
      // ちなみに operator!() じゃなくて
      // bool boolean_test() const { return p.get(); }
      // これでもいい（その場合、operator! は自動定義）
      
      // そのほか、コピーコンストラクタとか代入演算子とかは面倒だから省略
      
     private:
      // 面倒なポインタ管理とかしたくないので scoped_ptr を使う
      boost::scoped_ptr<T> p;
      
    };
    
    int main()
    {
      my_optional<int> a, b(1), c(0);
      
      // if の条件部として
      if( a )
      {
        assert( !"a is empty." );
      }
      
      // && や || と組み合わせて
      assert( b && c );
      
      // ゼロとの比較
      assert( a == 0 && b != 0 && c != 0 );
      // assert( b == 1 );  // 1 とは比較できない
      
      // bool 値との比較
      assert( a == false && b == true && c == true );
      // static_cast
      assert( static_cast<bool>(b) == static_cast<bool>(c) );
      // assert( a != b ); // コレはダメ（もし b と c を比較したら？）

  注意事項：
    ・boost::enable_if を使っています。古いコンパイラでは動きません。
    ・それが嫌なら GINTENLIB_BOOL_COMPARABLE_NO_ENABLE_IF マクロを定義すればＯＫです。
    ・ただし、その場合、上の例での「b == 1」的な表現がコンパイル通ってしまいます。
    
    ・「 a == b 」のような、派生先のクラス同士での比較は、デフォルトではコンパイルエラーになります。
    ・ですが、当然、派生先で operator== を定義しておけば、きちんと比較できるようになります。
    ・ただし、派生先のクラスが「 0 から暗黙変換可能」であり「operator==が定義されている」場合は、
      「 x == 0 」のような比較が、ゼロとの比較に限りですが、出来なくなってしまいます:
        struct convertible_from_int
          : gintenlib::bool_comparable<convertible_from_int>,
            private boost::equality_comparable<convertible_from_int>
        {
          int value;
          convertible_from_int( int x = 0 ) : value(x) {} // int から暗黙変換できると・・・
          
          bool operator!() const { return value == 0; }
          
          // operator== による比較が出来る場合
          friend bool operator==( const convertible_from_int& lhs, const convertible_from_int& rhs )
          {
            return lhs.value == rhs.value;
          }
         
        };  
        
        int main()
        {
          convertible_from_int x, y = 2;
          assert( !x ); // こういうのは普通に出来るが
          assert( y );
          assert( x == false );
          assert( y == true );
          
          // assert( x == 0 ); // コレが実行できない（曖昧になってしまう）
          // assert( y != 0 ); // 当然これもダメ
          
          assert( y == 2 ); // でもコレはＯＫ
          assert( x != 1 ); // コレもＯＫ
        }
       
    ・その場合は int との比較を別個定義してやればＯＫです：
        struct convertible_from_int
          : gintenlib::bool_comparable<convertible_from_int>,
            private boost::equality_comparable<convertible_from_int>,
            private boost::equality_comparable<convertible_from_int, int>
        {
          int value;
          convertible_from_int( int x = 0 ) : value(x) {} // int から暗黙変換できる場合でも
          
          bool boolean_test() const { return value; } // boolean_test を使ってみる。
          
          friend bool operator==( const piyo& lhs, const piyo& rhs )
          {
            return lhs.value == rhs.value;
          }
          // int との比較を明示的に定義さえすれば
          friend bool operator==( const convertible_from_int& lhs, int rhs )
          {
            return lhs.value == rhs;
          }
         
        };
        
        int main()
        {
          convertible_from_int x, y(2)
          assert( x == 0 ); // ＯＫ
          assert( y != 0 );
        }
    
*/

#ifndef GINTENLIB_BOOL_COMPARABLE_NO_ENABLE_IF
  #include "enable_if.hpp"
  #include <boost/type_traits/is_same.hpp>
#endif

namespace gintenlib
{
  // ヘルパ構造体
  namespace detail_
  {
    struct bool_comparable_helper
    {
      // operator bool_type() の戻り値として使われる値
      void bool_value() const {}
      
     private:
      // コンパイルエラーを起こすための private 関数
      bool this_type_does_not_support_comparisons_();
    
    };
  }
  
 namespace bool_comparable_ // ADL 回避
 {
  // 本体
  template<typename Derived>
  class bool_comparable
  {
    // safe-bool に使う型（メンバ関数へのポインタ）
    typedef void (bool_comparable::*bool_type)() const;
    
    // 派生したクラスを得る関数
    const Derived& derived() const { return *static_cast<const Derived*>(this); }
    
    // ヘルパ構造体名を適当に短縮
    typedef ::gintenlib::detail_::bool_comparable_helper helper;
    
   public:
    // 本題
    operator bool_type() const
    {
      return bool_test_() ? bool_value_() : 0;
      // bool_test_, bool_value_ は後で定義する
    }
  
    // bool との比較
    
  #ifndef GINTENLIB_BOOL_COMPARABLE_NO_ENABLE_IF
    
    // enable_if で bool と比較する時だけ有効にすることで、x == 1 のような比較を無力化する
    // todo: g++ 以外のコンパイラでの動作チェック
    
    // 実装
    template<typename U>
    friend typename enable_if<boost::is_same<U, bool>, bool>::type
      operator==( const Derived& lhs, U rhs )
    {
      return bool(lhs) == rhs;
    }
    template<typename U>
    friend typename enable_if<boost::is_same<U, bool>, bool>::type
      operator==( U lhs, const Derived& rhs )
    {
      return lhs == bool(rhs);
    }
    template<typename U>
    friend typename enable_if<boost::is_same<U, bool>, bool>::type
      operator!=( const Derived& lhs, U rhs )
    {
      return bool(lhs) != rhs;
    }
    template<typename U>
    friend typename enable_if<boost::is_same<U, bool>, bool>::type
      operator!=( U lhs, const Derived& rhs )
    {
      return lhs != bool(rhs);
    }
    
  #else   // #ifndef GINTENLIB_BOOL_COMPARABLE_NO_ENABLE_IF
    
    // 一応 enable_if を使わない版も用意
    // この場合、 x == 1 のような表現が通ってしまうが仕方ないね
    friend bool operator==( const Derived& lhs, bool rhs )
    {
      return bool(lhs) == rhs;
    }
    friend bool operator==( bool lhs, const Derived& rhs )
    {
      return lhs == bool(rhs);
    }
    friend bool operator!=( const Derived& lhs, bool rhs )
    {
      return bool(lhs) != rhs;
    }
    friend bool operator!=( bool lhs, const Derived& rhs )
    {
      return lhs != bool(rhs);
    }
    
  #endif  // #ifndef GINTENLIB_BOOL_COMPARABLE_NO_ENABLE_IF
  
   private:
    // 実装補助
    // bool_type として返す値
    static bool_type bool_value_()
    {
      // 関数を無駄に作らせない工夫。どうせ bool_type の値は使わないし
      return reinterpret_cast<bool_type>( &helper::bool_value );
    }
    
    // テスト関数
    // まず Derived::operator!() を見に行く
    bool bool_test_() const
    {
      return !( !derived() );
    }
    
   public:
    // Derived 側で operator!() が定義されて無いなら
    // Derived::boolean_test() 関数がないか探し、無ければコンパイルエラー
    bool operator!() const
    {
      return !( derived().boolean_test() );
    }
    // friend 関数版。存在する意味は特に無い。
    friend bool boolean_test( const bool_comparable& x )
    {
      return x.bool_test_();
    }
    
    // 比較はエラーになる（ Derived 側で特別に定義すればＯＫ）
    
    // 等値比較
    friend bool operator==( const bool_comparable& lhs, const bool_comparable& rhs )
    {
      return helper().this_type_does_not_support_comparisons_();
    }
    // 不等値比較
    friend bool operator!=( const bool_comparable& lhs, const bool_comparable& rhs )
    {
      return helper().this_type_does_not_support_comparisons_();
    }
  
   protected:
    // 派生クラス以外からの構築禁止
     bool_comparable() {}
    // ~bool_comparable() {}  // trivial にしたい
    
  };  // class bool_comparable<Derived>
  
 }  // namespace bool_comparable_
 
  // gintenlib 名前空間に引き入れる
  using namespace bool_comparable_;

} // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_BOOL_COMPARABLE_HPP_
