#ifndef GINTENLIB_INCLUDED_ASSIGN_HPP_
#define GINTENLIB_INCLUDED_ASSIGN_HPP_

/*

      <gintenlib/assign.hpp>

  assign ： コピーされる側の都合を優先する copy アルゴリズム

  宣言：
    template< typename FwdOutIte, typename InIte >
    InIte assign( FwdOutIte begin, FwdOutIte end, InIte src );

    template< typename Range, typename InIte >
    InIte assign( Range& dest, InIte src );

  機能：
    [ begin, end ) 区間、あるいは区間 dest を、イテレータ src によって初期化します。
    戻り値は最後に代入した src の次の要素を示すイテレータになります。
  
  等価コード：
    // イテレータ版のみ記述
    for( FwdOutIte ite = begin; ite != end; ++ite, ++src )
    {
      *ite = *src;
    }

  使用例：
    int a[10];
    boost::array<int, 10> b;
    
    // a を、boost::counting_iterator を使い 1 〜 10 の値で初期化
    gintenlib::assign( a, boost::counting_iterator<int>(1) );
    
    // b を逆順に a で初期化
    gintenlib::assign( b.rbegin(), b.rend(), &a[0] );
    
    // 表示
    cout << gintenlib::list_format( a ) << endl;
    cout << gintenlib::list_format( b ) << endl;

  補足：
    ・主に固定長のコンテナを初期化をする場面を想定しています。
    ・可変長のコンテナの場合は、insert_iterator を使って copy した方が効果的です。

*/

#include <boost/range.hpp>

namespace gintenlib
{
  // [ begin, end ) 区間を src で初期化
  template< typename FwdOutIte, typename InIte >
  inline InIte assign( FwdOutIte begin, FwdOutIte end, InIte src )
  {
    while( begin != end ) 
    {
      *begin = *src;
      ++begin; ++src;
    }
    return src;
  }
  
  // dest を src で初期化
  template< typename Range, typename InIte >
  inline InIte assign( Range& dest, const InIte& src )
  {
    return ::gintenlib::assign( boost::begin(dest), boost::end(dest), src );
  }

} // namespace gintenlib

#endif  // #ifndef GINTENLIB_INCLUDED_ASSIGN_HPP_
