/*
 * Copyright (C) 2011 OgakiSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ogakisoft.android.gestureime;

import java.io.File;

import ogakisoft.android.gesture.reform.GestureLibrary;
import ogakisoft.android.gesture.reform.Learner;
import ogakisoft.android.svm.SvmWrapper;
import ogakisoft.android.util.LOG;
import ogakisoft.android.util.Utils;
import android.os.AsyncTask;
import android.os.Environment;

/**
 * Load Gesture-data from a local-file
 * 
 * @author noritoshi ogaki
 * @version 1.1
 */
public class GesturesLoadTask extends AsyncTask<Void, NamedGesture, Integer> {
	/** serialized user-define data */
	public static final File[] LEARNER = {
			new File(GestureLibrary.DataDir, "learner.0"),
			new File(GestureLibrary.DataDir, "learner.1"),
			new File(GestureLibrary.DataDir, "learner.2"),
			new File(GestureLibrary.DataDir, "learner.3"),
			new File(GestureLibrary.DataDir, "learner.4"),
			new File(GestureLibrary.DataDir, "learner.5") };
	/** LOG tag */
	private final static String TAG = "GesturesLoadTask";
	// public static final int STATUS_CANCELLED = 1;
	public static final int STATUS_NO_STORAGE = 2;
	public static final int STATUS_NOT_LOADED = 3;
	public static final int STATUS_RUNNING = 5;
	public static final int STATUS_FINISH = 6;
	private final GestureLibrary mLibrary;
	private final Callback mCallback;
	private int mStatus;
//	private Integer mLock = 0;

	public interface Callback {
		void before();

		void post();

		void after();

		void fail(int status);

		void cancel();
	}

	public GesturesLoadTask(GestureLibrary library, Callback callback) {
		mLibrary = library;
		mCallback = callback;
	}

	@Override
	protected void onPreExecute() {
		super.onPreExecute();
		if (null != mCallback) {
			mCallback.before();
		}
	}

	@Override
	protected Integer doInBackground(Void... params) {
		long start;
		if (!Environment.MEDIA_MOUNTED.equals(Environment
				.getExternalStorageState())) {
			mStatus = STATUS_NO_STORAGE;
			return mStatus;
		}
		mStatus = STATUS_RUNNING;
		mLibrary.loadGestures();
		Learner learner;
		final int count = GestureLibrary.MAX_NUM_OF_STORE;
		for (int num = 0; num < count; num++) {
			LOG.d(TAG, "doInBackground: num={0,number,#}", num);
			learner = mLibrary.getLearner(num);
			if (canDeserializeData(num)) {
				start = Utils.currentTime();
				learner.deserialize(LEARNER[num]);
				LOG.d(TAG,
						"doInBackground: userLearner.deserialize() time={0,number,#} ms",
						(Utils.currentTime() - start));
			} else {
				// learning, output model-file
				start = Utils.currentTime();
				SvmWrapper.getInstance().svmTrain(num, learner);
				LOG.d(TAG, "doInBackground: svmTrain() time={0,number,#} ms",
						(Utils.currentTime() - start));
				start = Utils.currentTime();
				learner.serialize(LEARNER[num]);
				LOG.d(TAG,
						"doInBackground: userLearner.serialize() time={0,number,#} ms",
						(Utils.currentTime() - start));
			}
		}
		if (mLibrary.hasLoaded()) {
			mStatus = STATUS_FINISH;
		} else {
			mStatus = STATUS_NOT_LOADED;
		}
		return mStatus;
	}

	@Override
	protected void onPostExecute(Integer result) {
		super.onPostExecute(result);

		if (result == STATUS_FINISH) {
			if (null != mCallback) {
				mCallback.post();
			}
		} else {
			if (null != mCallback) {
				mCallback.fail(mStatus);
			}
		}
		if (null != mCallback) {
			mCallback.after();
		}
	}

	@Override
	protected void onCancelled() {
		if (null != mCallback) {
			mCallback.cancel();
		}
	}

	private boolean canDeserializeData(int num) {
		if (num < 0 || GestureLibrary.MAX_NUM_OF_STORE <= num) {
			return false;
		}
		final File dataFile = SvmWrapper.LibsvmData[num];
		if (!dataFile.exists()) {
			return false;
		}
		boolean result = false;
		final long date = dataFile.lastModified();
		if (LEARNER[num].exists()
				&& date < LEARNER[num].lastModified()) {
			result = true;
		}
		return result;
	}
}
