#include "foldermodel.h"
#include "global.h"
#include "preferences.h"

#include <QApplication>
#include <QDebug>
#include <QDesktopWidget>

///////////////////////////////////////////////////////////////////////////////
/// \brief DefaultPreferExtensions
/// \return 外部アプリを優先する拡張子のデフォルトを返します。
///
QString DefaultPreferExtensions()
{
    QStringList list;

    // 画像系
    list << "ico" << "ai" << "psd" << "xcf" << "tif" << "tiff" << "wmf";
    // 音・動画系
    list << "wav" << "mp3" << "ogg" << "midi" << "mid" << "aif" << "aiff";
    list << "mov" << "mpg" << "mpeg" << "wma" << "wmv" << "asf" << "avi";
    list << "flac" << "mkv";
    // 実行ファイル系
    list << "exe" << "com" << "msi" << "scr";
    // アーカイブ系
    list << "lzh" << "zip" << "cab" << "tar" << "rar" << "gz" << "tgz";
    list << "bz2" << "xz" << "jar" << "7z" << "dmg";
    // ドキュメント系
    list << "pdf" << "doc" << "docx" << "xls" << "xlsx" << "ppt" << "pptx";
    // フォント
    list << "ttf" << "ttc";

    list.sort();

    return list.join(",");
}

///////////////////////////////////////////////////////////////////////////////
/// \brief DefaultFixedFont
/// \return デフォルトの固定幅フォントを返します。
///
QFont DefaultFixedFont()
{
    QFont font = qApp->font();
#ifdef Q_OS_WIN
    font.setFamily("ＭＳ ゴシック");
#elif defined(Q_OS_MAC)
    font.setFamily("Monaco");
#else
#endif
    return font;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::Preferences
/// \param parent   親オブジェクト
///
/// コンストラクタ
///
Preferences::Preferences(QObject *parent) :
    QSettings(QSettings::IniFormat,
              QSettings::UserScope,
              qApp->organizationName(),
              qApp->applicationName(),
              parent)
{
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::Preferences
/// \param path     設定ファイルのパス
/// \param parent   親オブジェクト
///
/// コンストラクタ
///
Preferences::Preferences(const QString &path, QObject *parent) :
    QSettings(path, QSettings::IniFormat, parent)
{
}

#define Key_Filter(side)        (side + "/Filter")
#define Key_NameFilter(side)    (side + "/NameFilter")
#define Key_Sort(side)          (side + "/Sort")
#define Key_Path(side)          (side + "/Path")
///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::restoreModel
/// \param side "Left" or "Right"
/// \param m    フォルダモデルオブジェクト
///
/// モデルの状態を復元します。
///
void Preferences::restoreModel(const QString &side, FolderModel *m)
{
    qDebug() << "Preferences::restoreModel()" << side;

    int filter = value(Key_Filter(side), 0).toInt();
    filter |= QDir::NoDot | QDir::AllDirs | QDir::Files;
    m->setFilter(QFlag(filter));

    QString nameFilter = value(Key_NameFilter(side), "*").toString();
    m->setNameFilters(nameFilter.split(" ", QString::SkipEmptyParts));

    int sort = QDir::Name | QDir::DirsFirst | QDir::IgnoreCase;
    sort = value(Key_Sort(side), sort).toInt();
    m->setSorting(QFlag(sort));

    QString path = QDir::homePath();
    path = value(Key_Path(side), path).toString();
    m->setRootPath(path);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::saveModel
/// \param side "Left" or "Right"
/// \param m    フォルダモデルオブジェクト
///
/// モデルの状態を保存します。
///
void Preferences::saveModel(const QString &side, const FolderModel *m)
{
    qDebug() << "Preferences::saveModel()" << side;

    setValue(Key_Filter(side), static_cast<int>(m->filter()));
    setValue(Key_Sort(side), static_cast<int>(m->sorting()));
    setValue(Key_Path(side), m->rootPath());
}

#define Key_WindowGeometry  "WindowGeometry"
#define Key_WindowState     "WindowState"
///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::restoreWindow
/// \param w    メインウィンドウオブジェクト
///
/// メインウィンドウの状態を復元します。
///
void Preferences::restoreWindow(QMainWindow *w)
{
    qDebug() << "Preferences::restoreWindow()";

    QByteArray geometry = value(Key_WindowGeometry).toByteArray();
    if (geometry.isNull()) {
        QSize size(700, 400);
        QPoint point;
        point.setX((qApp->desktop()->width() - size.width()) / 2);
        point.setY((qApp->desktop()->height() - size.height()) / 2);
        w->setGeometry(QRect(point, size));
    }
    else {
        w->restoreGeometry(geometry);
    }

    QByteArray state = value(Key_WindowState).toByteArray();
    if (!state.isEmpty()) {
        w->restoreState(state);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::saveWindow
/// \param w    メインウィンドウオブジェクト
///
/// メインウィンドウの状態を保存します。
///
void Preferences::saveWindow(const QMainWindow *w)
{
    setValue(Key_WindowGeometry, w->saveGeometry());
    setValue(Key_WindowState, w->saveState());
}

#define Key_BookmarkEntry(i)    QString("Bookmark/Entry%1").arg(i)
#define Key_BookmarkPath(i)     QString("Bookmark/Path%1").arg(i)
///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::addBookmark
/// \param name 名前
/// \param path パス
///
/// ブックマークを追加します。
///
void Preferences::addBookmark(const QString &name, const QString &path)
{
    int n;
    for (n = 1; !value(Key_BookmarkEntry(n)).isNull(); n++)
        ;

    setValue(Key_BookmarkEntry(n), name);
    setValue(Key_BookmarkPath(n), path);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::clearBookmark
///
/// ブックマークをクリアします。
///
void Preferences::clearBookmark()
{
    beginGroup("Bookmark");
    remove("");
    endGroup();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::getBookmarkEntry
/// \param n    番号
/// \return n番目のブックマーク名を取得します。
///
QString Preferences::getBookmarkEntry(int n) const
{
    return value(Key_BookmarkEntry(n)).toString();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::getBookmarkPath
/// \param n    番号
/// \return n番目のブックマークパスを取得します。
///
QString Preferences::getBookmarkPath(int n) const
{
    return value(Key_BookmarkPath(n)).toString();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewFgColor
/// \param active   アクティブならtrue
/// \return フォルダービューの文字色を返します。
///
QColor Preferences::folderViewFgColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewFgColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewBgColor
/// \param active   アクティブならtrue
/// \return フォルダービューの背景色を返します。
///
QColor Preferences::folderViewBgColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewBgColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewMarkedFgColor
/// \param active   アクティブならtrue
/// \return フォルダービューのマーク文字色を返します。
///
QColor Preferences::folderViewMarkedFgColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewMarkedFgColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewMarkedBgColor
/// \param active   アクティブならtrue
/// \return フォルダービューのマーク背景色を返します。
///
QColor Preferences::folderViewMarkedBgColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewMarkedBgColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewSystemColor
/// \param active   アクティブならtrue
/// \return システム属性の文字色を返します。
///
QColor Preferences::folderViewSystemColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewSystemColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewHiddenColor
/// \param active   アクティブならtrue
/// \return 隠し属性の文字色を返します。
///
QColor Preferences::folderViewHiddenColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewHiddenColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::folderViewReadOnlyColor
/// \param active   アクティブならtrue
/// \return 読取専用属性の文字色を返します。
///
QColor Preferences::folderViewReadOnlyColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getFolderViewReadOnlyColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::locationBoxFgColor
/// \param active   アクティブならtrue
/// \return ロケーションボックスの文字色を返します。
///
QColor Preferences::locationBoxFgColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getLocationBoxFgColor().darker(darkFactor);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Preferences::locationBoxBgColor
/// \param active   アクティブならtrue
/// \return ロケーションボックスの背景色を返します。
///
QColor Preferences::locationBoxBgColor(bool active) const
{
    int darkFactor = 100;
    if (!active) {
        darkFactor += getDarkFacotr() * 100;
    }
    return getLocationBoxBgColor().darker(darkFactor);
}

#define IMPLEMENT_BOOL(key, defVal)                 \
    bool Preferences::is##key() const {             \
        return value(#key, def##key()).toBool();    \
    }                                               \
    bool Preferences::def##key() const {            \
        return defVal;                              \
    }                                               \
    void Preferences::set##key(bool value) {        \
        setValue(#key, value);                      \
    }

IMPLEMENT_BOOL(Reset, false)
IMPLEMENT_BOOL(CheckUpdate, true)
IMPLEMENT_BOOL(ConfirmQuit, true)
IMPLEMENT_BOOL(ConfirmCopy, true)
IMPLEMENT_BOOL(AutoCloseCopy, false)
IMPLEMENT_BOOL(ConfirmMove, true)
IMPLEMENT_BOOL(AutoCloseMove, false)
IMPLEMENT_BOOL(ConfirmDelete, true)
IMPLEMENT_BOOL(AutoCloseDelete, false)
IMPLEMENT_BOOL(ConfirmRename, true)
IMPLEMENT_BOOL(AutoCloseRename, false)
IMPLEMENT_BOOL(OpenAfterCreation, false)
IMPLEMENT_BOOL(MoveAfterCreation, false)

#define IMPLEMENT_STRING(key, defVal)                   \
    QString Preferences::get##key() const {             \
        return value(#key, def##key()).toString();      \
    }                                                   \
    QString Preferences::def##key() const {             \
        return defVal;                                  \
    }                                                   \
    void Preferences::set##key(const QString &value) {  \
        setValue(#key, value);                          \
    }

#ifdef Q_OS_LINUX

#elif defined(Q_OS_MAC)
    IMPLEMENT_STRING(ArchiverPath, QQ("/System/Library/CoreServices/Archive Utility.app"))
    IMPLEMENT_STRING(EditorPath, "/Applications/TextEdit.app")
    IMPLEMENT_STRING(TerminalPath, "/Applications/Utilities/Terminal.app --args -c cd")
#else
    IMPLEMENT_STRING(ArchiverPath, QQ("C:/Program Files/Lhaplus/Lhaplus.exe"))
    IMPLEMENT_STRING(EditorPath, "notepad.exe")
    IMPLEMENT_STRING(TerminalPath, "cmd.exe /k cd")
#endif

IMPLEMENT_STRING(CopyBehavior, "OverWriteIfNew")
IMPLEMENT_STRING(PreferExtensions, DefaultPreferExtensions())

#define IMPLEMENT_COLOR(key, defVal)                                    \
    QColor Preferences::get##key() const {                              \
        return value("Appearance/"#key, def##key()).value<QColor>();    \
    }                                                                   \
    QColor Preferences::def##key() const {                              \
        return defVal;                                                  \
    }                                                                   \
    void Preferences::set##key(const QColor &value) {                   \
        setValue("Appearance/"#key, value);                             \
    }

IMPLEMENT_COLOR(FolderViewFgColor, QPalette().text().color())
IMPLEMENT_COLOR(FolderViewBgColor, QPalette().base().color())
IMPLEMENT_COLOR(FolderViewMarkedFgColor, QColor(128, 0, 0))
IMPLEMENT_COLOR(FolderViewMarkedBgColor, QColor(0, 192, 0))
IMPLEMENT_COLOR(FolderViewSystemColor, QColor(128, 0, 128))
IMPLEMENT_COLOR(FolderViewHiddenColor, QColor(128, 128, 128))
IMPLEMENT_COLOR(FolderViewReadOnlyColor, QColor(0, 128, 0))
IMPLEMENT_COLOR(LocationBoxFgColor, QPalette().text().color())
IMPLEMENT_COLOR(LocationBoxBgColor, QPalette().base().color())
IMPLEMENT_COLOR(SearchBoxFgColor, QPalette().text().color())
IMPLEMENT_COLOR(SearchBoxBgColor, QPalette().base().color())
IMPLEMENT_COLOR(SearchBoxUnmatchFgColor, QColor(255, 0, 0))
IMPLEMENT_COLOR(SearchBoxUnmatchBgColor, QPalette().base().color())
IMPLEMENT_COLOR(TextViewFgColor, QPalette().text().color())
IMPLEMENT_COLOR(TextViewCtrlColor, Qt::cyan)
IMPLEMENT_COLOR(TextViewBgColor, QPalette().base().color())
IMPLEMENT_COLOR(HexViewFgColor, getTextViewFgColor())
IMPLEMENT_COLOR(HexViewBgColor, getTextViewBgColor())
IMPLEMENT_COLOR(ImageViewBgColor, QPalette().base().color())

#define IMPLEMENT_DOUBLE(key, defVal)                           \
    double Preferences::get##key() const {                      \
        return value("Appearance/"#key, def##key()).toDouble(); \
    }                                                           \
    double Preferences::def##key() const {                      \
        return defVal;                                          \
    }                                                           \
    void Preferences::set##key(double value) {                  \
        setValue("Appearance/"#key, value);                     \
    }

IMPLEMENT_DOUBLE(DarkFacotr, 0)
IMPLEMENT_DOUBLE(LineHeight, 1.5)
IMPLEMENT_DOUBLE(TextViewLineHeight, 1.0)

#define IMPLEMENT_FONT(key, defVal)                                 \
    QFont Preferences::get##key() const {                           \
        return value("Appearance/"#key, def##key()).value<QFont>(); \
    }                                                               \
    QFont Preferences::def##key() const {                           \
        return defVal;                                              \
    }                                                               \
    void Preferences::set##key(const QFont &value) {                \
        setValue("Appearance/"#key, value);                         \
    }

IMPLEMENT_FONT(FolderViewFont, DefaultFixedFont())
IMPLEMENT_FONT(LocationBoxFont, DefaultFixedFont())
IMPLEMENT_FONT(SearchBoxFont, DefaultFixedFont())
IMPLEMENT_FONT(TextViewFont, DefaultFixedFont())
IMPLEMENT_FONT(HexViewFont, getTextViewFont())
