#include "filereadworker.h"
#include "foldermodel.h"
#include "folderview.h"
#include "global.h"
#include "hexview.h"
#include "imageview.h"
#include "mainwindow.h"
#include "preferences.h"
#include "textview.h"
#include "panel.h"
#include "ui_panel.h"

#include <QDebug>
#include <QTextCodec>

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::Panel
/// \param parent   親ウィジェット
///
/// コンストラクタ
///
Panel::Panel(QWidget *parent) :
    QWidget(parent),
    ui(new Ui::Panel),
    m_worker()
{
    ui->setupUi(this);
    ui->progressBar->setVisible(false);
    ui->scrollArea->setVisible(false);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::~Panel
///
/// デストラクタ
///
Panel::~Panel()
{
    delete ui;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::folderPanel
/// \return フォルダパネルを返します。
///
FolderPanel *Panel::folderPanel() const
{
    return ui->folderPanel;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::initialize
/// \param w メインウィンドウオブジェクト
///
/// パネルを初期化します。
///
void Panel::initialize(MainWindow *w)
{
    qDebug() << "Panel::initialize()";

    ui->folderPanel->initialize(w);
    ui->progressBar->installEventFilter(w);
    ui->scrollArea->installEventFilter(w);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::model
/// \return フォルダモデルを返します。
///
FolderModel *Panel::model() const
{
    return ui->folderPanel->model();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::setModel
/// \param m    設定するモデル
///
/// モデルを設定します。
///
void Panel::setModel(FolderModel *m)
{
    qDebug() << "Panel::setModel()";

    ui->folderPanel->setModel(m);
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::setViewItem
/// \param index    設定するアイテムのインデックス
///
/// ビューにアイテムを設定し、表示します。
///
void Panel::setViewItem(const QModelIndex &index)
{
    qDebug() << "Panel::setViewItem()" << index;

    if (m_worker) {
        m_worker->abort();
    }

    ui->progressBar->setVisible(false);
    ui->scrollArea->setVisible(false);
    ui->folderPanel->setVisible(false);
    if (!index.isValid()) {
        ui->folderPanel->setVisible(true);
        return;
    }

    const FolderModel *m = static_cast<const FolderModel*>(index.model());
    QString path = m->filePath(index);
    if (m->isDir(index)) {
        model()->setRootPath(path);
        ui->folderPanel->setVisible(true);
        return;
    }

    m_worker = new FileReadWorker();
    connect(m_worker, SIGNAL(size(int)), ui->progressBar, SLOT(setMaximum(int)));
    connect(m_worker, SIGNAL(progress(int)), ui->progressBar, SLOT(setValue(int)));
    connect(m_worker, SIGNAL(resultReady(QByteArray)), this, SLOT(onReady(QByteArray)));
    ui->progressBar->setValue(0);
    ui->progressBar->setVisible(true);

    m_worker->open(path);
    m_worker->start();
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::updateAppearance
///
/// 外観を変更します。
///
void Panel::updateAppearance(const Preferences &prefs)
{
    qDebug() << "Panel::updateAppearance()";

    QPalette pal;

    pal = this->palette();
    pal.setColor(this->backgroundRole(), prefs.folderViewBgColor(objectName() == "FPanel"));
    this->setAutoFillBackground(true);
    this->setPalette(pal);

    ui->folderPanel->updateAppearance(prefs);
    if (model()) {
        model()->updateAppearance(prefs);
    }
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::visibleView
/// \return 可視状態のビューを返します。
///
QWidget *Panel::visibleView() const
{
    if (ui->scrollArea->isVisible()) {
        return ui->scrollArea/*->widget()*/;
    }
    if (ui->folderPanel->isVisible()) {
        return ui->folderPanel->itemView();
    }
    if (ui->progressBar->isVisible()) {
        return ui->progressBar;
    }

    qDebug() << ">>>>>>>>>> visibleView() Logic error <<<<<<<<<<";
    return NULL;
}

///////////////////////////////////////////////////////////////////////////////
/// \brief Panel::onReady
/// \param data ファイルの内容
///
/// ファイルの内容を表示します。
///
void Panel::onReady(const QByteArray &data)
{
    MainWindow *mainWnd;
    if (parent()->objectName() == "splitter") {
        mainWnd = static_cast<MainWindow*>(parent()->parent()->parent());
    }
    else {
        mainWnd = static_cast<MainWindow*>(parent()->parent());
    }
    ui->progressBar->setVisible(false);
    ui->folderPanel->setVisible(false);
    ui->scrollArea->setVisible(true);
    QWidget *w = ui->scrollArea->widget();
    if (w) {
        delete w;
    }

    QPixmap pixmap;
    if (pixmap.loadFromData(data)) {
        ImageView *view = new ImageView(ui->scrollArea);
        connect(view, SIGNAL(statusChanged(QString)), mainWnd, SLOT(view_statusChanged(QString)));
        view->installEventFilter(mainWnd);
        view->setData(pixmap);
        emit showed(ui->scrollArea);
        return;
    }

    std::string code = detectCode(data.left(1024));
    QTextCodec *codec = QTextCodec::codecForName(code.c_str());
    if (codec) {
        TextView *view = new TextView(ui->scrollArea);
        connect(view, SIGNAL(copyAvailable(bool)), mainWnd, SLOT(view_copyAvailable(bool)));
        connect(view, SIGNAL(statusChanged(QString)), mainWnd, SLOT(view_statusChanged(QString)));
        view->installEventFilter(mainWnd);
        view->setData(data);
        emit showed(ui->scrollArea);
        return;
    }

    HexView *view = new HexView(ui->scrollArea);
    connect(view, SIGNAL(copyAvailable(bool)), mainWnd, SLOT(view_copyAvailable(bool)));
    connect(view, SIGNAL(statusChanged(QString)), mainWnd, SLOT(view_statusChanged(QString)));
    view->installEventFilter(mainWnd);
    view->setData(data);
    emit showed(ui->scrollArea);
}
